/*
** sptree.h:  The following type declarations provide the binary tree
**  representation of event-sets or priority queues needed by splay trees
**
**  assumes that data and datb will be provided by the application
**  to hold all application specific information
**
**  assumes that key will be provided by the application, comparable
**  with the compare function applied to the addresses of two keys.
*/

# ifndef SPTREE_H
# define SPTREE_H

/**********************************************************************/
/*   Set  the  following  to  non-zero  to  enable  statistics to be  */
/*   gathered  on  a  per tree basis for fine granularity. We always  */
/*   keep the global statistics defined anyway.			      */
/**********************************************************************/
# define	INTERNAL_STATS	0

# if INTERNAL_STATS
#	define	SPI_INCR(x)	(x)++
# else
#	define	SPI_INCR(x)	
# endif

# ifndef NULL
# define NULL	0
# endif

# define STRCMP(a, b) ( (Sct = *(a) - *(b)) ? Sct : strcmp((a), (b)))

typedef struct _spblk {
	struct _spblk * leftlink;
	struct _spblk * rightlink;
	struct _spblk * uplink;

	char	* key;		/* formerly time/timetyp */
	char	* data;		/* formerly aux/auxtype */
} SPBLK;

# define	SPF_WALK	0x0001	/* If set, we cannot splay tree */
					/* because we are walking down the tree*/
typedef struct {
	SPBLK	*root;		/* root node */
	int	flags;
	int	num_items;	/* Number of items in the tree.	*/
	/***********************************************/
	/*   Statistics,   not   strictly  necessary,  */
	/*   but handy for tuning		       */
	/***********************************************/
# if INTERNAL_STATS
	long	lookups;	/* number of splookup()s */
	long	lkpcmps;	/* number of lookup comparisons */
    
	long	enqs;		/* number of spenq()s */
	long	enqcmps;	/* compares in spenq */
    
	long	splays;
	long	splayloops;
# endif
} SPTREE;

/**********************************************************************/
/*   Structure  to  keep  track  of  statistics on a global basis --  */
/*   useful for watching for memory leaks.			      */
/**********************************************************************/
typedef struct SPSTATS {
	unsigned long	trees_alloced;
	unsigned long	spblk_alloced;
	unsigned long	spinit;
	unsigned long	spfree;
	unsigned long	spblk;
	unsigned long	spfreeblk;
	unsigned long	spempty;
	unsigned long	sphead;
	unsigned long	spenq;
	unsigned long	spenq_loops;
	unsigned long	spdeq;
	unsigned long	splay;
	unsigned long	sprotate;
	unsigned long	splookup;
	unsigned long	splookup_loops;
	} SPSTATS;
extern SPSTATS __spstats;
# define	SP_INCR(x)	__spstats.x++
# define	SP_DECR(x)	__spstats.x--
/**********************************************************************/
/*   Following  macro  used  to  allow  compilation  with ANSI C and  */
/*   non-ANSI C compilers automatically.			      */
/**********************************************************************/
# if !defined(PROTO)
#	if	defined(__STDC__)
#		define	PROTO(x)	x
#	else
#		define	PROTO(x)	()
#	endif
# endif

/**********************************************************************/
/*   Return pointer to a new splay tree.			      */
/**********************************************************************/
SPTREE *spinit PROTO((void));

/**********************************************************************/
/*   Free memory associated with root of tree.			      */
/**********************************************************************/
void	spfree PROTO((SPTREE *));

/**********************************************************************/
/*   Allocate SPBLK structure.					      */
/**********************************************************************/
SPBLK	*spblk PROTO((int));

/**********************************************************************/
/*   Free SPBLK structure.					      */
/**********************************************************************/
void	spfreeblk PROTO((SPBLK *));

/**********************************************************************/
/*   Return TRUE if splay tree is empty.			      */
/**********************************************************************/
int spempty PROTO((SPTREE *));

/**********************************************************************/
/*   Add a new SPBLK to the tree.				      */
/**********************************************************************/
void	spenq PROTO((SPBLK *, SPTREE *));

/**********************************************************************/
/*   Remove an item from the tree.				      */
/**********************************************************************/
void	spdeq PROTO((SPBLK *, SPTREE *));

/**********************************************************************/
/*   Function   to   cause   the  tree  to  be  resplayed  with  the  */
/*   designated node at the top of the tree.			      */
/**********************************************************************/
void	splay PROTO((SPBLK *, SPTREE *));

/**********************************************************************/
/*   Function to move an item one layer in the tree.		      */
/**********************************************************************/
void	sprotate PROTO((SPBLK *));

/**********************************************************************/
/*   Lookup  a  key  in  the  tree, and bring the node to the top of  */
/*   the tree.							      */
/**********************************************************************/
SPBLK	*splookup PROTO((char *, SPTREE *));
/**********************************************************************/
/*   Lookup  a  key  in the tree, same as splookup(), but if we dont  */
/*   find  entry  tells  us  if  we  found  a partial match (i.e. an  */
/*   ambiguity).						      */
/**********************************************************************/
SPBLK	*sp_partial_lookup PROTO((char *, SPTREE *, int *, SPBLK **));

/**********************************************************************/
/*   Return pointer to node at top of tree.			      */
/**********************************************************************/
SPBLK	*sphead PROTO((SPTREE *));

/**********************************************************************/
/*   Function  to  visit  each  node  in tree and call user callback  */
/*   function  with  node pointer. This function is safe for calling  */
/*   splay tree to add/delete items from tree.			      */
/**********************************************************************/
void	spapply PROTO((SPTREE *, int (*)(), int));

/**********************************************************************/
/*   Function  to  visit  each  node  in tree and call user callback  */
/*   function,   but   not  safe  to  manipulate  tree  using  splay  */
/*   functions.							      */
/**********************************************************************/
void	spwalk PROTO((SPBLK *, int (*)(), int));

/**********************************************************************/
/*   Function returns number of nodes in tree.			      */
/**********************************************************************/
int	sp_size PROTO((SPTREE *));

/**********************************************************************/
/*   Flattens  splay  tree  by  returning  array  of pointers to all  */
/*   nodes  in  tree.  Last  element is NULL. Array must be freed by  */
/*   caller.							      */
/**********************************************************************/
SPBLK	**sp_flatten PROTO((SPTREE *));

/**********************************************************************/
/*   Function  returns  a string containing statistics on splay tree  */
/*   operations of the passed tree.				      */
/**********************************************************************/
char	*spstats PROTO((SPTREE *));
# endif
