/*	$NetBSD: t_cgd_adiantum.c,v 1.5 2020/08/20 13:33:54 riastradh Exp $	*/

/*-
 * Copyright (c) 2020 The NetBSD Foundation, Inc.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE NETBSD FOUNDATION, INC. AND CONTRIBUTORS
 * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
 * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE FOUNDATION OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <sys/types.h>

#include <fcntl.h>
#include <stdint.h>
#include <stdint.h>
#include <stdio.h>
#include <unistd.h>
#include <util.h>

#include <dev/cgdvar.h>

#include <rump/rump.h>
#include <rump/rump_syscalls.h>

#include <atf-c.h>

#include "h_macros.h"

#define	MAXSECSIZE	512	/* for now; should be cgd parameter */
#define	IMGSIZE		0x101*512

/* Used as buffer for cgd device I/O, must be at least 32-bit aligned.  */
static const uint8_t zerosector[512] __aligned(4);

static const struct {
	uint8_t		key[32];
	uint64_t	blkno;
	unsigned	secsize;
	const uint8_t	*ptxt;
	const uint8_t	*ctxt;
} C[] = {
	[0] = {
		.key = {0},
		.blkno = 0,
		.secsize = 512,
		.ptxt = zerosector,
		.ctxt = (const uint8_t[512]) {
			0x51,0x6d,0xe2,0x81, 0x26,0xd5,0xc8,0xd7,
			0xff,0xc6,0xc2,0xff, 0x39,0xbf,0x15,0x15,
			0x46,0x80,0x44,0x65, 0x76,0xa1,0x56,0xae,
			0xa0,0xb6,0x44,0x05, 0xb7,0xb1,0x32,0x23,
			0x80,0x07,0xdd,0x31, 0x57,0x69,0xf5,0x10,
			0x2d,0x53,0x54,0x8a, 0x1c,0x30,0x15,0x53,
			0x40,0xb4,0x75,0xb2, 0xa1,0x8a,0xbe,0xdf,
			0xf7,0x10,0xe0,0x38, 0xf9,0x70,0x29,0xda,
			0xf0,0x95,0xcd,0xe9, 0x47,0xa1,0x32,0xa3,
			0x83,0xca,0xe3,0x36, 0xc3,0x21,0x00,0xc2,
			0x0a,0xb4,0x0e,0x67, 0x69,0xe6,0xe8,0x14,
			0x74,0x98,0x69,0xd0, 0x6e,0xab,0x23,0xbc,
			0xa9,0x1e,0xf8,0x2d, 0x98,0x59,0x98,0x81,
			0x29,0x70,0xa8,0x1e, 0x26,0x13,0xba,0x53,
			0x9e,0x83,0xe9,0x35, 0x73,0x8c,0xf9,0xb6,
			0x10,0x17,0xda,0xe8, 0x21,0xcc,0x7d,0xd2,
			0x8e,0x23,0xb9,0x63, 0xde,0xcf,0xa7,0x53,
			0x56,0x1c,0xc8,0x53, 0x91,0x17,0x8f,0xec,
			0x93,0x66,0x8b,0x0f, 0x18,0x6e,0xa5,0x9d,
			0x8e,0x99,0x36,0x1c, 0x23,0xb6,0x0f,0x5d,
			0x75,0xc3,0xfd,0x35, 0xc5,0x68,0x9c,0xe1,
			0xba,0x19,0x1a,0x09, 0xca,0x40,0x1f,0xee,
			0x0f,0x76,0x84,0x92, 0x72,0xdf,0x62,0x1b,
			0x2e,0xa9,0x36,0xbe, 0xca,0x7e,0xc6,0x69,
			0xc6,0x27,0xf8,0x12, 0xbf,0x6e,0xd3,0xf0,
			0xb0,0x10,0x9c,0x67, 0x76,0x40,0xc8,0x36,
			0x8e,0x73,0xec,0xa2, 0xdb,0x4a,0x0a,0xd9,
			0x1b,0xa3,0x28,0x30, 0x84,0xa4,0xff,0xa0,
			0xe7,0x1e,0xf4,0xb2, 0xfe,0x59,0x79,0xdf,
			0x8d,0x66,0x12,0xac, 0xf6,0x1a,0x0f,0xa6,
			0x4e,0x86,0x8c,0x80, 0x95,0x11,0xee,0x55,
			0xe3,0xe0,0x43,0x56, 0xa2,0xfc,0xa2,0xbd,
			0xad,0x6f,0xfc,0xf9, 0x4c,0x04,0x51,0xf4,
			0xd9,0x17,0x96,0xdc, 0xd3,0xd0,0xd7,0xeb,
			0xa8,0xdc,0x34,0x65, 0xc7,0xcf,0xed,0x06,
			0xf8,0xa3,0xff,0x31, 0x3e,0x15,0x2f,0x62,
			0x8c,0x73,0x7f,0x8c, 0x80,0x4d,0x4b,0x6d,
			0xcf,0xc6,0xd0,0xdd, 0x7e,0x3a,0x1e,0x88,
			0xb7,0xdd,0x23,0xa6, 0xa0,0x0d,0x6c,0xaf,
			0xd6,0x5b,0xfd,0x76, 0x66,0xee,0x02,0xa6,
			0x10,0xda,0x42,0xfb, 0x15,0xc3,0xe4,0xa7,
			0x8b,0x2b,0xfa,0x5d, 0xba,0xce,0xcd,0x9f,
			0x76,0x38,0x66,0xff, 0x74,0x08,0x34,0xf3,
			0x3d,0x12,0xf4,0x8d, 0x5e,0x54,0x2b,0x37,
			0x06,0xd3,0x03,0xc9, 0xd9,0x29,0x53,0x65,
			0x76,0x00,0x24,0x50, 0x30,0x06,0x6c,0x69,
			0x31,0xcc,0x89,0x7c, 0x97,0xae,0xac,0x74,
			0x35,0x43,0xa3,0xe5, 0x40,0x58,0x3d,0xb9,
			0x08,0x46,0x5e,0x5f, 0x07,0xc5,0x41,0x32,
			0xab,0xa4,0x5a,0xab, 0x59,0x2b,0x54,0xee,
			0x24,0x92,0xd3,0x08, 0xb8,0x99,0x9e,0x13,
			0x3c,0x2c,0x05,0xe6, 0xc1,0x6f,0xa1,0x5d,
			0xa9,0x09,0x1a,0x96, 0x76,0xe4,0x31,0xc6,
			0xcc,0xad,0x28,0x58, 0x73,0x4d,0x1a,0x19,
			0x3d,0xcd,0xaf,0x8c, 0xd8,0x24,0xff,0x72,
			0xdc,0x4e,0x07,0x6e, 0xd8,0xbc,0x3b,0x2b,
			0xf5,0xe5,0xfa,0x30, 0x7d,0xaa,0x59,0x40,
			0x78,0x01,0xa4,0x55, 0xdc,0xe6,0x7b,0xae,
			0x87,0x8e,0x11,0xbb, 0x65,0xf7,0x8a,0x4f,
			0x37,0x7e,0xe1,0xac, 0x62,0xf1,0x64,0x8f,
			0xc1,0xfd,0x3e,0x34, 0x1f,0x60,0xba,0x61,
			0x98,0xae,0x19,0xce, 0x54,0x22,0x64,0x09,
			0x67,0x82,0x6b,0x4b, 0xdf,0x26,0x77,0xde,
			0xd6,0x13,0x00,0xee, 0x2c,0x18,0x49,0xd9,
		},
	},
	[1] = {
		.key = {0},
		.blkno = 1,
		.secsize = 512,
		.ptxt = zerosector,
		.ctxt = (const uint8_t[512]) {
			0xf2,0x23,0x68,0x5a, 0x15,0x11,0x56,0xa1,
			0x71,0x57,0x5c,0x5e, 0x32,0xd4,0xdd,0xbb,
			0x7a,0x0c,0x84,0x23, 0xe9,0x2f,0x1b,0x63,
			0x3c,0x4d,0xad,0xfd, 0x6e,0xc0,0xdb,0x79,
			0x23,0xa5,0x13,0xfe, 0x17,0x3c,0x4a,0x27,
			0xb9,0xbc,0xf0,0xf6, 0x67,0x98,0xa8,0x64,
			0xce,0xf0,0x17,0x0a, 0xa8,0x05,0x0f,0xf2,
			0xff,0xb0,0x7a,0x9e, 0x1a,0xcf,0x5d,0x0e,
			0x9f,0xb0,0x9a,0xd0, 0x7c,0xf2,0x88,0x96,
			0xe2,0x8d,0xdb,0xa2, 0x19,0x30,0x3d,0x5d,
			0x66,0x28,0x40,0x53, 0xb9,0x8d,0xbb,0x24,
			0x3a,0x4c,0x00,0xac, 0x20,0x86,0x96,0x83,
			0x2c,0x77,0x5e,0x18, 0x0c,0xfa,0x27,0x51,
			0xe1,0x5d,0xd3,0xd9, 0xe1,0x5d,0x27,0x1f,
			0x49,0x74,0xfd,0x2a, 0xc3,0xe5,0xa0,0xf6,
			0x5a,0x58,0xe3,0x1f, 0x4a,0xa6,0xc2,0x25,
			0xe4,0xb5,0xc8,0x0d, 0x9f,0xa7,0xc0,0x6e,
			0xab,0xb3,0xfc,0x9f, 0xe1,0x72,0x8a,0x69,
			0xf1,0xc6,0x54,0xb8, 0xeb,0x70,0xed,0xfe,
			0x95,0xf7,0x0d,0x55, 0x95,0x13,0x7a,0x82,
			0xac,0x83,0xd2,0xa3, 0xdc,0x5b,0xba,0x4e,
			0xae,0xdd,0xe9,0x22, 0x9e,0xe2,0x72,0xaf,
			0x9a,0xc0,0x53,0x96, 0xb9,0x7d,0x47,0x28,
			0x4a,0x93,0x6a,0xfb, 0x59,0x25,0x49,0x39,
			0xda,0x23,0xe8,0x28, 0x42,0xba,0x58,0x26,
			0x29,0xf5,0x4c,0x85, 0xbb,0x62,0xfc,0x12,
			0x28,0xbd,0xec,0x3f, 0xf4,0x86,0x80,0xf0,
			0x69,0x81,0x99,0xe3, 0x95,0x0d,0xe8,0x8f,
			0xeb,0x60,0xb6,0x2a, 0xbf,0xf1,0x41,0xe4,
			0x68,0x4f,0x4b,0xe3, 0x49,0x2c,0x1e,0xad,
			0x0d,0x8f,0x63,0x40, 0xb9,0xee,0x4d,0x09,
			0x12,0x45,0x97,0x64, 0x97,0xd5,0x5f,0xa3,
			0xb0,0x4b,0xdf,0x3f, 0x59,0x9f,0xab,0x12,
			0x3d,0x4b,0x54,0xdc, 0xea,0xe0,0x55,0x5e,
			0x1c,0xfd,0xe9,0x7e, 0x40,0x24,0x88,0x6c,
			0x8d,0xfc,0xc2,0x57, 0xd2,0x37,0xb2,0x12,
			0xc2,0x03,0x0d,0xac, 0xb8,0x9b,0x62,0x61,
			0x23,0xc0,0x7a,0x06, 0xdb,0x62,0x86,0x06,
			0xaf,0xa5,0x98,0x75, 0xd9,0x4e,0x8a,0xf2,
			0xc5,0x64,0xad,0xf2, 0xf4,0xc2,0x7f,0xa2,
			0x25,0xf4,0xd0,0x44, 0x57,0x8b,0x89,0xe2,
			0x08,0xea,0x86,0x72, 0x37,0xe3,0x7e,0x92,
			0x22,0xa0,0x32,0x05, 0x30,0x90,0xcc,0x44,
			0x6f,0x2c,0x75,0xae, 0x28,0x90,0x34,0xe3,
			0x05,0x88,0xcd,0x77, 0x1d,0x6a,0x72,0x56,
			0x49,0x3f,0x3d,0x0b, 0x49,0x04,0x98,0x65,
			0x66,0x0e,0xfd,0x7d, 0xca,0x32,0x74,0x66,
			0xa0,0xd7,0x04,0xdb, 0x83,0x4b,0x7f,0x83,
			0x22,0x43,0x98,0x93, 0x0d,0x0b,0xb1,0x8d,
			0x8c,0x8b,0x9e,0x08, 0xb9,0xb0,0xd9,0x82,
			0xcd,0x20,0x5e,0x19, 0x5d,0xa0,0x6a,0x71,
			0x05,0xf9,0x18,0x3d, 0x6b,0xb7,0xb6,0x56,
			0x03,0xa3,0x53,0x58, 0x7d,0xf8,0x25,0xca,
			0x26,0x02,0xc1,0xa6, 0x72,0x70,0xc3,0xe3,
			0x59,0x64,0xe1,0x25, 0x34,0x79,0xb3,0x5e,
			0x08,0xe9,0xb8,0x91, 0xb6,0x5d,0x3a,0x44,
			0x20,0x60,0x61,0xf4, 0x28,0x93,0x8f,0x89,
			0xbe,0xea,0x55,0xda, 0x43,0x38,0x96,0xc8,
			0x50,0x01,0x09,0xaf, 0x76,0x92,0x83,0xae,
			0x3b,0x82,0x6f,0x49, 0x0b,0x18,0x9c,0xef,
			0x92,0x06,0x11,0xeb, 0x41,0x34,0xf4,0x7b,
			0xc4,0x9a,0x9f,0xe4, 0xb4,0xe7,0x1a,0x84,
			0xd8,0x8b,0x3a,0x29, 0xb5,0x4e,0xf3,0x97,
			0x6c,0xef,0xe9,0x62, 0x21,0x89,0x23,0xfd,
		},
	},
	[2] = {
		.key = {0},
		.blkno = 0x100,
		.secsize = 512,
		.ptxt = zerosector,
		.ctxt = (const uint8_t[512]) {
			0x32,0x26,0xaf,0x56, 0xbc,0x43,0xac,0x37,
			0xb2,0x8d,0xa4,0xfb, 0x32,0xdc,0x09,0x03,
			0xd9,0x44,0xce,0x4e, 0x70,0xaf,0xed,0x83,
			0x4b,0x9c,0x85,0x11, 0xd2,0x6a,0x70,0x15,
			0xea,0x7b,0x5e,0xac, 0x5d,0x08,0x25,0xd7,
			0x8c,0x23,0x7c,0x15, 0xb7,0x20,0xd1,0x08,
			0xe0,0x81,0x71,0xbe, 0x68,0xca,0xe2,0xcd,
			0x98,0xe5,0x40,0xe0, 0xf5,0x84,0xcc,0x6a,
			0x3c,0xa0,0xe8,0x2c, 0x02,0x4c,0x95,0xb5,
			0x58,0x86,0x86,0x61, 0x71,0x7f,0xd7,0xf9,
			0xd9,0x64,0x80,0xf6, 0xea,0x92,0xbc,0x65,
			0x3b,0x07,0x77,0xaa, 0xb1,0xb1,0xf5,0xd6,
			0x6d,0x89,0x63,0x14, 0xc0,0xcc,0x7a,0x2b,
			0xc4,0x32,0x63,0xda, 0xa6,0xc6,0xc8,0xc6,
			0x4c,0x4e,0x10,0x63, 0x3b,0x93,0x80,0x77,
			0x3e,0x54,0xd2,0x38, 0x13,0x79,0xbc,0x6c,
			0x0b,0xd4,0x71,0x5c, 0x26,0xc0,0x81,0x09,
			0xc7,0xd8,0x7a,0x04, 0x58,0x2e,0x50,0x6a,
			0x3d,0xca,0xa1,0x66, 0x72,0xca,0xee,0x5a,
			0xdd,0x13,0x67,0xb1, 0x54,0x72,0x41,0x2d,
			0xfd,0x95,0x24,0xe3, 0x96,0x4a,0x41,0x03,
			0xeb,0xeb,0x99,0x49, 0x52,0xac,0x3a,0x28,
			0x81,0x54,0x1a,0xfb, 0xc3,0xcd,0x8e,0x9d,
			0x0c,0x64,0x95,0xbb, 0x27,0xb8,0x6b,0x51,
			0x7b,0xc4,0x57,0xc9, 0x29,0x4e,0x85,0x31,
			0x1c,0xaa,0x63,0x2e, 0x7a,0x37,0x2e,0x06,
			0xdc,0x58,0x39,0x3b, 0x60,0x34,0x59,0x15,
			0x4f,0xba,0x33,0x52, 0x13,0xb0,0x7b,0x7c,
			0x7e,0x00,0x0b,0x49, 0x15,0x9c,0x48,0xf4,
			0x67,0xdd,0xc6,0x72, 0x87,0xbe,0xe7,0xf7,
			0x21,0x95,0x82,0xc3, 0x41,0x3b,0x19,0xe3,
			0xf3,0x28,0xcc,0x14, 0x5f,0xae,0x6f,0x07,
			0x35,0x94,0x05,0x46, 0x02,0x5c,0x3c,0x46,
			0xb1,0x2d,0xeb,0x6e, 0xa0,0x0f,0xea,0x40,
			0x3e,0x35,0x6e,0x50, 0xc4,0x22,0xeb,0x93,
			0xba,0x49,0xfb,0xf0, 0x8e,0x2a,0xa1,0xaf,
			0xf4,0x91,0xb2,0xc5, 0x7d,0x8e,0xba,0x45,
			0x53,0x75,0xc3,0xcc, 0x3e,0x02,0x0e,0x4d,
			0x2e,0xda,0x45,0xd2, 0x31,0xc7,0x1b,0x6b,
			0x99,0x71,0x8d,0xd8, 0x8c,0x94,0xa2,0x02,
			0x6c,0xb0,0x32,0x8f, 0xce,0x04,0x61,0x0a,
			0x3f,0x17,0x3a,0x28, 0xda,0x31,0xdc,0xec,
			0xbc,0xea,0x1b,0x37, 0x9b,0x36,0x04,0xb1,
			0xb5,0x7f,0xfe,0x1a, 0xd8,0x11,0xb7,0x0a,
			0x77,0x2e,0x6d,0x22, 0x79,0x9e,0x54,0x47,
			0xea,0xf5,0x17,0x38, 0xd0,0xe2,0x23,0x68,
			0x92,0x88,0x42,0x59, 0x2c,0x61,0x53,0x2b,
			0x99,0xed,0x7b,0x85, 0xfb,0xb8,0xe8,0x0c,
			0x4b,0x81,0x1e,0x0f, 0x42,0x04,0x8b,0x55,
			0x2c,0x34,0x46,0x98, 0x9c,0x47,0x08,0x70,
			0x46,0x45,0x5e,0xa8, 0x62,0x92,0x94,0xcd,
			0x73,0x1c,0xef,0x8b, 0x96,0x5f,0x6d,0x76,
			0x07,0x99,0x6f,0xe0, 0x1d,0xdc,0x1d,0x1c,
			0x3f,0xb4,0x5f,0x9b, 0x34,0x0c,0x75,0x10,
			0x7e,0x0d,0xf8,0xbb, 0xc3,0x8a,0x2a,0x15,
			0x01,0x3a,0x56,0x73, 0x5b,0xe9,0x5f,0xf2,
			0x6a,0x1b,0x17,0xce, 0xf3,0x3e,0xc9,0xdf,
			0x76,0xe8,0xcd,0xf2, 0x6d,0xb1,0xdc,0x29,
			0x8c,0xa3,0x89,0x73, 0x69,0x86,0xa9,0x05,
			0xbe,0x63,0xc8,0x7c, 0x36,0xc0,0x88,0x74,
			0x64,0x91,0xdd,0xb7, 0x92,0x73,0x7e,0xc1,
			0x01,0x95,0xb3,0x95, 0x53,0x33,0x16,0xcd,
			0xe9,0xd7,0x56,0x61, 0x71,0x49,0x24,0x9b,
			0x9a,0x10,0x7e,0x50, 0x7e,0xd3,0xe2,0x9d,
		},
	},
};

static void
hexdump(const void *buf, size_t len)
{
	const unsigned char *p = buf;
	size_t i;

	for (i = 0; i < len; i++) {
		if (i % 16 == 8)
			printf(" ");
		printf(" %02hhx", p[i]);
		if ((i + 1) % 16 == 0)
			printf("\n");
	}
	if (i % 16)
		printf("\n");
}

static int
configure_cgd(int fd, const char *dkpath, const char *alg,
    const char *ivmethod, const void *key, size_t keybytes)
{
	struct cgd_ioctl ci;

	memset(&ci, 0, sizeof(ci));
	ci.ci_disk = dkpath;
	ci.ci_alg = alg;
	ci.ci_ivmethod = ivmethod;
	ci.ci_keylen = 8*keybytes;
	ci.ci_key = key;
	ci.ci_blocksize = (size_t)-1;

	return rump_sys_ioctl(fd, CGDIOCSET, &ci);
}

static int
unconfigure_cgd(int fd)
{
	struct cgd_ioctl ci;

	return rump_sys_ioctl(fd, CGDIOCCLR, &ci);
}

ATF_TC(cgd_adiantum);
ATF_TC_HEAD(cgd_adiantum, tc)
{

	atf_tc_set_md_var(tc, "descr", "Adiantum tests");
}

ATF_TC_BODY(cgd_adiantum, tc)
{
	static uint8_t buf[MAXSECSIZE];
	static const char imgpath[] = "adiantum.img";
	static const char dkpath[] = "/dev/dk";
	char cgdpath[MAXPATHLEN];
	int dkfd, cgdfd;
	unsigned i;
	ssize_t nwrit, nread;

	rump_init();

	RL(dkfd = open(imgpath, O_CREAT|O_RDWR|O_TRUNC, 0600));
	RL(ftruncate(dkfd, IMGSIZE));
	RL(rump_pub_etfs_register_withsize(dkpath, imgpath, RUMP_ETFS_BLK, 0,
		IMGSIZE));
	snprintf(cgdpath, sizeof cgdpath, "/dev/rcgd0%c",
	    getrawpartition() + 'a');
	RL(cgdfd = rump_sys_open(cgdpath, O_RDWR));

	for (i = 0; i < __arraycount(C); i++) {
		/* write the plaintext out via cgd */
		RL(configure_cgd(cgdfd, dkpath, "adiantum", "encblkno1",
			C[i].key, 32));
		RL(nwrit = rump_sys_pwrite(cgdfd, C[i].ptxt, C[i].secsize,
			C[i].blkno * C[i].secsize));
		RL(unconfigure_cgd(cgdfd));
		if ((size_t)nwrit != C[i].secsize) {
			atf_tc_fail_nonfatal("truncated write: %zd != %u",
			    nwrit, C[i].secsize);
			continue;
		}

		/* read the ciphertext out from the underlying file */
		RL(nread = pread(dkfd, buf, C[i].secsize,
			C[i].blkno * C[i].secsize));
		if ((size_t)nread != C[i].secsize) {
			atf_tc_fail_nonfatal("truncated read: %zd != %u",
			    nread, C[i].secsize);
			continue;
		}
		if (memcmp(buf, C[i].ctxt, C[i].secsize)) {
			hexdump(buf, C[i].secsize);
			hexdump(C[i].ctxt, C[i].secsize);
			atf_tc_fail_nonfatal("case %u ctxt mismatch", i);
			continue;
		}

		/* read the plaintext back via cgd */
		RL(configure_cgd(cgdfd, dkpath, "adiantum", "encblkno1",
			C[i].key, 32));
		RL(nread = rump_sys_pread(cgdfd, buf, C[i].secsize,
			C[i].blkno * C[i].secsize));
		RL(unconfigure_cgd(cgdfd));
		if ((size_t)nread != C[i].secsize) {
			atf_tc_fail_nonfatal("truncated read: %zd != %u",
			    nread, C[i].secsize);
			continue;
		}
		if (memcmp(buf, C[i].ptxt, C[i].secsize)) {
			hexdump(buf, C[i].secsize);
			atf_tc_fail_nonfatal("case %u ptxt mismatch", i);
			continue;
		}
	}

	RL(rump_sys_close(cgdfd));
	RL(close(dkfd));
}

ATF_TP_ADD_TCS(tp)
{

	ATF_TP_ADD_TC(tp, cgd_adiantum);

	return atf_no_error();
}
