/* sim.h -- simulation system definitions */

typedef long id_t;
typedef double systime_t;

struct entity {
	id_t id;
	systime_t time, qtime;
	double feature[1];
	struct entity * next;
	};

typedef struct entity ENTITY;

struct event{
	systime_t time;
#ifdef DEBUG
	char *fs;
#endif
	void (*f)();
	ENTITY *entity;
	struct event *next;
	};

typedef struct event EVENT;

struct queue{
	char * name;
	ENTITY *top, *end;
	unsigned short max_len, cur_len, min_len, count;
	systime_t total_wait;
	};

typedef struct queue QUEUE_STRUCT;

struct serv_data{
	char * name;
	unsigned long total;
	double total_busy, srvtime;
	};

typedef struct serv_data SERV_DATA;

void sdump();
#ifdef DEBUG
void schedule(void(*)(),char *,ENTITY *,systime_t /* time */);
#else
void schedule(void(*)(),ENTITY *,systime_t /* time */);
#endif
void engine();
EVENT *nextevent();
ENTITY *create(systime_t /* time of creation */,double /* feature */);
ENTITY 	*topq(QUEUE_STRUCT * /* entity queue */),
		*dq(QUEUE_STRUCT * /* entity queue */);
void nq(QUEUE_STRUCT * /* entity queue */, ENTITY *);

#ifndef DEF_SPACE
extern double now; /* current time in system */
extern EVENT *sched;
extern id_t entity_id;
#else
double now; /* current time in system */
EVENT *sched;
id_t entity_id;
#endif

#ifdef DEBUG
#define TRACE(name)	printf("%f " #name "\n",now);
#define SDUMP()	sdump()
#define PR_ENTITY(entity) \
	if (entity) printf("\tentity: id=%ld, time=%f, feature=%f, next=%x\n",\
			entity->id,entity->time,entity->feature[0],entity->next);\
	else printf("\tentity=0\n");
#else
#define TRACE(name)
#define SDUMP()
#define PR_ENTITY(entity)
#endif

#ifdef DEBUG
#define SCHEDULE(name,entity,time)	\
	schedule(name,#name,(ENTITY *)(entity),(systime_t)(time));
#else
#define SCHEDULE(name,entity,time)	\
	schedule(name,(ENTITY *)(entity),(systime_t)(time));
#endif

#define DEFINE_CREATE(name)		\
	void name();

#define CREATE(name,create_time,feature,create_freq,service)	\
void name()\
{\
extern systime_t now;\
	TRACE(name);\
	service(create((systime_t)create_time,(double)feature));\
	SCHEDULE(name,0,now+create_freq);\
	}

#define DEFINE_SERVICE(name)		\
	systime_t name(ENTITY *);\
	SERV_DATA name##_s={#name};

#define SERVICE(name,nservers,delay,next)	\
double name(entity)\
	ENTITY *entity;\
{\
static systime_t srvtime[nservers], wait_time;\
extern systime_t now;\
unsigned short i;\
	TRACE(name);\
	PR_ENTITY(entity);\
	for (i=0;i<nservers;i++) if (srvtime[i] <= now) break;\
	if (i == nservers) return -1;\
	/* note: delay may be a function call */\
	wait_time = delay;\
	srvtime[i] = now + wait_time;\
	SCHEDULE(next,entity,srvtime[i]);\
	/* compute statistics -- \
        only total served and total busy time done for now */\
	name##_s.total++;\
	name##_s.total_busy += wait_time;\
	name##_s.srvtime=srvtime[i];\
	return srvtime[i];\
	}

#define DEFINE_QUEUE(name)		\
	void name(ENTITY *);\
	QUEUE_STRUCT name##_q={#name};

#define QUEUE(name,service)		\
void name(entity)\
	ENTITY *entity;\
{\
systime_t srvtime;\
ENTITY * top;\
	TRACE(name);\
	PR_ENTITY(entity);\
	if (entity) nq(&name##_q,entity);\
	if ((top = topq(&name##_q)) &&\
			(srvtime=service(top)) >= 0){\
		dq(&name##_q);\
		SCHEDULE(name,0,srvtime);\
		}\
	/* compute statistics */\
	}
