(* The semantic objects in the static semantics of Modules,
   Definition v3, page 31 ff. *)

(*
$File: Common/MODULE_STATOBJECT.sml $
$Date: 1993/03/05 14:38:45 $
$Revision: 1.16 $
$Locker: birkedal $
*)

(*$MODULE_STATOBJECT*)
signature MODULE_STATOBJECT =
  sig
    (********
    Identifiers
    ********)

    type id

    (********
    Qualified structure identifiers
    ********)

    type longstrid

    (********
    Type constructors
    ********)

    type longtycon

    (********
    Exception constructors
    ********)

    type excon

    (********
    Structure identifiers
    ********)

    type strid

    (********
    Variables
    ********)

    type var
    val mk_var : id -> var

    (********
    Type constructors
    ********)

    type tycon

    (********
    Type names
    ********)

    type TyName
    val freshTyName : {name : tycon, arity : int, equality : bool} -> TyName

    (********
    Syntactic type variables
    ********)

    type SyntaxTyVar

    (*********
     Syntactic type expressions (abstract syntax tree for ty's)
     ********)

    type ty 
    val syntaxtyvarsTy : ty -> SyntaxTyVar EqSet.Set
    (********
    Type variables
    ********)

    eqtype TyVar
    val mkExplicitTyVar : SyntaxTyVar -> TyVar

    (****
    Types
    ****)

    type Type and ConsType

    val TypeExn : Type
    and mkTypeTyVar : TyVar -> Type
    and mkTypeConsType : ConsType -> Type
    and mkTypeArrow    : Type * Type -> Type

    val mkConsType : Type list * TyName -> ConsType

    (********
    TypeSchemes
    ********)

    type TypeScheme

    val Type_in_TypeScheme : Type -> TypeScheme

    (********
    Structure Names
    ********)

    eqtype StrName
    val freshStrName : unit -> StrName

    val bogus_StrName: StrName

    (********
    Structure Name sets
    ********)

    type StrNameSet 
    val singleM : StrName -> StrNameSet

    (********
    Name sets
    ********)

    type NameSet
    val isIn_StrName  : StrName * NameSet -> bool
    and isIn_TyName   : TyName  * NameSet -> bool
    and mkNameSet_Str : StrNameSet -> NameSet

    (********
    Environments
    ********)

    type Env

    (********
    Structures
    ********)

    type Str 
    val mkStr  : StrName * Env -> Str
    and unStr  : Str -> StrName * Env
    and namesS : Str -> NameSet

    (********
    Type functions
    ********)

    eqtype TypeFcn
    val admits_equality   : TypeFcn -> bool
    and grounded_TypeFcn  : TypeFcn * NameSet -> bool
    and TyName_in_TypeFcn : TyName -> TypeFcn
    and arity_TypeFcn     : TypeFcn -> int
    and unTyName_TypeFcn  : TypeFcn -> TyName Option

    (********
    Constructor environments
    ********)

    type ConEnv

    (********
    Type structures
    ********)

    eqtype TyStr			(* `eqtype' because we compare it
					   against `bogus_TyStr'. *)
    val mkTyStr      : TypeFcn * ConEnv -> TyStr
    and unTyStr      : TyStr -> TypeFcn * ConEnv
    and TyStr_shares : TyStr * TyStr -> bool
    and Theta_of     : TyStr -> TypeFcn
    and isTyName     : TyStr -> bool
    and CE_of_TyStr  : TyStr -> ConEnv

    val bogus_TyStr: TyStr

    (********
    Signatures
    ********)

    type Sig
    val trivSig  : Sig
    and unSig    : Sig -> NameSet * Str
    and mkSig    : NameSet * Str -> Sig 
    and namesSig : Sig -> NameSet

    val bogus_Sig: Sig

    (********
    Functor signatures
    ********)

    type FunSig
    val mkFunSig    : NameSet * Str * Sig -> FunSig
    and unFunSig    : FunSig -> NameSet * Str * Sig
    and namesFunSig : FunSig -> NameSet

    (********
    instanceSig returns the structure from a signature where names bound by
    the formal parameter have been instantiated to fresh (generative) names
    ********)

    val instanceSig : Sig -> Str

    (********
    Close a structure to get a signature
    *********
    The NameSet is the set of `rigid' names which cannot be quantified
    ********)

    val closeStr : NameSet * Str -> Sig


    (********
    Realisations
    ********)

    type Realisation
    val Id  : Realisation

    val onStrName : Realisation -> StrName -> StrName
    val onTypeFcn : Realisation -> TypeFcn -> TypeFcn
    val onCE      : Realisation -> ConEnv -> ConEnv
    val onTyStr   : Realisation -> TyStr -> TyStr
    val onS       : Realisation * Str -> Str
    val onE       : Realisation * Env -> Env

    val bogus_Realisation: Realisation

    val oo       : Realisation * Realisation -> Realisation
    val renaming : NameSet -> Realisation

    val mkRea :
      (StrName list * StrName) list * (TyName list * TypeFcn) list -> Realisation

   (* Datatype used to return results of matches. Rather than invent a big
      datatype and convert it into ErrorInfo.info, we just use the error info
      type directly. *)

    type ErrorInfo
    datatype 'a MatchResult =
        OK of 'a
      | ERROR of ErrorInfo

    (********
    If S'= sigMatchStr(Sigma,S) succeeds, then S matches Sigma
    and S' is an instance of Sigma and S enriches S'
    ********)

    val sigMatchStr : Sig * Str -> Str MatchResult

    (********
    If Sig'= funsigMatchStr(Phi,S) succeeds, then S matches the
    argument signature of the functor signature Phi and Sig' is 
    the signature of the actual result of the functor application
    ********)

    val funsigMatchStr : FunSig * Str -> Sig MatchResult

    (********
    Check a structure for type-explicitness
    *********
    The NameSet is the rigid NameSet of the Basis in which the check is made
    ********)

    val type_explicit : NameSet * Str -> bool

    (********
     Well-formedness of signature 
     ********
     The NameSet is the rigid NameSet of the Basis in with the check is made
     ********)

     val wellformedsig : NameSet * Str -> bool


    (*******
    equality-principal signature
    ********
    The NameSet is the rigid NameSet of the Basis in which the check is made
    ********)

    datatype EqPrincipalResult =
	PRINCIPAL of Str 
      | FAIL of longtycon list

    val equality_principal : NameSet * Str -> EqPrincipalResult

    (********
    PrettyPrinting
    ********)

    type StringTree
    val layoutSig: Sig -> StringTree
    and layoutFunSig: FunSig -> StringTree
  end;
