
/* Copyright (C) 1988, 1989 Herve' Touati, Aquarius Project, UC Berkeley */

/* Copyright Herve' Touati, Aquarius Project, UC Berkeley */

% List processing utilities used in the PLM compiler.
% These are a subset of a much larger collection.

%    list(L) succeeds if and only if L is a list.
%    nonlist(S) succeeds if and only if S is not a list.
%    No unification is done.

list(Term) :- nonvar(Term), Term=[_|_].

nonlist(Term) :- list(Term), !, fail.
nonlist(_).

%    full_list(L) succeeds if and only if L is a complete list (all
%    the cdrs are also lists) or [].
%    No unification is done.

full_list(L) :- var(L), !, fail.
full_list([]) :- !.
full_list([_|L]) :- full_list(L).


%   concat(Part1, Part2, Combined) and
%   append(Part1, Part2, Combined)
%   are true when all three arguments are lists, and the members of Combined
%   are the members of Part1 followed by the members of Part2.  It may be
%   used to form Combined from a given Part1 and Part2, or to take a given
%   Combined apart.  E.g. we could define member/2 (from SetUtl.Pl) as
%	member(X, L) :- append(_, [X|_], L).

concat([], L, L).
concat([H|L1], L2, [H|Res]) :- concat(L1, L2, Res).

append(A, B, C) :- concat(A, B, C).


%   concat(Part1, Part2, Part3, Combined)
%   concat(Part1, Part2, Part3, Part4, Combined)
%   concat(Part1, Part2, Part3, Part4, Part5, Combined)
%   are extensions of concat for three, four, and five sublists respectively.
%   Concat can also be used to decompose lists into all combinations of 
%   three, four, and five parts.

concat([], L2, L3, Res) :- concat(L2, L3, Res).
concat([H|L1], L2, L3, [H|Res]) :- concat(L1, L2, L3, Res).

concat([], L2, L3, L4, Res) :- concat(L2, L3, L4, Res).
concat([H|L1], L2, L3, L4, [H|Res]) :- concat(L1, L2, L3, L4, Res).

concat([], L2, L3, L4, L5, Res) :- concat(L2, L3, L4, L5, Res).
concat([H|L1], L2, L3, L4, L5, [H|Res]) :- concat(L1, L2, L3, L4, L5, Res).


% length of a list

my_length([], 0).
my_length([_|L], N) :- my_length(L, N1), N is N1+1 .

%   last(List, Last)
%   is true when List is a list and Last is its last element.  This could
%   be defined as last(L,X) :- append(_, [X], L).

last([Last], Last) :- !.
last([_|List], Last) :- last(List, Last).


%   member(Elem, List)
%   is true if Elem is a member of List.  This can be used as a checker, as a
%   generator of elements, or as a generator of lists.

member(X, [X|_]).
member(X, [_|L]) :- member(X, L).


%   memberchk(Elem, List)
%   same as member, but used only to test membership.   This is faster and uses
%   less memory than the more general version.
%   memberv does not use unification to test for membership.

memberchk(X, [X|_]) :- !.
memberchk(X, [_|L]) :- memberchk(X, L).

memberv(X, [Y|_]) :- X==Y, !.
memberv(X, [_|L]) :- memberv(X, L).


%   reverse(List, Reversed)
%   is true when List and Reversed are lists with the same elements
%   but in opposite orders.  rev/2 is a synonym for reverse/2.

rev(List, Reversed) :- reverse(List, [], Reversed).

reverse(List, Reversed) :- reverse(List, [], Reversed).

reverse([], Reversed, Reversed).
reverse([Head|Tail], Sofar, Reversed) :-
	reverse(Tail, [Head|Sofar], Reversed).



%   flatten(List, FlatList-Link)
%   flattens a list by removing all nesting.  FlatList consists of all atoms
%   nested to any depth in List, but all on one level.

flatten([], Link-Link).
flatten([A|L], [A|F]-Link) :- 
	(atomic(A);var(A)), !, flatten(L, F-Link).
flatten([A|L], F-Link) :- 
	flatten(A, F-FL), flatten(L, FL-Link). 


%   mapcar(Structure, List1, List2) 
%   Calls the goal Structure+elem of List1+elem of List2 for each pair
%   of elements of List1 and List2.
%   generalization of the Lisp function mapcar.

mapcar(Call, List1, List2) :- !,
	Call=..[Func|Args],
	xmapcar(Func, Args, List1, List2).

xmapcar(Func, Args, [A|L1], [B|L2]) :- !,
	concat(Args, [A,B], GoalArgs),
	Goal=..[Func|GoalArgs],
	call(Goal), !,
	xmapcar(Func, Args, L1, L2).
xmapcar(_, _, [], []).

%   mapcar(Functor, List1, List2, List3)
%   same as Lisp's mapcar, except has three arguments.

mapcar(Func, [A|L1], [B|L2], [C|L3]) :- !,
	Term=..[Func, A, B, C],
	call(Term), !,
	mapcar(Func, L1, L2, L3).
mapcar(_, [], [], []).



%   listify(Structure, ListForm)
%   converts a general structure to a Lisp-like list form.

listify(X, X) :- (atomic(X); var(X)), !.
listify(Structure, [Func|LArgs]) :-
	Structure=..[Func|Args],
	mapcar(listify, Args, LArgs).


%   linkify(List, DiffList-Link)
%   converts a list into a difference list.

linkify([], Link-Link).
linkify([A|List], [A|DiffList]-Link) :-
	linkify(List, DiffList-Link).
