(* Copyright 1989 by AT&T Bell Laboratories *)

(***************************************************************************
 *                         IMPORTANT NOTES                                 *
 *                                                                         *
 *        OFFSET and RECORD accesspath SELp should not be                  *
 *                  generated by this module.                              *
 ***************************************************************************)



signature CONVERT = 
  sig val convert : Lambda.lexp -> CPS.function * CPS.lty Intmap.intmap 
  end

structure Convert : CONVERT = struct

open Access CPS

(***************************************************************************
 *              CONSTANTS AND UTILITY FUNCTIONS                            *
 ***************************************************************************)
val OFFp0 = OFFp 0

fun sublist test =
  let fun subl(a::r) = if test a then a::(subl r) else subl r
        | subl x = x
  in  subl
  end

local open Lambda 
in
  fun translatepath [v] = VAR v
    | translatepath (x::p) = SELECT(x,translatepath p)
    | translatepath nil = ErrorMsg.impossible "convert.translatepath nil"

  fun isboxedRep(CONSTANT _) = false
    | isboxedRep(_) = true

  fun isboxed (DATAcon(_,rep,_)) = isboxedRep rep
    | isboxed (REALcon _) = true
    | isboxed (STRINGcon s) = (size s <> 1)
    | isboxed _ = false
end 

val INTty = Lambda.INTty
val REALty = Lambda.REALty
val BOOLty = Lambda.BOOLty
val BOXEDty = Lambda.BOXEDty
val RECORDty = Lambda.RECORDty
val ARROWty = Lambda.ARROWty
val CONTty = Lambda.CONTty
val IntOpTy = ARROWty(RECORDty[INTty,INTty],INTty)
val seqTy = ARROWty(RECORDty[BOXEDty,BOXEDty],BOOLty)
val checkArrowTy = Transtypes.checkArrowTy
val checkRecordTy = Transtypes.checkRecordTy
val checkContTy = Transtypes.checkContTy
fun err0 s = ErrorMsg.impossible ("representation analysis: "^s)

fun getConTy(Lambda.ARROWty(t1,t2)) = t2    (* non-constant constructor *)
  | getConTy t = t                          (* constant constructor *)

fun getDeconTy(Lambda.ARROWty(t1,t2)) = t1    
  | getDeconTy t = t

val sortcases = Sort.sort (fn ((i:int,_),(j,_)) => i>j)

fun nthcdr(l, 0) = l 
  | nthcdr(a::r, n) = nthcdr(r, n-1)
  | nthcdr _ = ErrorMsg.impossible "nthcdr in convert"

fun count test =
  let fun subl acc (a::r) = subl(if test a then 1+acc else acc) r
        | subl acc nil = acc
   in subl 0
  end

(***************************************************************************
 *                        THE MAIN FUNCTION                                *
 *     convert : Lambda.lexp -> CPS.cexp * CPS.lty Intmap.intmap           *
 ***************************************************************************)
fun convert lexp = let 
 
val repFlag = !System.Control.CG.representations
val NumOfReg = if repFlag then 10 else 0

local open Intmap
      exception Rename
      val m : value intmap = new(32, Rename)
      val rename = map m

   in fun ren v = rename v handle Rename => VAR v
      val newname = add m
  end

local open Intmap
   in exception TypeInfo
      val typtable : lty intmap = new(32, TypeInfo)
      val mapty = 
           if repFlag then 
             (fn v => (map typtable v) 
                   handle TypeInfo => 
       	            (List.app System.Print.say ["The lvar ", makestring v,
                        " is not in the current hashtable!\n"];
                        err0 "TypeInfo hash table in convert.sml"))
           else (fn v => BOXEDty)
      val addty = if repFlag then (add typtable) else (fn v => ())
      val rmvty = if repFlag then (rmv typtable) else (fn v => ())
      val nthty = if repFlag then nth else (fn _ => BOXEDty)
      fun grabty(VAR v) = mapty v
        | grabty(INT _) = INTty
        | grabty(REAL _) = REALty
        | grabty _ = BOXEDty
  end

fun mkfn(f,t) = let val v = mkLvar()
                    val _ = addty(v,t)
                 in f v
                end

fun mkv(t) = let val v = mkLvar()
                 val _ = addty(v,t)
              in v
             end

(***************************************************************************
 *        mkArgIn : lty * lvar -> lvar list * (cexp -> cexp)               *
 *       mkArgOut : lty * value -> value list * (cexp -> cexp)             *
 *                                                                         *
 * When the type of the argument x of a function f(x) is an "small"        *
 * unboxed record, f will be transformed to a multi-argument function      *
 * with #1(mkArgIn(...,x)) as its list of arguments.                       *
 *                                                                         *
 * When a function f is applied to a argument x, and x is of a "small"     *
 * unboxed record type, x will be flattened. #1(mkArgOut(...,x)) will      *
 * become the actual arguments of the function call f.                     *
 *                                                                         *
 * When the System.Control.CG.representations flag is turned off, all      *
 * these effects are gone.  (l >> NumOfReg)                                *
 ***************************************************************************)
fun mkArgIn(t,v) = 
  let val l = Transtypes.lengthOf(t)
      fun megl((vl1,f1),(vl2,f2)) = ((vl1 @ vl2), f1 o f2)

      fun h(tt as Lambda.RECORDty args, p) = 
	  let val wl = map (fn t => (t,mkv(t))) args
              val header = fn ce => 
                     RECORD(RK_RECORD,
                            map (fn (_,x) => (VAR x,OFFp0)) wl, p, ce)
 
           in fold megl (map h wl) (nil,header)
          end
        | h(tt, p) = ([p],fn ce => ce)

   in if l < NumOfReg then h(t,v)
       else ([v],fn ce => ce)
  end

fun mkArgOut(t,z as VAR v) = 
  let val l = Transtypes.lengthOf(t)
      fun megr((vl1,f1),(vl2,f2)) = ((vl1 @ vl2), f2 o f1)
  
      fun h(tt as Lambda.RECORDty args, p) = 
	  let val wl = map (fn t => (t,mkv(t))) args
              fun sel((_,x)::tl,i) = 
                   let val header = sel(tl,i+1)
                    in fn ce => SELECT(i, VAR p, x, header(ce))
                   end
                | sel(nil,i) = fn ce => ce

              val header = sel(wl,0)
                     
           in fold megr (map h wl) (nil,header)
          end
        | h (tt, p) = ([VAR p],fn ce => ce)
                            
   in if l < NumOfReg then h(t,v)
      else ([z],fn ce => ce)
  end
  | mkArgOut(t,z) = ([z],fn ce => ce) 


(***************************************************************************
 *                       PROCESSING THE SWITCH                             *
 ***************************************************************************)
fun switch1(e : value, cases : (int*cexp) list, d : lvar, (lo,hi)) =
  let val delta = 2
      fun collapse (l as (li,ui,ni,xi)::(lj,uj,nj,xj)::r ) =
	     if ((ni+nj) * delta > ui-lj) then collapse((lj,ui,ni+nj,xj)::r)
	     else l
	| collapse l = l
      fun f (z, x as (i,_)::r) = f(collapse((i,i,1,x)::z), r)
	| f (z, nil) = z
      fun tackon (stuff as (l,u,n,x)::r) = 
	    if n*delta > u-l andalso n>4 andalso hi>u
	    then tackon((l,u+1,n+1,x@[(u+1,APP(VAR d, [INT 0]))])::r)
	    else stuff
        | tackon nil = ErrorMsg.impossible "convert.3217"
      fun separate((z as (l,u,n,x))::r) =
	    if n<4 andalso n>1 
	    then let val ix as (i,_) = nth(x, (n-1))
		  in (i,i,1,[ix])::separate((l,l,n-1,x)::r)
		 end
	    else z :: separate r
	| separate nil = nil
      val chunks = rev (separate (tackon (f (nil,cases))))
      fun g(1,(l,h,1,(i,b)::_)::_,(lo,hi)) = 
      	    if lo=i andalso hi=i then b
	    else BRANCH(P.ineq,[e, INT i],mkv(BOXEDty),APP(VAR d, [INT 0]), b)
	| g(1,(l,h,n,x)::_,(lo,hi)) =
	    let fun f(0,_,_) = nil
		  | f(n,i,l as (j,b)::r) =
		       if i+lo = j then b::f(n-1,i+1,r)
		       else ((APP(VAR d, [INT 0]))::f(n,i+1,l))
                  | f _ = ErrorMsg.impossible "convert.987"
		val list = f(n,0,x)
		val body = if lo=0 then SWITCH(e, mkv(BOXEDty), list)
			   else (let fun f v = ARITH(P.-,[e, INT lo],v,
                                               SWITCH(VAR v,mkv(BOXEDty),list))
                                  in mkfn(f,INTty)
                                 end)
		val a = if (lo<l) then BRANCH(P.<,[e,INT l],mkv(BOXEDty),
					      APP(VAR d,[INT 0]),body)
			else body
	        val b = if (hi > h) then BRANCH(P.>,[e,INT h],mkv(BOXEDty),
					        APP(VAR d,[INT 0]), a)
			else a
	     in b
	    end
	| g(n,cases,(lo,hi)) =
	    let val n2 = n div 2
		val c2 = nthcdr(cases, n2)
                val (l,r) = case c2 of (l1,_,_,_)::r1 => (l1,r1)
                                     | _ => ErrorMsg.impossible "convert.111"
	     in BRANCH(P.<,[e, INT l], mkv(BOXEDty), 
		       g(n2,cases,(lo,l-1)), g(n-n2,c2,(l,hi)))
	    end
   in g (length chunks, chunks, (lo, hi))
  end

fun switch(e: value, l, d, inrange, t) =
  let val len = List.length l
      fun isbig i = i < ~0x20000000 orelse i >= 0x20000000
      val anybig = exists (isbig o #1) l
      val d' = case d of SOME d' => d' 
                       | NONE => mkv(ARROWty(INTty,t))
      fun ifelse nil = APP(VAR d',[INT 0])
        | ifelse ((i,b)::r) = 
            if isbig i 
            then if i = 0x3fffffff  (* maxint is extra-special *)
		 then let val i'=mkv(INTty) and i''=mkv(INTty)
		       in ARITH(P.+,[INT 0x1fffffff, INT 0x1fffffff],i',
			   ARITH(P.+,[VAR i', INT 1], i'',
			    BRANCH(P.ineq,[VAR i'',e],mkv(INTty),ifelse r,b)))
		      end				       
		 else let val j = i div 2
		          val i' = mkv(INTty)
		       in ARITH(P.+,[INT(i-j),INT j],i',
			    BRANCH(P.ineq,[VAR i',e],mkv(BOXEDty),ifelse r,b))
		      end
	    else BRANCH(P.ineq,[INT i, e], mkv(BOXEDty), ifelse r, b)
      fun ifelseN [(i,b)] = b
	| ifelseN ((i,b)::r) = 
             BRANCH(P.ineq,[INT i, e], mkv(BOXEDty), ifelseN r,b)
	| ifelseN _ = ErrorMsg.impossible "convert.224"  
      val l = sortcases l
   in case (anybig orelse len<4, inrange)
       of (true, NONE) => ifelse l
	| (true, SOME n) =>  if n+1=len then ifelseN l else ifelse l
	| (false, NONE) =>
	     let fun last [x] = x 
                   | last (_::r) = last r
                   | last nil = ErrorMsg.impossible "convert.last132"
		 val (hi,_) = last l 
                 val (low,r) = case l of (low',_)::r' => (low',r')
                                       | _ => ErrorMsg.impossible "convert.23"
	      in BRANCH(P.>,[INT low, e], mkv(BOXEDty), APP(VAR d',[INT 0]),
	       	   BRANCH(P.<,[INT hi, e],mkv(BOXEDty), APP(VAR d',[INT 0]),
			       switch1(e, l, d', (low,hi))))
	     end
	| (false, SOME n) => switch1(e, l, d', (0,n))
  end

(***************************************************************************
 *           preventEta : cexp * lty -> cexp * value                       *
 ***************************************************************************)
fun preventEta(c,argt) =
  let val f = mkv(CONTty argt) and v = mkv(argt)
      val (vl,header) = mkArgIn(argt,v)
      val b = header(c(VAR v))
   in case b
       of APP(w as VAR w', [VAR v']) => 
            if v=v' andalso v<>w'
		(* The case v=w' never turns up in practice,
		   but v<>v' does turn up. *)
	    then (fn x => x,w)
	    else (fn x => FIX([(f,vl,b)],x),VAR f)
	| _ => (fn x => FIX([(f,vl,b)],x),VAR f)
  end

(***************************************************************************
 *   convlist : Lambda.lexp list * (value list -> cexp) -> cexp            *
 ***************************************************************************)
fun convlist (el,c) =
  let fun f(le::r, vl) = conv(le, fn v => f(r,v::vl))
	| f(nil, vl) = c (rev vl)
   in f (el,nil)
  end

(***************************************************************************
 *   getargs : int * Lambda.lexp * (value list -> cexp) -> cexp            *
 ***************************************************************************)
and getargs(1,a,g) = conv(a, fn z => g[z])
  | getargs(n,Lambda.RECORD l,g) = convlist(l,g)
  | getargs(n,Lambda.VECTOR l,g) = convlist(l,g)
  | getargs(n,a,g) =
     let fun kont(v) = 
           let val lt = checkRecordTy(grabty(v))
               fun f(j,wl) = 
                 if j = n then g(rev wl)
                 else (let fun h(w) = SELECT(j,v,w,f(j+1,VAR w :: wl))
                        in mkfn(h,nthty(lt,j))
                       end)
            in f(0,nil)
           end
      in conv(a,kont)
     end

(***************************************************************************
 *   conv : Lambda.lexp * (value list -> cexp) -> cexp                     *
 ***************************************************************************)
and conv (le, c : value -> cexp) = 

 case le 
  of Lambda.APP(Lambda.PRIM(Access.P.CALLCC,_), f) =>
       let fun kont(vf) =
             let val (t1,t2) = checkArrowTy(grabty(vf))
                 val h = mkv(BOXEDty) and k' = mkv(t1) and x' = mkv(t2)
                 val (header,F) = preventEta(c,t2)
                 val (vl,_) = mkArgIn(t2,x')
              in header(LOOKER(P.gethdlr, [], h,
                         FIX([(k', vl, SETTER(P.sethdlr, [VAR h],
                                              APP(F, map VAR vl)))],
                             APP(vf,[VAR k',F]))))
             end
        in conv(f,kont)
       end
   | Lambda.APP(Lambda.PRIM(Access.P.CAPTURE,_), f) =>
       let fun kont(vf) = 
             let val (_,t) = checkArrowTy(grabty(vf))
                 val (header,F) = preventEta(c,t)
              in header(APP(vf,[F,F]))
             end
        in conv(f,kont)
       end
   | Lambda.APP(Lambda.PRIM(Access.P.THROW,_), v) => 
       let fun kont(kv) = 
             let val t = checkContTy(grabty(kv))
                 val f = mkv(ARROWty(t,BOXEDty))
                 val f'' = mkv(CONTty(BOXEDty)) and x = mkv(t)
                 val (vl,_) = mkArgIn(t,x)
              in FIX([(f,vl@[f''],APP(kv, map VAR vl))], c(VAR f))
             end
        in conv(v,kont)
       end
   | Lambda.APP(Lambda.PRIM(Access.P.CAST,lt), x) => 
       let fun kont(vx) =
             let val (_,t) = checkArrowTy(lt)
                 val u = mkv(t)
              in (* PURE(P.cast,[vx],u,c(VAR u)) *) c(vx)
             end
        in conv(x,kont)
       end
   | Lambda.APP(Lambda.PRIM(i,lt), a) => 
       let val (_,t) = checkArrowTy(lt)

           fun arith(n,i) = 
             let fun kont(vl) = mkfn(fn w => ARITH(i,vl,w,c(VAR w)),t)
              in getargs(n,a,kont)
             end

           fun setter(n,i) = 
             let fun kont(vl) = SETTER(i,vl,c(INT 0))
              in getargs(n,a,kont)
             end

           fun looker(n,i) =
             let fun kont(vl) = mkfn(fn w => LOOKER(i,vl,w,c(VAR w)),t)
              in getargs(n,a,kont)
             end

           fun pure(n,i) =
             let fun kont(vl) = mkfn(fn w => PURE(i,vl,w,c(VAR w)),t)
              in getargs(n,a,kont)
             end

  	   fun branch(n,i)= 
             let val (header,F) = preventEta(c,t) 
                 fun kont(vl) = header(BRANCH(i,vl,mkv(BOXEDty),
                                              APP(F,[INT 1]),APP(F,[INT 0])))
              in getargs(n,a,kont)
             end

        in case i
	    of Access.P.BOXED => branch(1,P.boxed)
	     | Access.P.UNBOXED => branch(1,P.boxed)
	     | Access.P.ILT => branch(2,P.<)
	     | Access.P.ILE => branch(2,P.<=)
	     | Access.P.IGT => branch(2,P.>)
	     | Access.P.IGE => branch(2,P.>=)
	     | Access.P.LESSU => branch(2,P.lessu)
	     | Access.P.GEQU => branch(2,P.gequ)
	     | Access.P.IEQL => branch(2,P.ieql)
	     | Access.P.INEQ => branch(2,P.ineq)
	     | Access.P.FEQLd => branch(2,P.feql)
	     | Access.P.FGEd => branch(2,P.fge)
	     | Access.P.FGTd => branch(2,P.fgt)
	     | Access.P.FLEd => branch(2,P.fle)
	     | Access.P.FLTd => branch(2,P.flt)
	     | Access.P.FNEQd => branch(2,P.fneq)

	     | Access.P.IMUL => arith(2,P.* )
	     | Access.P.IADD => arith(2,P.+)
	     | Access.P.ISUB => arith(2,P.-)
	     | Access.P.IDIV => arith(2,P.div)
	     | Access.P.FADDd => arith(2,P.fadd)
	     | Access.P.FDIVd => arith(2,P.fdiv)
	     | Access.P.FMULd => arith(2,P.fmul)
	     | Access.P.FSUBd => arith(2,P.fsub)
	     | Access.P.INEG => arith(1,P.~)
	     | Access.P.FLOOR => arith(1,P.floor)
	     | Access.P.ROUND => arith(1,P.round)

	     | Access.P.ORB => pure(2,P.orb)
	     | Access.P.ANDB => pure(2,P.andb)
	     | Access.P.XORB => pure(2,P.xorb)
	     | Access.P.RSHIFT => pure(2,P.rshift)
	     | Access.P.LSHIFT => pure(2,P.lshift)
	     | Access.P.SUBSCRIPTV => pure(2,P.subscriptv)
	     | Access.P.MAKEREF => pure(1,P.makeref)
	     | Access.P.NOTB => pure(1,P.notb)
	     | Access.P.LENGTH => pure(1,P.length)
	     | Access.P.OBJLENGTH => pure(1,P.objlength)
	     | Access.P.REAL => pure(1,P.real)
	     | Access.P.FNEGd => pure(1,P.fnegd)
	     | Access.P.FABSd => pure(1,P.fabsd)
	     | Access.P.GETTAG => pure(1, P.gettag)
	     | Access.P.MKSPECIAL => pure(2, P.mkspecial)

	     | Access.P.SUBSCRIPT => looker(2,P.subscript)
	     | Access.P.ORDOF => looker(2,P.ordof)
	     | Access.P.FSUBSCRIPTd => looker(2,P.subscriptf)
	     | Access.P.DEREF => looker(1,P.!)
	     | Access.P.GETHDLR => looker(0,P.gethdlr)
	     | Access.P.GETVAR  => looker(0,P.getvar)
	     | Access.P.GETSPECIAL => looker(1, P.getspecial)
	     | Access.P.DEFLVAR  => looker(0,P.deflvar)

	     | Access.P.SETHDLR => setter(1,P.sethdlr)
	     | Access.P.STORE => setter(3,P.store)
	     | Access.P.FUPDATEd => setter(3,P.updatef)
	     | Access.P.UNBOXEDUPDATE => setter(3,P.unboxedupdate)
	     | Access.P.BOXEDUPDATE => setter(3,P.boxedupdate)
	     | Access.P.UPDATE => setter(3,P.update)
	     | Access.P.SETVAR => setter(1,P.setvar)
	     | Access.P.SETSPECIAL => setter(2, P.setspecial)
	     | Access.P.USELVAR => setter(1,P.uselvar)

	     | _ => ErrorMsg.impossible ("calling with bad primop \"" 
                                         ^ (Access.P.pr_primop i) ^ "\"")
       end
   | Lambda.PRIM(i,lt) => 
       let val (t,_) = checkArrowTy(lt)
           val v = mkLvar()
           val e = Lambda.FN(v,t,Lambda.APP(le,Lambda.VAR v))
        in conv(e,c)
       end
   | Lambda.VAR v => c(ren v)
   | Lambda.APP(Lambda.FN(v,_,e),a) =>
       let fun kont(w) = 
             let val _ = newname(v,w)
                 val _ = case w of VAR w' => Access.sameName(v,w')
                                 | _ => ()
              in conv(e,c)
             end
        in conv(a,kont)
       end
   | Lambda.FN(v,t,e) =>   (* using "save" the reference cell is 
                              just dirty, but i can't find better way *)
       let val _ = addty(v,t)
           val save = ref BOXEDty and k = mkLvar()
           fun kont(vb) =
             let val t = grabty(vb)
                 val _ = (save := t)
                 val (ul,header) = mkArgOut(t,vb)
              in header(APP(VAR k,ul))
             end
           val ce = conv(e,kont)
           val t1 = !save
           val f = mkv(ARROWty(t,t1)) and _ = (addty(k,t1))
           val (vl,header) = mkArgIn(t,v)
        in FIX([(f,vl@[k],header(ce))],c(VAR f))
       end
   | Lambda.APP(f,a) =>   (* different from the old version in 
                             that header is now put in the middle
                             of evaluations between f and a, a bit odd *)
       let fun kont(vf) =
             let val (_,t) = checkArrowTy(grabty(vf))
                 val (header,F) = preventEta(c,t)
                 fun kont'(va) = 
                   let val (ul,header') = mkArgOut(t,va)
                    in header'(APP(vf,ul@[F]))
                   end
              in header(conv(a,kont'))
             end
        in conv(f,kont)
       end
   | Lambda.FIX(fl, tl,  el, body) =>
       let fun g(f::fl, t::tl, Lambda.FN(v,_,b)::el) =
                let val (t1,t2) = checkArrowTy(t)
                    val _ = addty(v,t1) and k = mkv(CONTty(t2))
                    val (vl,header) = mkArgIn(t2,v)
                    fun kont(vb) = 
                       let val (ul,header') = mkArgOut(t2,vb)
                        in header'(APP(VAR k,ul))
                       end
                    val be = conv(b,kont)
                 in (f,vl@[k],header(be))::g(fl,tl,el)
                end
             | g(nil, nil, nil) = nil
             | g _ = ErrorMsg.impossible "convert.conv.FIX1"

           fun h(f::fl,t::tl) = (addty(f,t);h(fl,tl))
             | h(nil,nil) = ()
             | h _ = ErrorMsg.impossible "convert.conv.FIX2"

           val _ = h(fl,tl)
        in FIX(g(fl,tl,el),conv(body,c))
       end
   | Lambda.INT i => ((i+i; c(INT i))
      handle Overflow =>
	 let open Lambda
 	  in conv(APP(PRIM(Access.P.IADD,IntOpTy),
                      RECORD([INT(i div 2),INT(i - i div 2)])),c)
	 end)
   | Lambda.REAL i => c(REAL i)
   | Lambda.STRING i => (case size i
			  of 1 => c(INT(ord i))   (* need coercion here *)
			   | _ => c(STRING i))
   | Lambda.RECORD [] => c (INT 0)
   | Lambda.VECTOR [] => c (INT 0)
   | Lambda.RECORD l => 
       let fun kont(vl) =
             let val w = mkv(RECORDty(map grabty vl))
              in RECORD(RK_RECORD, map (fn v => (v, OFFp0)) vl, w, c(VAR w))
             end
        in convlist(l,kont)
       end                
   | Lambda.VECTOR l => 
       let fun kont(vl) =
             let val w = mkv(BOXEDty)
              in RECORD(RK_VECTOR, map (fn v => (v, OFFp0)) vl, w, c(VAR w))
             end
        in convlist(l,kont)
       end
   | Lambda.SELECT(i, e) => 
       let fun kont(v) =
            let val t = nthty(checkRecordTy(grabty(v)),i)
                val w = mkv(t)
             in SELECT(i, v, w, c(VAR w))
            end
        in conv(e,kont)
       end
   | Lambda.CON((_,CONSTANT i,_),e) =>            (* coercion needed *)
       conv(Lambda.INT i, c)              
   | Lambda.CON((_,TAGGED i,t),e) => 
       let val t1 = BOXEDty      (* temporary: RECORDty[INTty,getConTy t] *)
           fun kont(v) =
              mkfn(fn x => RECORD(RK_RECORD,[(INT i,OFFp0), (v,OFFp0)], 
                                  x,c(VAR x)),t1)
        in conv(e,kont)
       end
   | Lambda.CON((_,TRANSPARENT,_),e) => conv(e,c)
   | Lambda.CON((_,UNTAGGEDREC _,t),e) => 
       let val t1 = getConTy t
           fun kont(VAR v) = (rmvty v; addty(v,t1); c(VAR v))
             | kont w = c(w)
        in conv(e,kont)
       end
   | Lambda.CON((_,UNTAGGED,t),e) => 
       let val t1 = getConTy t
           fun kont(VAR v) = (rmvty v; addty(v,t1); c(VAR v))
             | kont w = c(w)
        in conv(Lambda.RECORD[e],kont)
       end
   | Lambda.CON((_,TAGGEDREC(tag,n),t),e) => 
       let val t1 = getConTy t
           fun kont(v) =
             let val argty = checkRecordTy(grabty(v))
                 fun f(i) =
                   if i=n then (nil,fn ce => ce)
                   else (let val (vl,header) = f(i+1)
                             val z = mkv(nthty(argty,i))
                          in ((VAR z)::vl, fn ce => SELECT(i,v,z,header(ce)))
                         end)
                 val (wl,header) = f(0)
                 val ul = map (fn x => (x, OFFp0)) ((INT tag)::wl)
                 val u = mkv(t1)
              in header(RECORD(RK_RECORD,ul,u,c(VAR u)))
             end
        in conv(e,kont)
       end
   | Lambda.CON((_,VARIABLE(PATH p),t),e) =>
       let val t1 = getConTy t
           fun kont(v) = 
             let fun g([v]) = (fn ce => ce, ren v, mapty v)
                   | g(i::r) = 
                       let val (header,nv,t) = g(r)
                           val t1 = nthty(checkRecordTy(t),i)
                           val z = mkv(t1)
                        in (fn ce => header(SELECT(i,nv,z,ce)), VAR z, t1)
                       end
                   | g _ = ErrorMsg.impossible "convert.278"
                 val (header,nv,t) = g(p)
                 val u = mkv(BOXEDty) (* temporary: RECORDty[t,grabty(v)] *)
              in header(RECORD(RK_RECORD,[(nv,OFFp0),(v,OFFp0)],u,c(VAR u)))
             end        
        in conv(e,kont)
       end
   | Lambda.CON((_,VARIABLEc(PATH p),_),e) => 
       let fun g [v] = Lambda.VAR v
	     | g (i::r) = Lambda.SELECT(i, g r)
             | g nil = ErrorMsg.impossible "convert.2137"
        in conv(g p, c)
       end
   | Lambda.DECON((_,CONSTANT _,_),e) => c(INT 0)       (* coercion needed *)
   | Lambda.DECON((_,TAGGED i,t),e) => 
       let val t1 = getDeconTy(t)
           fun kont(v) = mkfn(fn x => SELECT(1,v,x,c(VAR x)), t1)
        in conv(e,kont)
       end
   | Lambda.DECON((_,TRANSPARENT,_),e) => conv(e,c)
   | Lambda.DECON((_,UNTAGGEDREC _,t),e) => 
       let val t1 = getDeconTy t
           fun kont(VAR v) = (rmvty v; addty(v,t1); c(VAR v))
             | kont w = c(w)
        in conv(e,kont)
       end
   | Lambda.DECON((_,UNTAGGED,t),e) => 
       let val t1 = getDeconTy(t)
           fun kont(v) = mkfn(fn x => SELECT(0,v,x,c(VAR x)), t1)
        in conv(e,kont)
       end
   | Lambda.DECON((_,VARIABLE(PATH p),t),e) => 
       let val t1 = getDeconTy(t)
           fun kont(v) = mkfn(fn x => SELECT(1,v,x,c(VAR x)), t1)
        in conv(e,kont)
       end
   | Lambda.DECON((_,TAGGEDREC(tag,n),t),e) => 
       let val t1 = getDeconTy(t)
           val args = checkRecordTy(t1)
           fun kont(v) = 
             let fun f(i) = 
                  if i = n then (fn ce => ce,nil)
                  else (let val (header,vl) = f(i+1)
                            val w = mkv(nthty(args,i))
                         in (fn ce => SELECT(i+1,v,w,header(ce)), (VAR w)::vl)
                        end)
                 val (header,vl) = f(0)
                 val w = mkv(t1)
              in header(RECORD(RK_RECORD,(map (fn v => (v,OFFp0)) vl),
                               w,c(VAR w)))
             end
        in conv(e,kont)
       end
   | Lambda.SWITCH(e,_,l as (Lambda.DATAcon(_,Access.VARIABLE _,_), _)::_, 
                   SOME d) => exnswitch(e,l,d,c)
   | Lambda.SWITCH(e,_,l as (Lambda.DATAcon(_,Access.VARIABLEc _,_), _)::_, 
                   SOME d) => exnswitch(e,l,d,c)
   | Lambda.SWITCH(e,_,l as (Lambda.REALcon _, _)::_, SOME d) =>
       let val k = mkLvar() and save = ref BOXEDty
           fun kont(w) = 
             let fun kont1(z) = 
                   let val t = grabty(z)
                       val _ = (save := t)
                       val (vl,header) = mkArgOut(t,z)
                    in header(APP(VAR k,vl))
                   end
                 fun g((Lambda.REALcon rval, x)::r) =
		        BRANCH(P.fneq,[w, REAL rval],mkv(BOXEDty),
		               g r,conv(x,kont1))
	           | g nil = conv(d,kont1)
	           | g _ = ErrorMsg.impossible "convert.81"
	      in g l 
             end
           val body = conv(e,kont)
           val t = !save
           val v = mkv(t) and _ = (addty(k,CONTty t))
           val (vl,header) = mkArgIn(t,v)
        in FIX([(k,vl,header(c(VAR v)))],body)
       end
   | Lambda.SWITCH(e,_,l as (Lambda.INTcon _, _)::_, SOME d) =>
       let val k = mkLvar() and save = ref BOXEDty and df = mkv(CONTty INTty)
           fun kont1(z) = 
             let val t = grabty(z) 
                 val _ = (save := t)
                 val (vl,header) = mkArgOut(t,z)
              in header(APP(VAR k,vl))
             end 
           fun kont(w) = 
             let fun g(Lambda.INTcon j, a) = (j,conv(a,kont1))
                   | g _ = ErrorMsg.impossible "convert.42894"
              in switch(w,map g l,SOME df,NONE,!save)
             end
           val de = conv(d,kont1)
           val body = conv(e,kont)
           val t = !save 
           val v = mkv(t) and _ = addty(k,CONTty t)
           val (vl,header) = mkArgIn(t,v)
        in FIX([(k,vl,header(c(VAR v)))],FIX([(df,[mkv(INTty)],de)],body))
       end
   | Lambda.SWITCH(e,_,l as (Lambda.STRINGcon _, _)::_, SOME d) =>
       let val df = mkv(CONTty INTty) and F = mkLvar() and save = ref BOXEDty
           fun kont1(z) =
             let val t = grabty(z)
                 val _ = (save := t)
                 val (vl,header) = mkArgOut(t,z)
              in header(APP(VAR F,vl))
             end
           val b = sublist (isboxed o #1) l 
           val u = sublist (not o isboxed o #1) l
           fun g(Lambda.STRINGcon j,e) = (ord j, conv(e,kont1))
             | g _ = ErrorMsg.impossible "convert.899"
           val z = map g u
           val (p1,p2) = case (!CoreInfo.stringequalPath)
                          of [a,b] => (a,b)
                           | _ => ErrorMsg.impossible "convert.p1p2"

           fun genbs(w,nil) = APP(VAR df,[INT 0])
             | genbs(w,cases) =
                 let val len1 = mkv(INTty)
                     fun g nil = APP(VAR df,[INT 0])
                       | g((Lambda.STRINGcon s, x)::r) = 
                           let val ssize = size s
  			       val k = mkv(CONTty INTty) and seq = mkv(seqTy)
                               and c2= mkv(CONTty INTty) and ans=mkv(INTty)
                               and pair = mkv(RECORDty[BOXEDty,BOXEDty])

                               val xe = conv(x,kont1)
                               val kfl = [(k,[mkv(INTty)],g r)]
                               val flhd = 
                                 if ssize=0 then fn ce => ce 
                                 else (fn ce => 
                                         FIX([(c2,[ans],BRANCH(P.ieql,
                                              [VAR ans,INT 0],mkv(BOXEDty),
                                              APP(VAR k,[INT 0]),xe))],ce))
                               fun header ce = 
                                  BRANCH(P.ineq,[INT ssize,VAR len1],
                                         mkv(BOXEDty),APP(VAR k,[INT 0]),ce)
                               val body = 
                                 if ssize=0 then header(xe)
                                 else header(SELECT(p1,ren p2,seq,
                                   if repFlag then
                                        APP(VAR seq,[w,STRING s,VAR c2])
                                   else RECORD(RK_RECORD,[(w,OFFp0),
                                         (STRING s,OFFp0)],pair,
                                         APP(VAR seq,[VAR pair,VAR c2]))))

		            in FIX(kfl,flhd(body))
                           end
                       | g _ = ErrorMsg.impossible "convert.73287"
                  in PURE(P.length,[w],len1,g cases)
		 end
        
           fun kont(w) =
             let val genu = switch(w,z,SOME df,NONE,!save)
                 val genb = genbs(w,b)
	      in BRANCH(P.boxed,[w],mkv(BOXEDty),genb,genu)
             end

           val de = conv(d,kont1)
           val body = conv(e,kont)
           val t = !save 
           val x = mkv(t) and _ = addty(F,CONTty t)
           val (xl,header) = mkArgIn(t,x)
        in FIX([(F,xl,header(c(VAR x)))],FIX([(df,[mkv(INTty)],de)],body))
       end

   | Lambda.SWITCH(e,l,[a as (Lambda.DATAcon(_,Access.CONSTANT 0,_),_),
		        b as (Lambda.DATAcon(_,Access.CONSTANT 1,_),_)], 
                   NONE) =>
       conv(Lambda.SWITCH(e,l,[b,a],NONE),c)
   | Lambda.SWITCH(x as (Lambda.APP(Lambda.PRIM(i,_), args),_,
                   [(Lambda.DATAcon(_,Access.CONSTANT 1,_),e1),
	           (Lambda.DATAcon(_,Access.CONSTANT 0,_),e2)],NONE)) =>
       let fun g i' =
	     let val k = mkLvar() and save = ref BOXEDty
                 fun kont(w) = 
                   let val t = grabty(w) 
                       val _ = (save := t)
                       val (ul,header1) = mkArgOut(t,w)
                    in header1(APP(VAR k,ul))
                   end
                 val ce1 = conv(e1,kont) and ce2 = conv(e2,kont)
                 val t = !save
                 val _ = addty(k,CONTty t) and v = mkv(t)
                 val (vl,header) = mkArgIn(t,v)
	      in FIX([(k,vl,header(c(VAR v)))],
                  getargs(2,args,fn vl => BRANCH(i',vl,mkv(BOXEDty),ce1,ce2)))
	     end
        in case i
	    of Access.P.ILT => g(P.<)
	     | Access.P.ILE => g(P.<=)
  	     | Access.P.IGT => g(P.>)
	     | Access.P.IGE => g(P.>=)
	     | Access.P.LESSU => g(P.lessu)
	     | Access.P.GEQU => g(P.gequ)
	     | Access.P.IEQL => g(P.ieql)
	     | Access.P.INEQ => g(P.ineq)
	     | Access.P.FEQLd => g(P.feql)
	     | Access.P.FGEd => g(P.fge)
	     | Access.P.FGTd => g(P.fgt)
	     | Access.P.FLEd => g(P.fle)
	     | Access.P.FLTd => g(P.flt)
	     | Access.P.FNEQd => g(P.fneq)
	     | _ => genswitch(x,c)
       end
   | Lambda.SWITCH x => genswitch(x,c)
   | Lambda.RAISE(e,t) =>
       let fun kont(w) = 
             let val h = mkv(CONTty(grabty w))
              in LOOKER(P.gethdlr,[],h,APP(VAR h,[w]))
             end
        in conv(e,kont)
       end
   | Lambda.HANDLE(a,b) =>
       let fun kont(vb) =
             let val (_,t) = checkArrowTy(grabty(vb))
                 val h = mkv(CONTty BOXEDty) and v = mkv(BOXEDty)
                 val k = mkv(CONTty BOXEDty)
                 val (header,F) = preventEta(c,t)
                 fun kont1(va) = 
                   let val (ul,header1) = mkArgOut(t,va)
                    in SETTER(P.sethdlr,[VAR h],
                              header1(APP(F,ul)))
                   end
              in LOOKER(P.gethdlr,[],h,
                   header(FIX([(k,[v],SETTER(P.sethdlr,[VAR h],
                                            APP(vb,[VAR v,F])))],
                           SETTER(P.sethdlr,[VAR k],conv(a,kont1)))))
             end
        in conv(b,kont)
       end
   | Lambda.WRAP(t,le) => 
       let fun kont(VAR w) = (rmvty(w); addty(w,BOXEDty); c(VAR w))
             | kont u = c(u)   (* coercions may be needed here *)
        in conv(le,kont)
       end
   | Lambda.UNWRAP(t,le) => 
       let fun kont(VAR w) = (rmvty(w); addty(w,t); c(VAR w))
             | kont u = c(u)
        in conv(le,kont)
       end
   | _ => ErrorMsg.impossible "convert.sml 7432894"

(***************************************************************************
 * exnswitch : Lambda.lexp * (Lambda.con * Lambda.lexp) list *             *
 *               Lambda.lexp * (value -> cexp) -> cexp                     *
 ***************************************************************************)
and exnswitch(e,l,d,c) =
  let val k = mkLvar() and save = ref BOXEDty
      fun kont1(w) = 
        let val t = grabty(w)
            val _ = (save := t)
            val (ul,header) = mkArgOut(t,w)
         in header(APP(VAR k,ul))
        end

      val cl = map (fn (s,le) => (s,conv(le,kont1))) l
      val de = conv(d,kont1)

      val t = !save
      val v = mkv(t) and _ = addty(k,CONTty t)
      val (vl,header) = mkArgIn(t,v)

      fun kont(w) =
        let fun g((Lambda.DATAcon(_,VARIABLEc(PATH p),_),x)::r) =
	          conv(translatepath(0::p), 
                       fn v => BRANCH(P.ineq,[w,v],mkv(BOXEDty),g r,x))
	      | g((Lambda.DATAcon(_,VARIABLE(PATH p),_),x)::r) =
		  conv(translatepath p, 
                       fn v => BRANCH(P.ineq,[w,v],mkv(BOXEDty),g r,x))
	      | g nil = de
	      | g _ = ErrorMsg.impossible "convert.21"
	 in g cl
	end

   in FIX([(k,vl,header(c(VAR v)))],conv(Lambda.SELECT(0,e),kont))
  end


(***************************************************************************
 * genswitch : (Lambda.lexp * Access.conrep list * (Lambda.con *           *
 *                 Lambda.lexp) list * Lambda.lexp option) *               *
 *              (value -> cexp) -> cexp                                    *
 ***************************************************************************)
and genswitch ((e, sign, l: (Lambda.con * Lambda.lexp) list, d),c) =
  let val df = mkv(CONTty INTty) and save = ref BOXEDty and F = mkLvar()
      fun kont1(z) = 
        let val t = grabty z
            val _ = (save := t)
            val (ul,header) = mkArgOut(t,z)
         in header(APP(VAR F,ul))
        end

      fun tag (Lambda.DATAcon(_,CONSTANT i,_), e) = (i,conv(e,kont1))
        | tag (Lambda.DATAcon(_,TAGGED i,_), e) = (i,conv(e,kont1))
        | tag (Lambda.DATAcon(_,TAGGEDREC(i,_),_), e) = (i,conv(e,kont1))
	| tag (c,e) = (0,conv(e,kont1))

      val dfhd = case d 
         of NONE => (fn ce => ce) 
	  | SOME d' => (fn ce => FIX([(df,[mkv(INTty)],conv(d',kont1))],ce))

      val boxed = sublist (isboxed o #1) l
      val unboxed = sublist (not o isboxed o #1) l
      val b = map tag boxed and u = map tag unboxed
      val t = !save
      val v = mkv(t) and _ = (addty(F,CONTty t)) and x = mkv(INTty)
      val (vl,header) = mkArgIn(t,v)
      val fl = [(F,vl,header(c(VAR v)))]

      fun kont(w) = 
	case (count isboxedRep sign, count (not o isboxedRep) sign)
	 of (0, n) => switch(w,u,SOME df,SOME(n-1),t)
	  | (n, 0) => SELECT(0,w,x,switch(VAR x,b,SOME df,SOME(n-1),t))
	  | (1, nu) => BRANCH(P.boxed,[w],mkv(BOXEDty),
		              switch((INT 0),b,SOME df,SOME 0,t),
		              switch(w,u,SOME df,SOME(nu-1),t))
	  | (nb,nu) => BRANCH(P.boxed,[w],mkv(BOXEDty),
		              SELECT(0,w,x,
                                     switch(VAR x,b,SOME df,SOME(nb-1),t)),
		              switch(w,u,SOME df,SOME(nu-1),t))
   in FIX(fl,dfhd(conv(e,kont)))
  end

val save = ref BOXEDty and k = mkLvar() and f = mkLvar() and v = mkLvar()
fun kont(w) = 
  let val t = grabty(w)
      val (t1,t2) = checkArrowTy(t)
      val _ = (addty(k,CONTty t2); addty(f,t); addty(v,t1); save := t1)
      val (ul,header) = mkArgOut(t1,VAR v)
   in header(APP(w,ul@[VAR k]))
  end
val body = conv(lexp,kont)
val (vl,header) = mkArgIn(!save,v)
 
in ((f,vl@[k],header(body)),typtable)
end

end

