#charset "us-ascii"
/* 
 *  Copyright (c) 2005 by Kevin Forchione. All rights reserved.
 *   
 *  This file is part of the TADS 3 Story Guide
 *
 *  storyguide.t
 *
 *  A mechanism for defining, activating, and deactivating Story Points.
 *  These objects can be used to signal to author code that an important 
 *  story event is in progress. Story Points can set Actor States, and 
 *  execute any code necessary to the event they represent. Each Story
 *  Point, once completed, can then generate a pool of further candidate
 *  Story Points. 
 *
 *  The general process consists of:
 *
 *  a.  Select a "story point" object from a candidates pool 
 *      based on the story point object's evaluation of game state.
 *  b.  The story point object then sets a global value to 
 *      indicate that we are at that particular stage (Possibly 
 *      useful for modifying object behaviors)
 *  c.  Adjust Actor States (i.e. in TADS 3 this governs actor 
 *      behaviors) to those appropriate for the story point
 *  d.  Execute any code pertinent to the story point (i.e. perhaps 
 *      initiate daemons/fuses, set other object states, play out 
 *      "cut" scenes, etc.)
 *  e.  Determine if the particular story point is completed. If so, 
 *      then
 *      1.  Adjust Actor States to their default values
 *      2.  Create a new pool of candidates out of an overall story
 *          point pool.
 *  f.  Goto a.
 */
#pragma once
 
/* include the TADS and T3 system headers */
#include "tads.h"
#include "t3.h"
#include "storyguide.h"


/*
 *  An abstract class that stores the candidate story points 
 *  pool and defines methods that are meant to keep StoryModerator
 *  objects synchronized.
 */
class StoryModerator: Schedulable, PreinitObject
{
    /*
     *  An abstract property that represents the initial
     *  story point(s) of this story.
     */
    initStoryPoints     = []

    /*
     *  An abstract property that represents the pool of 
     *  candidate story points.
     */
    storyPointPool      = []

    /*
     *  An abstract property that represents the current
     *  story point being processed by StoryModerator objects.
     */
    currStoryPoint      = nil

    /* 
     *  An abstract property that represents the previous
     *  story point processed by StoryModerator objects.
     */
    prevStoryPoint      = nil

    /* 
     *  We let the StoryModerator objects have a turn as soon as 
     *  the game starts 
     */
    nextRunTime         = 0    
    
    /*
     *  Each instance of StoryModerator should override this 
     *  method to control its functioning. Two things must
     *  be accomplished. The method should increment the 
     *  next run time, and it should return true to indicate
     *  that the method has finished its job.
     */
    executeTurn() { return true; }     

    /*
     *  Select the next story point to activate 
     *  or deactivate. Each instance of StoryGuid
     *  must override this method.
     */
    selectNextStoryPoint() { return nil; }

    /*
     *  Initialize our story point candidates.
     */
    execute()
    {
        setNextStoryPointPool(initStoryPoints);
    }

    /*
     *  Sets the next story point pool.
     *
     *  This method should be used to keep this property 
     *  in StoryModerator objects synchronized.
     */
    setNextStoryPointPool(lst)
    {
        StoryModerator.storyPointPool = lst;  
    }

    /*
     *  Sets the current story point value.
     *
     *  This method should be used to keep this property 
     *  in StoryModerator objects synchronized.
     */
    setCurrStoryPoint(point)
    {
        StoryModerator.currStoryPoint = point;
    }

    /*
     *  Sets the previous story point value.
     *
     *  This method should be used to keep this property 
     *  in StoryModerator objects synchronized.
     */
    setPrevStoryPoint(point)
    {
        StoryModerator.prevStoryPoint = point;
    }

    /*
     *  Sets the next run time for this object.
     *
     *  This method should be overridden by StoryModerator instances.
     */
    setNextRunTime() {}
}

/*
 *  The storyModeratorStart object should be the first Schedulable
 *  object to be executed by the runScheduler() function. This 
 *  allows this object to select an appropriate Story Point and
 *  activate it prior to command execution.
 */
storyModeratorStart: StoryModerator
{
    /*
     *  This object should be the first Schedulable object
     *  to execute in the runScheduler() function.
     */
    scheduleOrder       = 1
    
    executeTurn()
    {
        local point;

        /*
         *  If we don't have a current Story Point yet we will
         *  select one. Otherwise the current one will remain in
         *  effect until it has been deactivated by the storyModeratorEnd.
         */
        if (gCurrStoryPoint == nil)
        {
            /*
             *  Select the next plot point to activate
             */
            point = selectNextStoryPoint();

            if (point)
                point.activateStoryPoint();
        }
        else
            gCurrStoryPoint.continueStoryPoint(true);

        /*
         *  Increment the next run time in the StoryModerator
         *  class. This keeps the instances of StoryModerator 
         *  in sync with each other. We only increment the 
         *  StoryModerator value at this stage because we don't 
         *  know the minimum run time value of other Schedulable
         *  objects at this stage.
         */
        setNextRunTime();
        
        /*
         *  Signal that we're done with our processing.
         */
        return true;
    }

    /*
     *  Select the next story point to activate 
     *  from our story point candidates list.
     */
    selectNextStoryPoint()
    {
        local vec, point;

        vec = new Vector(10);
        foreach (local cand in storyPointPool)
        {
            if (cand.checkActive() && !cand.active_ && !cand.completed_)
                vec.append(cand);
        }

        /*
         *  If our vector has any elements then we'll 
         *  select the first element by default.
         */
        if (vec.length())
            point = vec[1];

        return point;
    }

    /*
     *  Sets the next run time for this StoryModerator object. By default
     *  we merely increment the time by 1 because we don't know the 
     *  next run times of other Schedulable objects.
     */
    setNextRunTime()
    {
        incNextRunTime(1);
    }
}

/*
 *  The storyModeratorEnd object should be the last Schedulable
 *  object to be executed by the runScheduler() function. This 
 *  allows this object to select an appropriate Story Point and
 *  deactivate it after command & daemon/fuse execution.
 */
storyModeratorEnd: StoryModerator
{
    /*
     *  This object should be the last Schedulable object
     *  to execute in the runScheduler() function.
     */
    scheduleOrder       = 9999
    
    /*
     *  Executes a single turn of this StoryModerator object.
     */
    executeTurn()
    {
        local point;

        /*
         *  Select the next story point to deactivate. 
         *  This will either be the story point assigned
         *  to gCurrStoryPoint or it will be nil.
         */
        point = selectNextStoryPoint();

        if (point)
        {
            local lst;

            /*
             *  Send the current story point the message 
             *  to deactivate and retrieve the list of candidate
             *  story points from the current story point.
             */
            lst = point.deactivateStoryPoint();

            /*
             *  Set up the StoryModerator's next story point pool
             */
            setNextStoryPointPool(lst);
        }
        else if (gCurrStoryPoint)
            gCurrStoryPoint.continueStoryPoint(nil);

        /*
         *  Increment the next run time in the StoryModerator
         *  class. This keeps the instances of StoryModerator 
         *  in sync with each other. This instance indicates
         *  that the next run time should be set to the minimum
         *  run time of other non-StoryModerator Schedulable objects.
         */
        setNextRunTime();

        /*
         *  Synchronnize the runtimes of both story frames
         *  so that they execute within the same turn.
         */
        syncStoryModeratorRunTimes();
        
        /*
         *  Signal that we're done with our processing.
         */
        return true;
    }

    /*
     *  Returns the global story point set in the storyModeratorBegin
     *  object if the point is to be marked as completed; otherwise
     *  the method returns nil.
     */
    selectNextStoryPoint()
    {
        local cand;

        cand = gCurrStoryPoint;

        if (cand == nil)
            return nil;

        if (!cand.checkCompleted())
            return nil;
            
        if (!cand.active_ || cand.completed_)
            return nil;
       
        return cand;
    }

    /*
     *  Sets the next run time for this StoryModerator object. By default
     *  we set the time to the minimum run time of the collection of
     *  non-StoryModerator Schedulable objects.
     */
    setNextRunTime()
    {
        nextRunTime = computeMinNextRunTime();
    }

    /*
     *  Returns the minimum next run time value of all
     *  non-StoryModerator Schedulable objects.
     *
     *  This code is a derivation of that provided by the ADV3 library's
     *  runScheduler() by Michael J. Roberts.
     */
    computeMinNextRunTime()
    {
        local minTime;

        /* find the lowest time at which something is ready to run */
        minTime = nil;
        foreach (local cur in Schedulable.allSchedulables)
        {
            local curTime;

            /* 
             *  Eliminate StoryModerator class objects from 
             *  our next run time calculation. 
             */
            if (cur.ofKind(StoryModerator))
                continue;

            /* get this item's next eligible run time */
            curTime = cur.getNextRunTime();

            /* 
             *   if it's not nil, and it's equal to or below the
             *   lowest we've seen so far, note it 
             */
            if (curTime != nil && (minTime == nil || curTime <= minTime))
            {
                /* note the new lowest schedulable time */
                minTime = curTime;
            }
        }

        return minTime;
    }
    
    /*
     *  Synchronizes the runtime of the storyModeratorStart with
     *  the runtime of the storyModeratorEnd. This keeps both 
     *  objects executing within the same turn.
     */
    syncStoryModeratorRunTimes()
    {
        storyModeratorStart.nextRunTime = nextRunTime;
    }
}

/*
 *  Defines a Story Point. This object is activated when its
 *  checkActive() method returns true and is deactivated when 
 *  its checkCompleted() method returns true.
 *
 *  When activated this object will set the global current story 
 *  point value, establish any Actor States specific to this story
 *  point, and execute any other necessary code.
 *
 *  When deactivated this object will execute any oode required after 
 *  command and daemon/fuse execution, restore or reset any Actor States 
 *  specific to this story point, and set the global current story point
 *  value to nil and the previous story point value to this story point,
 *  and finally return a list of new candidate story points.
 */
class StoryPoint: object
{
    /*
     *  Indicates that this story point is active
     */
    active_             = nil

    /*
     *  Indicates that this story point has completed
     */
    completed_          = nil

    /*
     *  This is a count of the number of times this Story 
     *  object has been continued beyond the immediate turn
     *  that activated it.
     */
    contCount_          = 0

    /*
     *  This is the list of candidate story points from which 
     *  the storyModeratorStart can select the next time it executes.
     */
    nextStoryPointPool  = []

    /*
     *  A list of ActorMark instances belonging to this
     *  specific StoryPoint. ActorMarks will be activated
     *  when the StoryPoint is activated (during setActorStates())
     *  and deactivated when the StoryPoint is deactivated (during
     *  restoreActorStates()).
     */
    actorMarks          = []

    /*
     *  This method should be overridden to return true when the 
     *  desired game state/story goal has been reached that should 
     *  trigger the activation of this story point; otherwise the 
     *  method should return nil.
     */
    checkActive() { return nil; }

    /*
     *  This method should be overridden to return true when the 
     *  desired game state/story goal has been reached that should
     *  trigger the deactivation of this story point; otherwise the
     *  method should return nil.
     *
     *  By default we return the negation of checkActive.
     */
    checkCompleted() { return !checkActive(); }
    
    /*
     *  Activate the story point, which should set up conditions
     *  prior to the execution of player command and daemon/fuse 
     *  execution.
     */
    activateStoryPoint()
    {
        /*
         *  Mark the story point as active.
         */
        setStoryPointActive();

        /*
         *  Sets the global story point to the current value 
         *  that can be referenced in author code.
         */
        setGlobalCurrStoryPoint();

        /*
         *  Sets actor states specific to this story point
         */
        setActorStates();

        /*
         *  Executes any story code required by this story point
         *  prior to the execution of the player command.
         */
        doPreCmdCode();
    }

    /*
     *  Sets this story point as active and not yet completed.
     */
    setStoryPointActive() 
    { 
        active_     = true;
        completed_  = nil;
    }

    /*
     *  Sets the global current story point to this Story Point
     */
    setGlobalCurrStoryPoint() { gCurrStoryPoint = self; }

    /*
     *  This method can be overridden to set up any desired
     *  actor states related to this Story Point prior to command
     *  execution.
     *
     *  By default we iterate over each actor mark and 
     *  call its activateActorMark() method.
     */
    setActorStates()
    {
        foreach (local actorMark in actorMarks)
            actorMark.activateActorMark();
    }

    /*
     *  This method should be overridden to execute any desired
     *  code related to this Story Point prior to command execution.
     */
    doPreCmdCode() {}

    /*
     *  Continue the story point, which should handle any continuing
     *  story point processing both before and after command execution/ 
     *  daemon/fuse execution.
     */
    continueStoryPoint(preCmd)
    {
        /*
         *  Increment the continuation counter to indicate
         *  the number of times this object has been executed
         *  beyond the intial turn that activated it.
         */
        if (preCmd)
            ++contCount_;

        /*
         *  Executes any story code required by this continuing story point
         *  before or after the execution of the player command and daemons/fuses.
         */
        doContCmdCode(preCmd);

        /*
         *  Sets up actor states that have been modified by
         *  this continuing story point.
         */
        continueActorStates(preCmd);
    }

    /*
     *  This method should be overridden to execute any desired
     *  code related to this continuing Story Point before or after 
     *  command & daemon/fuse execution as determined by the preCmd
     *  argument.
     */
    doContCmdCode(preCmd) {}

    /*
     *  This method can be overridden to set up any desired
     *  actor states related to this continuing Story Point before 
     *  or after command & daemon/fuse execution as determiend by the
     *  preCmd argument.
     *
     *  By default we iterate over each actor mark and 
     *  call its continueActorMark() method.
     */
    continueActorStates(preCmd)
    {
        foreach (local actorMark in actorMarks)
            actorMark.continueActorMark(preCmd);
    }

    /*
     *  Deactivate the story point, which should handle any 
     *  remaining story point processing after command execution
     *  and daemon/fuse execution and return a list of candidates
     *  for the next story point.
     */
    deactivateStoryPoint()
    {
        /*
         *  Executes any story code required by this story point
         *  after the execution of the player command and daemons/fuses.
         */
        doPostCmdCode();

        /*
         *  Restores actor states that have been modified by
         *  this story point.
         */
        restoreActorStates();

        /*
         *  Sets the global story point to nil.
         */
        restoreGlobalStoryPoint();

        /*
         *  Marks the story point as completed.
         */
        setStoryPointCompleted();

        /*
         *  Return the list of candidate story points 
         *  that should follow next from this story point.
         */
        return getNextStoryPointPool();
    }

    /*
     *  This method should be overridden to execute any desired
     *  code related to this Story Point after command & daemon/fuse 
     *  execution.
     */
    doPostCmdCode() {}

    /*
     *  This method can be overridden to set up any desired
     *  actor states related to this Story Point after command &
     *  daemon/fuse execution.
     *
     *  By default we iterate over each actor mark and 
     *  call its deactivateActorMark() method.
     */
    restoreActorStates()
    {
        foreach (local actorMark in actorMarks)
            actorMark.deactivateActorMark();
    }

    /*
     *  Sets the global current story point to nil and the
     *  global previous story point to this Story Point.
     */
    restoreGlobalStoryPoint()
    {
        gCurrStoryPoint = nil;
        gPrevStoryPoint = self;
    }

    /*
     *  Sets this story point as not active and completed.
     */
    setStoryPointCompleted() 
    { 
        active_     = nil;
        completed_  = true;
    }

    /*
     *  Returns the list of next Story Point candidates that
     *  should follow from this Story Point.
     */
    getNextStoryPointPool()
    {
        return nextStoryPointPool;
    }

    /*
     *  Adds the ActorMark instance to the StoryPoint's 
     *  actorMarks list
     */
    registerActorMark(mark)
    {
        actorMarks = actorMarks.append(mark);
    }

    /*
     *  Initialize this StoryPoint instance. By default we do nothing.
     */
    initialize() {}
}

/*
 *  ActorMark class objects control the activation and deactivation
 *  of a particular actor's state during a story point. The class 
 *  also allows for any "props" and "scene-setting" to be handled 
 *  for the actor during doActorMarkSetup()/doActorMarkWrap().
 */
class ActorMark: object
{
    /*
     *  The actor for whom the actor state is to
     *  be handled by this actor mark.
     */
    actor       = nil

    /*
     *  The initial state that is established for the 
     *  indicated actor of this ActorMark, when the 
     *  ActorMark is activated.
     */
    initState      = (self)

    /*
     *  A list of actor states that can become activated
     *  during continuing story point execution.
     */
    contStates      = []

    /*
     *  A candidates list of  states that can become activated
     *  during continuing story point execution.
     */
    tmpContStates   = []

    /*
     *  The final state for the actor upon deactivation
     *  of this ActorMark. If no next state is provided 
     *  then the previous state will be used.
     */
    finalState       = (previousState)

    /*
     *  The state of the actor prior to activation of
     *  this ActorMark.
     */
    previousState   = nil
 
    /*
     *  Activates this ActorMark, setting up the actor's state
     *  and performing any ActorMark activation code required
     *  for this actor.
     */
    activateActorMark()
    {
        doActorMarkInit();

        /*
         *  If we don't have an actor or valid initial actor
         *  state then we don't have anything to activate
         *  as far as the actor state is concerned.
         */
        if (actor && initState && initState.ofKind(ActorState))
        {
            previousState = actor.curState;
            actor.setCurState(initState.createInstance(actor));
        }
    }

    /*
     *  Perform any set-up code that might be requited
     *  by the actor when he's on this mark.
     */
    doActorMarkInit() {}

    /*
     *
     */
    continueActorMark(preCmd)
    {
        doActorMarkCont(preCmd);

        /*
         *  If we don't have an actor, initial actor state
         *  and continuous actor state list, then we don't 
         *  have anything to activate as far as the actor 
         *  state is concerned.
         */
        if (actor && initState && initState.ofKind(ActorState)
            && contStates.length())
        {
            /*
             *  Iterate through the temporary continuos actor state list
             *  and check to see if any are eligible for activation
             */
            foreach (local contState in tmpContStates)
            {
                if (dataType(contState) == TypeObject 
                    && contState.ofKind(ActorState)
                    && checkActorMarkCont(preCmd, contState))
                {
                    /*
                     *  Activate this state for the actor
                     */
                    actor.setCurState(contState.createInstance(actor));

                    /*
                     *  Remove the state from our temporary
                     *  continuous states list.
                     */
                    tmpContStates -= contState;
                }
            }
        }
    }

    /*
     *  Checks to see if we wish to activate any of the intermediate
     *  actor states during a continuous story point action.
     *  
     *  By default we won't activate any intermediary actor states.
     */
    checkActorMarkCont(preCmd, contState) { return nil; }

    /*
     *  Perform any continuing code that might be
     *  requuired by the actor when he's on this mark.
     */
    doActorMarkCont(preCmd) {}

    /*
     *  Deactivates this ActorMark by setting the actor's
     *  state to the next desired state (or the one previous
     *  to this ActorMark's activation), and performing any
     *  ActorMark deactivation code required for this actor.
     */
    deactivateActorMark()
    {
        /*
         *  If we don't have an actor, valid initial actor
         *  state, and valid final actor state, then we don't 
         *  have anything to activate as far as the actor state 
         *  is concerned.
         */
        if (actor && initState && initState.ofKind(ActorState)
            && finalState && finalState.ofKind(ActorState))
        {
            actor.setCurState(finalState.createInstance(actor));
        }

        doActorMarkFinal();
    }

    /*
     *  Perform any wrap-up code that might be required
     *  by the actor when he's leaving this mark.
     */
    doActorMarkFinal() {}

    /*
     *  Register this ActorMark instance with the StoryPoint
     *  that is its location.
     */
    initializeActorMark() 
    {
        /*
         *  If we don't have a location or our location
         *  isn't a StoryPoint then we do nothing.
         */
        if (location && location.ofKind(StoryPoint))
        {
            location.registerActorMark(self);
            tmpContStates = contStates;
        }
    }
}


/*
 *  Initilize Story Guide objects
 */
StoryGuidePreinit: PreinitObject
{
    execute()
    {
        /*
         *  Loop over each StoryPoint instance and initialize it.
         */
        for (local point = firstObj(StoryPoint); point != nil; 
            point = nextObj(point, StoryPoint))
                point.initialize();
        /*
         *  Loop over each ActorMark instance and initialize it.
         */
        for (local mark = firstObj(ActorMark); mark != nil; 
            mark = nextObj(mark, ActorMark))
                mark.initializeActorMark();
    }
}

/* ------------------------------------------------------------------------ */
/*
 *   The TADS 3 Story Guide Module ID.
 */
ModuleID
{
    name = 'TADS 3 Story Guide'
    byline = 'by Kevin L.\ Forchione'
    htmlByline = 'by <a href="mailto:kevin@lysseus.com">Kevin L.\ Forchione</a>'
    version = '0.9b'

    /*
     *   We use a listing order of 60 so that, if all of the other credits
     *   use the defaults, we appear after the TADS 3 Library's own credits
     *   (conventionally at listing order 50) and before any other extension
     *   credits (which inherit the default order 100), but so that
     *   there's room for extensions that want to appear before us, or
     *   after us but before any default-ordered extensions.  
     */
    listingOrder = 65
}