/*
** A Tenuous Hold
**
** Written by Stephen Granade for the TextFire project.
*/

#include <adv.t>
#include "bugs.t"
#include "plurals.t"
#include "consist.t"
#include "senses.t"
#include "funcs.t"

#pragma C+

die: function
{
    "\b*** You have lost ***\b";
    scoreRank();
    "\bYou may restore a saved game, start over, quit, or undo
        the current command.\n";
    while ( 1 )
    {
        local resp;

	"\nPlease enter RESTORE, RESTART, QUIT, or UNDO: >";
        resp = upper(input());
        if ( resp == 'RESTORE' )
	{
	    resp = askfile( 'File to restore' );
	    if ( resp == nil ) "Restore failed. ";
	    else if ( restore( resp )) "Restore failed. ";
	    else
	    {
	        Me.location.lookAround(true);
	        scoreStatus( global.wongames, global.numgames );
		abort;
	    }
	}
        else if ( resp == 'RESTART' )
	{
	    scoreStatus( 0, 0 );
            restart();
	}
	else if ( resp == 'QUIT' || resp == 'Q' )
        {
	    terminate();
            quit();
	    abort;
        }
	else if (resp == 'UNDO')
	{
	    if (undo())
	    {
		"(Undoing one command)\b";
		Me.location.lookAround(true);
	        scoreStatus(global.wongames, global.numgames);
		abort;
	    }
	    else
		"Sorry, no undo information is available. ";
	}
    }
}

/*
 *   The scoreRank() function displays how well the player is doing.
 *   This default definition doesn't do anything aside from displaying
 *   the current and maximum scores.  Some game designers like to
 *   provide a ranking that goes with various scores ("Novice Adventurer,"
 *   "Expert," and so forth); this is the place to do so if desired.
 *
 *   Note that "global.maxscore" defines the maximum number of points
 *   possible in the game; change the property in the "global" object
 *   if necessary.
 */
scoreRank: function
{
    local i;

    "You have not yet scored any points. ";
}

// Tenuous initialization
tenuousinit: initialization
    preinit_phase = {
         initSearch();
    }
;

replace gameinit: function
{
    // put introductory text here
    "\bAn endless river of steel and plastic; the red glow of taillights; a
        cacophany of growls and bleating honks.\b
        You look down at your hands, unsurprised to find that you have been
        massaging the steering wheel, molding and kneading it.
        The veins on the back of your hands stand out, pulsing
        with the uneven flow of your blood.\b
        God, you hate traffic.\n";

    version.sdesc;                // display the game's name and version number

    Me.location = startroom;                // move player to initial location
    startroom.lookAround(true);             // show player where he is
    startroom.isseen = true;                // note that we've seen the room
    setscore('Score: 0');
    randomize();
    notify(global, &startStopDaemon, 0);
    notify(global, &rainFuse, 2);
    notify(global, &event, 4 + RAND(3));
    notify(global, &meetExit, 25 + RAND(10));
    global.ssTimer = 5 + RAND(5);
    global.movingEventNums = shuffle(global.movingEventNums);
    global.stoppedEventNums = shuffle(global.stoppedEventNums);
}


/*
 *   The terminate() function is called just before the game ends.  It
 *   generally displays a good-bye message.  The default version does
 *   nothing.  Note that this function is called only when the game is
 *   about to exit, NOT after dying, before a restart, or anywhere else.
 */
terminate: function
{
"\bThank you for playing the demo version of \(A Tenuous Hold\), a TextFire
release.  On-line registration for TextFire products will be available at
http://www.textfire.com on or before June 30, 1998.  Thanks for your
patience!";
}

/*
 *   The pardon() function is called any time the player enters a blank
 *   line.  The function generally just prints a message ("Speak up" or
 *   some such).  This default version just says "I beg your pardon?"
 */
pardon: function
{
    "Say what? ";
}

/*
 *   The numObj object is used to convey a number to the game whenever
 *   the player uses a number in his command.  For example, "turn dial
 *   to 621" results in an indirect object of numObj, with its "value"
 *   property set to 621.
 */
numObj: basicNumObj  // use default definition from adv.t
;

/*
 *   strObj works like numObj, but for strings.  So, a player command of
 *     type "hello" on the keyboard
 *   will result in a direct object of strObj, with its "value" property
 *   set to the string 'hello'.
 *
 *   Note that, because a string direct object is used in the save, restore,
 *   and script commands, this object must handle those commands.
 */
strObj: basicStrObj     // use default definition from adv.t
;

/*
 *   The "global" object is the dumping ground for any data items that
 *   don't fit very well into any other objects.  The properties of this
 *   object that are particularly important to the objects and functions
 *   are defined here; if you replace this object, but keep other parts
 *   of this file, be sure to include the properties defined here.
 *
 *   Note that awakeTime is set to zero; if you wish the player to start
 *   out tired, just move it up around the sleepTime value (which specifies
 *   the interval between sleeping).  The same goes for lastMealTime; move
 *   it up to around eatTime if you want the player to start out hungry.
 *   With both of these values, the player only starts getting warnings
 *   when the elapsed time (awakeTime, lastMealTime) reaches the interval
 *   (sleepTime, eatTime); the player isn't actually required to eat or
 *   sleep until several warnings have been issued.  Look at the eatDaemon
 *   and sleepDaemon functions for details of the timing.
 */
global: object
    turnsofar = 0                            // no turns have transpired so far
    score = 0                            // no points have been accumulated yet
    maxscore = 100                                    // maximum possible score
    verbose = true                             // we are currently in TERSE mode
    moving = true                        // Is the car moving?
    lane = 2                             // Freeway lane: |1|2|3|4|
    raining = nil
    nearBridge = nil
    passedBridge = nil
    nearExit = nil
    passedExit = nil
    nearMotorcycle = nil
    hitMotorcycle = nil
    heardJingle = nil
    nearCrash = nil
    stopMoving = {
        "\bIn front of you, taillights flash red. Everyone, including you,
            grinds to a halt. ";
        self.moving = nil;
    }
    startMoving = {
        "\bTraffic begins its slow, winding way along the freeway once again.
            You dutifully follow";
        if (car_door.open) {
            ", swinging shut your door as best you can";
            car_door.open = nil;
        }
        ". ";
        self.moving = true;
    }
    ssTimer = 0
    startStopDaemon = {
        if (nearMotorcycle || nearBridge || nearExit || nearCrash) return;
        self.ssTimer--;
        if (self.ssTimer == 0) {
            self.ssTimer = 5 + RAND(5);
            if (self.moving) self.stopMoving;
            else self.startMoving;
        }
    }
    rainFuse = {
        "\bA light rain begins, covering your windshield. Great. Just great. ";
        self.raining = true;
        notify(self, &checkWipers, 0);
    }
    wiperTurns = 0
    checkWipers = {
        if (!self.moving) return;
        if (wipers.location != startroom) {
            self.wiperTurns++;
            if (self.wiperTurns > 6) {
                "\bYou don't see the taillights of the semi stopping in front
                    of you until nearly too late. You wrench the steering wheel
                    left, avoiding the semi, sailing across lanes and towards
                    the median. ";
                Me.death;
            }
            else if (self.wiperTurns == 5)
                "\bYou blink as you realize you can no longer see the car in
                    front of you through the water on your windshield. ";
            else if (self.wiperTurns == 2)
                "\bThe rain accumulating on your windshield is making it
                    difficult to see. ";
        }
        else self.wiperTurns = 0;
    }
    passBridge = {
        "\bThe overpass moves over you, momentarily sheltering you from the
            rain, then is behind you and quickly receeding. ";
        abandoned_car.moveInto(nil);
        car_plate.moveInto(nil);
        bridge.moveInto(nil);
        self.nearBridge = nil;
        self.passedBridge = true;
    }
    passCycle = {
        if (self.nearMotorcycle == nil) return;
        "\bThe motorcycle zips past, weaving between the stopped cars. ";
        motorcycle.moveInto(nil);
        self.nearMotorcycle = nil;
    }
    meetExit = {
        if (self.nearBridge) {
            notify(self, &meetExit, 1);
            return;
        }
        if (!self.moving) {
            notify(self, &meetExit, self.ssTimer + 1);
            return;
        }
        "\bAhead of you and to the right, you see an exit ramp. You might be
            able to take it and avoid some of the traffic, as bad as it's been
            today. ";
        exit_ramp.moveInto(startroom);
        self.nearExit = true;
        notify(self, &passExit, 2);
    }
    passExit = {
        "\bYou watch with some longing as you pass the exit. ";
        exit_ramp.moveInto(nil);
        self.nearExit = nil;
        self.passedExit = true;
        self.nearCrash = true;
        notify(self, &crash, 3);
    }
    crash = {
        "\bYou don't see the semi edging over until it is too late. The rear
            end of the semi negligently swats you aside. Your car caromes off
            another, hits a slick patch of road, and begins a slow-motion
            tumble. ";
        Me.death;
    }
    // A list of events which are shuffled at start
    movingEventNums = [1, 2, 3, 4, 5, 6, 7]
    stoppedEventNums = [1, 2, 3, 4, 5, 6]
    movingCounter = 1
    stoppedCounter = 1
    event = {
        if (self.nearExit || self.nearCrash) return;
        if (self.moving)
            self.movingEvent(movingEventNums[movingCounter++]);
        else self.stoppedEvent(stoppedEventNums[stoppedCounter++]);
        notify(self, &event, 4 + RAND(3));
    }
    movingEvent(i) = {
        "\b";
        switch (i) {
            case 1:
                "A <<RAND(100) < 50 ? "car" : "truck">> suddenly cuts in
                    front of you, sending you swerving. You curse the driver
                    as the vehicle zips away. ";
                break;
            case 2:
                "You edge too close to the car in front of you and are
                    temporarily blinded by its spray. ";
                break;
            case 3:
                "With a careless lob, someone in the semi in front of you
                    throws a <<RAND(100) < 33 ? "cigarette" : "soda can">> out
                    the window. You watch it arc up and behind you. ";
                break;
            case 4:
                "You find yourself having to merge with another lane, slowing
                    you tremendously. Your heart begins beating so hard that
                    your vision ripples with every rush of blood through your
                    eyes. ";
                break;
            case 5:
                "A newspaper flutters from under the wheel of the truck ahead
                    of you and onto your windshield. ";
                if (wipers.location == startroom)
                    "It is eventually pushed off by your wipers. ";
                else "It slowly slides up the windshield and is gone. ";
                break;
            case 6:
                "Ahead of you, an overpass appears. Parked under it is an
                    abandoned car. ";
                abandoned_car.moveInto(startroom);
                car_plate.moveInto(startroom);
                bridge.moveInto(startroom);
                self.nearBridge = true;
                notify(self, &passBridge, 2);
                break;
            case 7:
                if (RAND(100) < 50) {
                    "There is a sharp \(BANG\) from the semi beside you.
                        You jerk, then realize that one of the semi's wheels
                        has exploded, sending shards of rubber onto the
                        car behind it. You speed up, leaving the limping semi
                        behind. ";
                }
                else {
                    "Behind you, you see a low-slung car racing towards you.
                        It weaves around you, then cuts back in front of you,
                        nearly taking your bumper as a souvenir. You hear its
                        Doppler-shifted horn as it sails past several other
                        cars before being hidden by a semi. ";
                }
                break;
        }
    }
    stoppedEvent(i) = {
        "\b";
        switch (i) {
            case 1:
                "Somewhere behind you, someone starts honking. The honking
                    spreads, a wave of sound which overtakes and passes you.
                    It takes a few moments to die down. ";
                break;
            case 2:
                "The car behind you edges forward, nearly touching your bumper.
                    In response you move ahead, causing the car in front of you
                    to roll forward a bit. You can see the effect spreading
                    in front of you like a pond ripple. ";
                break;
            case 3:
                "You sigh heavily at the winding trail of red taillights in
                    front of you. On the other side of the freeway, white
                    headlights pour past, unhindered by traffic. ";
                break;
            case 4:
                "Traffic moves forward a bit, giving you the opportunity to
                    ease into and out of a gigantic pothole. ";
                break;
            case 5:
                "To pass the time, you idly think about that government safety
                    report you saw some time back, the one about what various
                    cars did to their occupants' legs in a collision. As you
                    recall, your car was noted for shredding legs and pulping
                    pelvises. ";
                break;
            case 6:
                "Behind you, you see a motorcycle weaving between stopped cars.
                    It looks like the Harley will pass next to you in a few
                    moments. ";
                motorcycle.moveInto(startroom);
                self.nearMotorcycle = true;
                notify(self, &passCycle, 2);
                break;
        }
    }
;

/*
 *   The "version" object defines, via its "sdesc" property, the name and
 *   version number of the game.  Change this to a suitable name for your
 *   game.
 */
version: object
    sdesc = "\b\(A Tenuous Hold\)\n
\nDEMO version for the TextFire 12-PACK
\n(Type \"about\" for more information)
\nCopyright (c) 1998 by TextFire, Inc. All rights reserved.
\nDeveloped with TADS, the Text Adventure Development System.\b"
;

/*
 *   "Me" is the player's actor.  Pick up the default definition, basicMe,
 *   from "adv.t".
 */
Me: basicMe
    ldesc = {
        if (self.location == blackness)
            "You can see nothing in the blackness. ";
        else if (self.location == darkness)
            "You had best stand up if you want to take stock of your
                injuries. ";
        else pass ldesc;
    }
    death = {
        unnotify(global, &checkWipers);
        unnotify(global, &startStopDaemon);
        unnotify(global, &event);
        unnotify(global, &meetExit);
        unnotify(global, &passExit);
        if (global.heardJingle)
            "You are dismayed to realize that the only thing going through
                your mind is the burger jingle. \"Try our shakes...try our
                fries...\" ";
        "\bThere is an anti-climactic crunch.\b";
        Me.travelTo(blackness);
    }
;

/*
 *   darkTravel() is called whenever the player attempts to move from a dark
 *   location into another dark location.  By default, it just says "You
 *   stumble around in the dark," but it could certainly cast the player into
 *   the jaws of a grue, whatever that is...
 */
darkTravel: function
{
    "You stumble around in the dark, and don't get anywhere. ";
}

/*
 *   goToSleep - carries out the task of falling asleep.  We just display
 *   a message to this effect.
 */
goToSleep: function
{
    "***\bYou wake up some time later, feeling refreshed. ";
}

replace scoreStatus: function(points, turns)
{
}

startroom: room
    sdesc = "Inside Your Car"
    ldesc = {
        "The inside of your car is an all-too-familiar sight, thanks in part
            to the hours you
            have spent travelling to and from work. Its cracked dash and
            malformed steering wheel glow dully in the evening light. ";
    }
    listendesc = "You can hear the pinging and wheezing of your car over the
        hum of traffic around you. "
    noexit = {
        if (seatbelt.worn)
            "Your seatbelt has you tied down. ";
        else if (car_door.open)
            "The thought of abandoning your car and walking the fifteen miles
                to your house is worse than enduring the traffic. ";
        else "Not in this traffic. ";
        return nil;
    }
    up = { "Not likely. "; return nil; }
    down = { "Not likely. "; return nil; }
    out = {
        if (exit_ramp.location == self)
            exit_ramp.doTake(Me);
        else self.noexit;
        return nil;
    }
;

fake_car: fixedItem
    noun = 'car'
    adjective = 'my'
    location = startroom
    sdesc = "your car"
    thedesc = "your car"
    adesc = "your car"
    ldesc = "No matter how much you try, you are unable to convince yourself
        that the several shades of primer which cover it are
        attractive. But it was cheap. And it runs--most of the time. The
        asthmatic wheeze it has lately developed is worrying, though. "
    dobjGen(a, v, i, p) = {
        if (!v.issysverb && v != inspectVerb && v != askVerb &&
            v != tellVerb) {
            "Don't worry about your car as a whole; just use what's
                inside the cab with you. ";
            exit;
        }
    }
    iobjGen(a, v, d, p) = { self.dobjGen(a, v, d, p); }

;

freeway: distantItem
    noun = 'freeway' 'highway' 'interstate' 'road'
    adjective = 'gray' 'grey'
    location = startroom
    sdesc = "freeway"
    ldesc = {
        "The freeway is a ribbon of gray ";
        if (global.moving) "gliding under you. ";
        else "holding the snarl of traffic. ";
    }
;

lanes: distantItem
    noun = 'lane' 'lanes' 'dash' 'dashes'
    adjective = 'white'
    location = startroom
    sdesc = "lanes"
    ldesc = "The freeway is made of four lanes, delineated by white dashes. "
    verDoChange(actor) = {
        if (global.moving)
            "You can change lanes by turning the steering wheel. ";
        else "Traffic all around you is stopped, filling the available
            lanes. ";
    }
;

traffic: distantItem
    noun = 'traffic' 'cars'
    location = startroom
    sdesc = "traffic"
    ldesc = {
        if (global.moving)
            "The traffic is flowing sluggishly along the freeway. ";
        else "A sea of cars surround you, nearly unmoving. ";
    }
;

light_rain: distantItem
    noun = 'rain'
    adjective = 'light'
    sdesc = "light rain"
    ldesc = "It slants down, coating the road, the car, everything. "
;

exit_ramp: distantItem
    noun = 'ramp' 'exit'
    adjective = 'exit'
    sdesc = "exit ramp"
    ldesc = "On your right, an exit ramp, curving upwards and away from the
        freeway. "
    verDoTake(actor) = {}
    doTake(actor) = {
        "The sudden decision sends you swerving ";
        if (global.lane != 4)
            "across lanes of traffic, ignoring the startled bleats of other
                cars, and ";
        "onto the exit ramp. Behind you the freeway slides out of view, hidden
            by a few trees.\b
        A switchback to the left appears, surprising you. Dreamlike, you
            feel the back end of your car begin to outrace the front. Your
            trunk overtakes you, sending you into a spiral that is cut short
            by a guardrail. ";
        Me.death;
    }
;

bridge: distantItem
    noun = 'bridge' 'overpass'
    sdesc = "overpass"
    ldesc = "The overpass, arching above and in front of you, shelters an
        abandoned car. "
;

abandoned_car: distantItem
    noun = 'car'
    adjective = 'abandoned'
    sdesc = "abandoned car"
    ldesc = "Its owner has left this car, apparently for good. Its driver door
        is open, its license plate askew. "
;

car_plate: distantItem
    noun = 'plate'
    adjective = 'license'
    sdesc = "license plate"
    ldesc = "It reads, 'D-FENS'. "
    verDoRead(actor) = {}
    doRead(actor) = { self.ldesc; }
;

motorcycle: distantItem
    noun = 'motorcycle' 'cycle' 'bike' 'motorbike' 'harley' 'rider' 'man'
    sdesc = "motorcycle"
    ldesc = "The motorcycle is drawing ever closer, headed between your
        door and the car next to you. "
;

windshield: fixeditem
    noun = 'windshield'
    location = startroom
    sdesc = "windshield"
    ldesc = {
        "The windshield curves in front of you";
        if (light_rain.location == startroom)
            ", rain sliding off of it";
        ". ";
        if (wipers.location == startroom)
            "Wipers scrape erratically across its surface. ";
    }
    verDoLookthru(actor) = {}
    doLookthru(actor) = { "Traffic surrounds you. "; }
;

seatbelt: fixeditem
    worn = true
    noun = 'seatbelt' 'belt'
    adjective = 'blue' 'baby' 'worn'
    location = startroom
    sdesc = "seatbelt"
    ldesc = {
        if (self.worn)
            "It tugs and pulls at you, pressing you into the seat upholstery. ";
        else "It dangles limply beside you. ";
    }
    verDoWear(actor) = {
        if (self.worn)
            "The seatbelt is already wound about you. ";
    }
    doWear(actor) = {
        "You latch the seatbelt around you. It immediately tenses up, pulling
            on you. ";
        self.worn = true;
    }
    doSynonym('Wear') = 'Close' 'Fasten'
    verDoUnwear(actor) = {
        if (!self.worn)
            "The seatbelt is already unfastened. ";
    }
    doUnwear(actor) = {
        "With a click, you release the seatbelt. It hits the parking brake,
            then slithers across your lap and into the floor. ";
        self.worn = nil;
    }
    doSynonym('Unwear') = 'Take' 'Open' 'Unfasten' 'Unplug' 'Detach' 'Undo'
    verDoPull(actor) = {
        if (self.worn)
            "You yank your seatbelt, almost strangling yourself. ";
        else "More of the seatbelt unwinds into the floor before
            listlessly winding back. ";
    }
;

dash: fixeditem
    noun = 'dash'
    adjective = 'car' 'cracked' 'plastic' 'worn' 'faded' 'warped'
    location = startroom
    sdesc = "car dash"
    ldesc = {
        "Warped and faded. Over the many years you've owned the car, the
            sun has drawn deep cracks in the dash. ";
        if (radio_wires.location == startroom)
            "Wires from the radio run over it and down into the depths of
                your car. ";
    }
    verDoAttack(actor) = {}
    doAttack(actor) = { "The cracks widen a bit as you pound the dash. "; }
;

cracks: decoration
    noun = 'cracks' 'crack'
    location = startroom
    sdesc = "cracks"
    ldesc = "There are numerous cracks in the dash, ranging from miniscule
        to ones as thick as your little finger. "
;

turn_signal: fixeditem
    noun = 'signal' 'control' 'indicator'
    adjective = 'turn'
    location = startroom
    sdesc = "turn control"
    ldesc = "The turn control fell out some time ago, and you haven't bothered
        to replace it yet. It's not like anyone else on this freeway takes
        the time to use theirs. "
;

steering_wheel: fixeditem
    noun = 'wheel'
    adjective = 'steering'
    location = startroom
    sdesc = "steering wheel"
    ldesc = "It was once round but, with perserverence and pressure, you have
        managed to deform it noticeably. Jutting from one side is the wiper
        control stick. "
    verDoAttack(actor) = {}
    doAttack(actor) = { "You pound on the steering wheel in frustration. "; }
    verDoTurn(actor) = {
        if (!global.moving)
            "You tug on the steering wheel, scraping your front tires against
                the freeway as they turn. ";
    }
    doTurn(actor) = {
        "You jerk the steering wheel back and forth, sending your car into
            spasms. A car somewhere behind you honks loudly. ";
    }
    verDoTurnCW(actor) = { self.verDoTurn(actor); }
    verDoTurnCCW(actor) = { self.verDoTurn(actor); }
    doTurnCW(actor) = {
        if (exit_ramp.location == startroom) {
            exit_ramp.doTake(actor);
            return;
        }
        if (global.lane == 4) {
            "You swerve right, coming perilously close to ";
            if (global.nearBridge) "the concrete abutments of the bridge";
            else "the edge of the road";
            " before drifting back to the center of the lane. ";
        }
        else {
            global.lane++;
            "A quick jerk of the wheel and you have changed lanes";
            if (RAND(100) < 20)
                ", nearly forcing another car off the road";
            ". ";
        }
    }
    doTurnCCW(actor) = {
        if (global.lane == 1) {
            "You swerve left, coming perilously close to ";
            if (global.nearBridge) "the concrete abutments of the bridge";
            else "the edge of the road";
            " before drifting back to the center of the lane. ";
        }
        else {
            global.lane--;
            "A quick jerk of the wheel and you have changed lanes";
            if (RAND(100) < 20)
                ", nearly forcing another car off the road";
            ". ";
        }
    }
;

car_horn: fixedItem
    noun = 'horn'
    adjective = 'car'
    location = startroom
    sdesc = "car horn"
    ldesc = "The horn is built into the steering wheel. "
    verDoHonk(actor) = {}
    doHonk(actor) = {
        "You blow your horn, which makes an anemic sound. ";
    }
    doSynonym('Honk') = 'Push'
;

car_door: fixeditem
    open = nil
    noun = 'door'
    adjective = 'car'
    location = startroom
    sdesc = "car door"
    ldesc = {
        if (global.hitMotorcycle) {
            "The door is in worse shape than it was prior to its encounter
                with the motorcycle. ";
            if (self.open)
                "It hangs at a forlorn angle, canted several degrees from
                    true. ";
            else "It is more or less closed, fitting poorly in the car
                frame. ";
        }
        else {
            if (self.open)
                "The door gapes open, displaying its rust spots for all the
                    world to see. ";
            else "The closed door rattles from time to time, vibrated by your
                engine. ";
        }
    }
    verDoOpen(actor) = {
        if (self.open) "You have already opened the door. ";
        if (global.moving)
            "Opening the door while moving through this traffic would be a
                quick way to get it hit. Your car
                could barely withstand a collision with a tricycle; what
                makes you think its door would fare any better? ";
    }
    doOpen(actor) = {
        self.open = true;
        "You pound on the door until it finally creaks open. ";
        if (!global.nearMotorcycle) return;
        "The thump of motorcycle meeting door comes several seconds later.
            The cyclist pinwheels over his handlebars, vanishing into traffic
            a few cars ahead of you. When everyone moves forward
            moments later, the cycle is left behind, a twisted wreck. ";
        global.nearMotorcycle = nil;
        global.hitMotorcycle = true;
        motorcycle.moveInto(nil);
    }
    verDoClose(actor) = {
        if (!self.open) "You have already closed the door. ";
    }
    doClose(actor) = {
        self.open = nil;
        "It takes some tugging for you to convince the door to close, but you
            finally have your way with it. ";
        if (global.hitMotorcycle)
            "It takes more work to get the door to stay closed; it is
                significantly more warped than it was to begin with. ";
    }
;

parking_brake: fixedItem
    noun = 'brake' 'lever'
    adjective = 'parking'
    location = startroom
    sdesc = "parking brake"
    ldesc = "\"The parking broke,\" as your friends used to call it. It hasn't
        worked since sometime during the Carter administration. "
    verDoPull(actor) = {}
    doPull(actor) = {
        "As you yank on the lever, it ratchets up to about a fourty-five
            degree angle. When you let go, it sinks back down to its usual
            position. ";
    }
;

brake_pedal: fixedItem
    noun = 'brake' 'pedal'
    adjective = 'brake'
    location = startroom
    sdesc = "brake pedal"
    ldesc = "Square and worn from use. "
    verDoPush(actor) = {}
    doPush(actor) = { decelVerb.action(actor); }
;

accel_pedal: fixedItem
    noun = 'accelerator' 'pedal'
    adjective = 'accelerator'
    location = startroom
    sdesc = "accelerator pedal"
    ldesc = "Some of it has flaked away, making it hard for you to maintain
        a constant speed. "
    verDoPush(actor) = {}
    doPush(actor) = { accelVerb.action(actor); }
;

wiper_control_stick: fixedItem
    on = nil
    noun = 'stick'
    adjective = 'wiper' 'control'
    location = startroom
    sdesc = "wiper control stick"
    ldesc = "The stick has two settings:\ off, and slightly faster than off.
        Pulling it will turn on your wipers; pushing it will turn them off. 
        Right now the stick is <<self.on ? "towards" :
        "away from">> you. "
    verDoPush(actor) = {
        if (!self.on)
            "The stick won't go any farther away from you. ";
    }
    doPush(actor) = {
        "Your windshield wipers shudder, then slide to the bottom of your
            windshield and out of sight. ";
        self.on = nil;
        wipers.moveInto(nil);
    }
    verDoPull(actor) = {
        if (self.on)
            "The stick won't move any farther towards you. ";
    }
    doPull(actor) = {
        "With a loud groan, your windshield wipers begin scraping across the
            windshield. ";
        self.on = true;
        wipers.moveInto(startroom);
    }
    doSynonym('Push') = 'Turnoff'
    doSynonym('Pull') = 'Turnon'
;

wipers: fixedItem
    noun = 'wipers'
    adjective = 'windshield'
    sdesc = "windshield wipers"
    ldesc = {
        "They scrape across the windshield, smearing bugs";
        if (global.raining)
            " and water";
        ". ";
    }
    doTurnoff->wiper_control_stick
    doTurnon->wiper_control_stick
    dobjGen(a, v, i, p) =
    {
        if (v != inspectVerb && v != turnOffVerb && v != turnOnVerb) {
            "You can't reach it through the glass. ";
            exit;
        }
    }
    iobjGen(a, v, d, p) = { self.dobjGen(a, v, d, p); }
;

radio_wires: fixedItem
    noun = 'wire' 'wires'
    location = startroom
    sdesc = "wires"
    ldesc = "The wires criss-cross the dash in their journey from the radio
        to somewhere underneath. "
    verDoPull(actor) = {}
    doPull(actor) = {
        "You yank on the wires, pulling them free from the radio. You drop
            them, where they are lost beneath your seat. ";
        self.moveInto(nil);
    }
    doSynonym('Pull') = 'Take'
;

radio: fixedItem
    on = nil
    noun = 'radio'
    location = startroom
    sdesc = "radio"
    ldesc = {
        "Another victim of time and entropy. Most of its controls are missing;
            what controls still remain are worn and cracked. ";
        if (self.on)
            "Despite you having turned it on, it remains silent. ";
    }
    listendesc = "It is stubbornly silent. "
    verDoTurnon(actor) = {
        if (self.on)
            "You fiddle with the radio some, verifying that you have turned
                it on. Despite its silence, you have. ";
    }
    doTurnon(actor) = {
        self.on = true;
        "There is a click, followed by the hiss of static. You fiddle with
            the radio, attempting to tune in a station. ";
        if (!global.heardJingle) {
            "The best you are able to find before the radio dies completely
                is the catchy jingle of a local burger place. ";
            global.heardJingle = true;
        }
        else "For a moment you hear a tinny snatch of music, but the radio
            falls silent, not even giving out static. ";
    }
    verDoTurnoff(actor) = {
        if (!self.on)
            "You have already turned off the radio. ";
    }
    doTurnoff(actor) = {
        self.on = nil;
        "You turn off the radio, not that it matters. ";
    }
    verDoAttack(actor) = {}
    doAttack(actor) = {
        if (self.on)
            "The radio lets out a squeal when you whack it with the palm of
                your hand, but remains mute thereafter. ";
        else "Satisfying, but nothing happens. ";
    }
;

blackness: room
    hintCounter = 0
    sdesc = "Blackness"
    ldesc = "You can feel nothing, smell nothing, hear nothing. "
    roomAction(a, v, d, p, i) = {
        if (v == sleepVerb) {
            "For all you can tell, you're already asleep. ";
            exit;
        }
        else if (!v.issysverb && v != openVerb && v != inspectVerb) {
            "Nothing happens. ";
            self.hintCounter++;
            if (self.hintCounter == 5)
                "\bThere is a sudden flash of light, allowing you to see a
                    pattern of veins, as if you were looking at
                    your...eyelids? Then the light is gone. ";
            exit;
        }
    }
;

eyes: fixedItem
    noun = 'eye' 'eyes' 'eyelid' 'eyelids'
    location = blackness
    sdesc = "eyes"
    ldesc = "You become conscious of your closed eyes. "
    verDoOpen(actor) = {}
    doOpen(actor) = {
        "You slowly open your eyes, afraid of what you might see.\b";
        actor.travelTo(darkness);
    }
;

here: room
;

darkness: room
    isListed = nil
    sdesc = "Darkness"
    ldesc = "You are lying on your back, staring up at the cloudy sky. Some
        time must have passed:\ night has fallen. "
    location = here
    doUnboard(actor) = {
        "As you stand, you feel pain scamper down the length of your body, an
            unpleasant pulling, tearing sensation.
            Then, you are standing, the pain gone. You glance around at
            your mangled car, the bent guardrail, your broken body.\b
            Your body. Your body is lying at your feet, as if you had stepped
            from it. A quick glance at your legs shows that you can see
            through them to the ground beyond.\b
            You may have just found something you hate more than traffic.\b";
        terminate();
        quit();
        abort;
    }
;

night_sky: decoration
    noun = 'sky'
    adjective = 'cloudy'
    location = darkness
    sdesc = "cloudy sky"
    ldesc = "The sky above you is faintly lit by the nearby city. "
;

modify distantItem
    dobjGen(a, v, i, p) = {
        if (!v.issysverb && v != inspectVerb && v != askVerb &&
            v != tellVerb) {
            "You can't reach <<self.isThem ? "those" : "that">> from the car. ";
            exit;
        }
    }
    iobjGen(a, v, d, p) = { self.dobjGen(a, v, d, p); }
;

modify pullVerb
    verb = 'set'
;

modify attackVerb
    doAction = 'Attack'
;

modify detachVerb
    verb = 'unlatch'
;

changeVerb: deepverb
    sdesc = "change"
    verb = 'change'
    doAction = 'Change'
;

driveVerb: deepverb
    sdesc = "drive"
    verb = 'drive'
    action(actor) = {
        if (global.moving)
            "That would be what you are, in general, doing. ";
        else "Be patient. ";
    }
;

honkVerb: deepverb
    sdesc = "honk"
    verb = 'honk' 'blow'
    action(actor) = { car_horn.doHonk(actor); }
    doAction = 'Honk'
;

accelVerb: deepverb
    sdesc = "accelerate"
    verb = 'accelerate' 'speed up'
    action(actor) = {
        if (global.moving)
            "You can't speed up:\ the traffic has you effectively boxed in. ";
        else "Would that you could speed up, but the back end of the car in
            front of you has you trapped. ";
    }
;

decelVerb: deepverb
    sdesc = "decelerate"
    verb = 'decelerate' 'brake' 'stop' 'slow down'
    action(actor) = {
        if (global.moving)
            "You tap your brakes, causing the car behind you to swerve around.
                The driver yells at you as <<RAND(100) < 70 ? "he" :
                "she">> passes. ";
        else "Seeing as you're not moving right now, it's a moot point. ";
    }
;

turnclockwiseVerb: deepverb
    touch = true
    sdesc = "turn clockwise"
    verb = 'turn clockwise' 'turn cw' 'turn right'
    doAction = 'TurnCW'
;

turnccwVerb: deepverb
    touch = true
    sdesc = "turn counterclockwise"
    verb = 'turn counterclockwise' 'turn ccw' 'turn anticlockwise' 'turn left'
    doAction = 'TurnCCW'
;

cwPrep: Prep
    sdesc = "clockwise"
    preposition = 'clockwise' 'cw'
;

ccwPrep: Prep
    sdesc = "counterclockwise"
    preposition = 'counterclockwise' 'anticlockwise' 'ccw'
;

rightPrep: Prep
    sdesc = "right"
    preposition = 'right'
;

leftPrep: Prep
    sdesc = "left"
    preposition = 'left'
;

modify thing
    verDoAttack(actor) = { "There is no need to hit <<self.thedesc>>. "; }
    verDoUndo(actor) = { "There is no need. "; }
    verDoChange(actor) = { "There is no need. "; }
;

aboutVerb : sysverb
    verb = 'info' 'about'
    sdesc = "about"
    action(actor) = {
        "\bThe demo version of \(A Tenuous Hold\) was written by Riley Hawkins
        for TextFire, Inc.
        \b\(About the author:\)
\bAfter receiving a degree in library science from Texas Woman's
University, Riley Hawkins was directionless. Eventually, like so many
other Okies, Riley went west in search of opportunity, finally stopping in
Piedmont, CA. Unfortunately, Piedmont has a noticeable dearth of
libraries; fortunately, one of the numerous branches of the Oakland
Public Library system had a job opening.

\bA few months later, Riley met Shea Davis when Shea was doing research for
a game. In an odd series of coincidences, Deborah Keyes attended the same
PFLAG meeting Riley did, and Riley and Flip Winkler crossed paths at a
show at Slim's. Though still working at the library, Riley is now spending
more and more time writing interactive fiction.
        \b\(About TextFire:\)
        \bTextFire, Inc.\ is a partnership of authors formed for the
purpose of providing quality works of interactive fiction to a worldwide
community of players, hobbyists, and enthusiasts.  Help support our efforts
to keep interactive fiction alive! ";
        abort;
    }
;

hintVerb : sysverb
    verb = 'hint' 'hints' 'help'
    sdesc = "help"
    action(actor) = {
        "Hints are available with the full registered version of
\(A Tenuous Hold\).  Type REGISTER for additional information.";
        abort;
    }
;

registerVerb : sysverb
    verb = 'register' 'registration'
    sdesc = "register"
    action(actor) = {
        "On-line registration for TextFire products will be available at
http://www.textfire.com on or before June 30, 1998.  Thanks for your
patience!";
        abort;
    }
;

modify quitVerb
    verb = 'q'
;

modify undoVerb
    doAction = 'Undo'
;

modify sleepVerb
    action(actor) = {
        if (actor.location == blackness)
            "Despite the blackness, you cannot sleep. ";
        else if (actor.location == darkness)
            "You find that thoughts of the crash fill your head, driving
                away sleep. ";
        else "You find yourself unable to sleep. ";
    }
;

modify creditsVerb
    action(actor) = { aboutVerb.action(actor); }
;

modify lookThruVerb
    verb = 'look out'
;
