/* ex:set ts=4 sw=4:
 *
 * darkness.t: An different kind of darkness
 *
 * This module provides a slightly more complicated set of methods for
 * supporting darkness in TADS.  It should be compatible with the original
 * TADS methods except that it disables the original darkroom.roomCheck.
 *
 * This was originally implemented so that my port of Dungeon would allow
 * the player to turn the lamp on in the dark.  After I implemented it, I
 * discovered that the original did not allow this.  However, I liked the
 * idea and left it in.  It also provided the ability to produce the
 * specialised error messages that Dungeon used ("Its impossible to read in
 * the dark") rather than the generic ("%You% can't see a thing.")
 *
 * Replaces:
 *	darkroom.roomAction
 *		This method has been replaced with one which interrogates each
 *		verb rather than assuming that no deepverbs work in the dark.  The
 *		verb.validDark method returns a number (-1, 0, +1).
 *		If the method returns '0', the verb is allowed to continue.
 *		If it returns '+1', roomAction will print a default message and
 *		exit verb processing.
 *		If it returns '-1', roomAction assumes that the verb issued a
 *		message and just exits verb processing.
 *
 *	darkroom.roomCheck
 *		This method was originally used to handle verbs in the dark but has
 *		been disabled since it did not allow verb-specific testing without
 * 		messing around considerably.
 *
 * Adds:
 *	deepverb.validDark
 *		This method determines what happens if the verb is attempted in the
 *		dark.  See description of darkroom.roomAction for possible values.
 *		The default is that *no* deepverbs work in the dark.
 *
 *	sysverb.validDark
 *		We override the default validDark method to ensure that *all*
 *		sysverbs (save, quit, etc) work in the dark.  If we were nasty, we
 *		could forbid 'save' by overriding saveVerb.validDark
 *
 *	travelVerb.validDark
 *		We override the default validDark method to ensure that *all*
 *		travel verbs (north, sw, up, etc) work in the dark.
 *
 *	darkroom.visibleObjects(actor)
 *		This method returns all objects which are visible in the current
 *		room unless the lights are out in which case it returns nil
 *
 *	darkroom.takeableObjects(actor)
 *		This method returns all objects which are takeable in the current
 *		room unless the lights are out in which case it returns nil
 *
 * Note that this now requires 'wallpaper.t' as thats a part of my standard
 * frame-work these days.  I have no idea whether it will be hard to integrate
 * into the standard TADS libraries otherwise.  'wallpaper.t' should be
 * available from the same place as 'darkness.t'
 *
 * This module is Copyright (c) 1993 Jeff Laing.  Permission to use any or all
 * of this code in other TADS games is granted provided credit is given in
 * your "CREDITS" command for this code and that you leave this copyright
 * message in the source whenever distributed.  Please make all changes in
 * an backward compatible manner (where possible) and make them available
 * for free (like I did ;-)
 */
darknessVersion : versionTag, initialization
	id="$Id: darkness.t_v 1.9 1994/05/01 02:10:29 jeffl Exp jeffl $\n"
	author='Jeff Laing'
	func='darkness support'

	/*
	 * called from preinit()
	 */
	preinit_phase={
		local o;

		global.lamplist := [];
		o := firstobj(lightsource);
		while( o <> nil )
		{
			global.lamplist := global.lamplist + o;
			o := nextobj( o, lightsource );
		}
	}

	/*
	 * called from init()
	 */
	//init_phase={
	//}
;

// its important that lamplist default to an empty list so that other
// preinit tests will be able to compute islit properly
modify global
	lamplist = []
;

/*
 *	Darkrooms ask the verb being attempted whether it can work in the dark.
 *	This was necessary to allow the player to 'turn the lamp on' in the dark.
 */
modify class darkroom

	// room's get a change to process actions - this is ours
    replace roomAction( actor, v, dobj, prep, io ) =
    {
    	// if room is bright, just pass it on
		if ( self.islit ) pass roomAction;
		
		// its dark - check with the verb if its ok
		switch( v.validDark( actor, dobj, prep, io ) ) {

			// ok
			case 0:		pass roomAction;

			// +1 means issue a standard error message
			case +1:	"%You% can't see a thing. ";

			// -1 means verb issued the message for us
			case -1:	exit;
		}
    }

	// we disable Mike's original check because we handle it so much better 8-)
    replace roomCheck( v ) = { return( true ); }

	// called by object validation mechanism.
	visibleObjects(actor) = {
		if (not self.islit) return [];		// its dark, "all" = nothing
		pass visibleObjects;				// its light, do it normally
	}

	// called by object validation mechanism.
	takeableObjects(actor) = {
		if (not self.islit) return [];		// its dark, "all" = nothing
		pass takeableObjects;				// its light, do it normally
	}
;

// each verb must know whether it works in the dark.  by default, they all fail
// with the default message
modify class deepverb
	// called if this room is dark.  1 means it should just print the
	// default "Its dark.".  -1 means method printed message already.
	// 0 means verb will work.
	validDark( actor, obj, prep, iobj ) = 1
;

// each verb knows whether it works in the dark.  sysverbs must work!
modify class sysverb
	// called if this room is dark.  1 means it should just print the
	// default "Its dark.".  -1 means method printed message already.
	// 0 means verb will work.
	validDark( actor, obj, prep, iobj ) = 0
;

// each verb knows whether it works in the dark.  travelverbs must work.
modify class travelVerb
	// called if this room is dark.  1 means it should just print the
	// default "Its dark.".  -1 means method printed message already.
	// 0 means verb will work.
	validDark( actor, obj, prep, iobj ) = 0
;

/* These are a couple of examples of how to override specific verbs */

// examine "yellow gooey thing I am holding" should have a better message
// than "You can't see a thing." in a dark room
modify inspectVerb
	// called if this room is dark.  1 means it should just say
	// "Its dark.".  -1 means method printed message already.
	// 0 means verb will work.
	validDark( a, o, p, i ) = {
		if (o.isIn(a)) return 0;		// valid if holding it
		"Its too dark to examine <<o.thedesc>>";
		return -1;						// otherwise too dark to see
	}
;

// user should be able to turn on a lamp that they are holding even in a
// dark room.
modify turnOnVerb
	// called if this room is dark.  1 means it should just say
	// "Its dark.".  -1 means method printed message already.
	// 0 means verb will work.
	validDark( a, o, p, i ) = {
		if (o.isIn(a)) return 0;		// can't see / not holding it
		"Its too dark to turn on <<o.thedesc>>";
		return -1;						// otherwise too dark to see
	}
;

// reading in the dark is impossible but a specialized error message is nicer
// than the generic one.
modify readVerb
	// called if this room is dark.  1 means it should just say
	// "Its dark.".  -1 means method printed message already.
	// 0 means verb will work.
	validDark( actor, obj, prep, iobj ) = {
		"It is impossible to read in the dark. ";
		return -1;
	}
;
