/* Copyright (c) 1999, 2000 by Kevin Forchione.  All Rights Reserved. */
/*
 *  TADS ADV.T/STD.T LIBRARY EXTENSION
 *  OBSERVER.T				
 *  version 1.0
 *
 *
 *		This file defines the observer class, which allows the player to
 *		communicate with actors that are not in the parserGetMe().location. The
 *		observer acts as a proxy for the parserGetMe() object, allowing the
 *		player to reference "me" and "myself" as direct and indirect objects in
 *		commands to unpresent actors.
 *
 *----------------------------------------------------------------------
 *  REQUIREMENTS
 *
 *      + HTML TADS 2.2.6 or later
 *      + Should be #included after ADV.T and STD.T 
 *
 *----------------------------------------------------------------------
 *  IMPORTANT LIBRARY INTERFACE AND MODIFICATION
 *
 *      This file modifies thing validActor() method and deepverb
 *      validDoList()
 *
 *----------------------------------------------------------------------
 *  COPYRIGHT NOTICE
 *
 *  	You may modify and use this file in any way you want, provided that
 *		if you redistribute modified copies of this file in source form, the
 *   	copies must include the original copyright notice (including this
 *   	paragraph), and must be clearly marked as modified from the original
 *   	version.
 *
 *------------------------------------------------------------------------------
 *  REVISION HISTORY
 *
 *		99-03-21:	Creation.
 */
 
#define __OBSERVER_MODULE_


// Predefining the functions used in this file.
setobserverActive: function;
parent: function;

/*
 *  observer: Actor
 *
 *  This is a special object that can "shadow" the movements of a
 *  character as it moves from room to room.  The purpose of an observer
 *  is to allow the player to communicate with an actor as it travels from room
 *	to room.  Each actor that is to be observed must
 *  have its own observer object.  The observer object should
 *  define all of the same vocabulary words (nouns and adjectives) as the
 *  parserGetMe() object to which it refers.  The observer must also
 *  define the myactor property to be the Actor object that
 *  the observer shadows.  The observer will always be in the room with
 *	the character it observes; no commands are effective
 *  with an observer except for "tell" "ask" and "show".
 */
class observer: Actor, floatingItem
	isobserver = true
    sdesc = { parserGetMe().sdesc; }
    adesc = { parserGetMe().adesc; }
    thedesc = { parserGetMe().thedesc; }
    ldesc = { caps(); parserGetMe().thedesc; " <<self.myactor.isdesc>> not here. "; }
    actorDesc = {}
    noun = 'me' 'myself'
    isactive = nil
    myactor = nil      			// set to the Actor to be observed
    /*
     *	We only return a location if the observer is active and the player is
     *	not in the same location as the actor being addressed. By this time we
     *	know who the actor is and we have passed through the deepverb, setting
     *	the observer.isactive to true only for the actor being addressed and 
     *	only if it met the observer.checkactive method's conditions.
     */
    location =
    {
    	if ( parserGetMe().location = self.myactor.location )
    		return nil;
   		else if ( parent( parserGetMe() ) = parent( self.myactor ) )
    		return nil;
    	else if ( self.isactive )
    			return self.myactor.location;
    		else
    			return nil;
    }
    /*
     *	This method is customisable by the author and should contain
     *	conditionals to indicate when or where the observer should be 
     *	active (i.e. the observer.location has an open vent through 
     *	which the actor and player can communicate.
     */
    checkactive = { return true; }
    actorAction(v, d, p, i) = {}
    dobjGen(a, v, i, p) =
    {
    	"dobjGen ";
    	switch( v )
    	{
    		case askVerb:
    		case tellVerb:
    		case showVerb:
    			break;
	    	default:
    			"<<a.sdesc>> can't reach <<self.sdesc>> from here. ";
    			exit;
    	}
    }
    iobjGen(a, v, d, p) =
    {
    	"iobjGen ";
    	switch( v )
    	{
    		case askVerb:
    		case tellVerb:
    			break;
    		case showVerb:
    			if ( parserGetMe().isVisible( self ) )
    				break;
    			else
    				{
    					"<<a.sdesc>> doesn't see <<self.sdesc>> here. ";
    					exit;
    				}
    		default:
    			"<<a.sdesc>> can't reach <<self.sdesc>> from here. ";
    			exit;
    	}
    }
    // these properties are for the format strings
    fmtYou = "you"
    fmtYour = "your"
    fmtYoure = "you're"
    fmtYoum = "you"
    fmtYouve = "you've"
    fmtS = ""
    fmtEs = ""
    fmtHave = "have"
    fmtDo = "do"
    fmtAre = "are"
    fmtMe = "me"
;

/*
 *	We modify the validActor for Actor to return true if the actor has an
 *	observer and the observer would be ( but isn't yet ) active for this 
 *	situation. At this stage of the game we don't know the actor, we're only
 *	creating a list of valid actors. This modification allows us to give other
 *	actors commands when they are not in the player's location.
 */
modify Actor
	validActor =
	{
		if ( self.myobserver and self.myobserver.checkactive )
		{
			return true;
		}
		else
			pass validActor;
	}
;

/*
 *	We modify the deepverb validDoList() to set the observer.isactive. At this
 *	point we know the actor, and are determining the valid direct and indirect
 *	objects for the command. Since validDoList() returns all of the objects in
 *	global.floatingList we would have trouble with disambiguation if there are
 *	more than one observer in a room. We solve this problem by setting the 
 *	observer.isactive := true for the actor being addressed. Then in the
 *	observer.location method we only return the active observer, all others
 *	return a nil location.
 */
modify deepverb
    validDoList(actor, prep, iobj) =
    {
        local ret;
        local loc;
 		
        setobserverActive( actor );
        
        loc := actor.location;
        while (loc.location)
            loc := loc.location;
        ret := visibleList(actor, actor) + visibleList(loc, actor)
               + global.floatingList;
        return ret;
    }
;

/*
 *	setobserverActive: function( actor )
 *	
 *	This function is called from deepverb.validDoList(). It sets the
 *	observer.isactive for the actor when the observer is being addressed as
 *	as a dobj or iobj, as in, ">Bob, tell me about the book." 
 *
 *	The function checks to see if the objects in global.floatingList are 
 *	observers, then determines whether the observer.myactor is the actor 
 *	being addressed and if so whether the observer should be active in this
 *	instance.
 */
setobserverActive: function( actor ) 
{
	local o, f;
        
	f := global.floatingList;
    o := car( f );
    while( o )
    {
    	if ( o.isobserver )
        {
        	if ( o.myactor = actor and o.checkactive )
        		o.isactive := true;
        	else
        		o.isactive := nil;
        }
        f := cdr( f );
        o := car( f );
    }
}

/*
 *	parent: function( obj )
 *
 *	Returns the highest parent of an object. 
 */
parent: function( obj ) 
{
	local p, l;

	p := obj;
	l := p.location;
	while( l )
	{
		p := l;
		l := p.location;
	};
	return p;
}
