////////////////////////////////////////////////////////////////////////
//  
//  ALT Library File: Stack 010123
//
//  Copyright (c) 2000, 2001 Kevin Forchione. All rights reserved.
//  Based on ADV.T (c) and STD.T (c) Michael Roberts.
//
//  This file is part of the ALT replacement library for ADV.T and 
//  STD.T and requires TADS 2.5.1 or later.
//
////////////////////////////////////////////////////////////////////////

#ifndef _STACK_H_
#define _STACK_H_

#include <vector.h>

#pragma C+

/*
 *  Stack: object
 *
 *  This implements a FIFO (First-In-First-Out) or LIFO
 *  (Last-In-First-Out) stack that can be used for pushing and popping.
 *  The Stack will automatically create a Vector class object, which it
 *  uses for keeping track of pushed objects.
 *
 *  To use a stack simply create it dynamically. The Stack default is
 *  LIFO. If you wish it to behave as a FIFO stack simply set its isLIFO
 *  attribute to nil.
 *
 *      local stack = new Stack;
 *      stack.isLIFO = nil;
 *
 *  You can push items onto the stack with the stack's push method:
 *
 *      stack.push(item);
 *
 *  and pop items off the stack using its pop method:
 *
 *      item = stack.pop;
 *
 *  When the stack is no longer needed you can delete it with the delete
 *  statement. The stack will automatically clean-up the Vector class
 *  object that it dynamically created.
 */
class Stack: object
    isLIFO = true
    vector = nil    // pointer to dynamically-created Vector class obj
    /*
     *  Push an item onto the stack. All items are pushed to the top of
     *  the stack.
     */
    push(item) = {
        self.vector.addElement(item);
        return item;
    }
    /*
     *  Push an item to the bottom of the stack.
     */
    pushB(item) = {
        self.vector.addElementToHead(item);
        return item;
    }
    /*
     *  Pop an item off the stack. If stack.isLIFO then it pops the
     *  item from the top of the stack; otherwise it pops the item from
     *  the bottom of the stack.
     */
    pop = {
        local item;
        if (self.isLIFO) {
            item = self.vector.elementAt(self.vector.getSize);
            self.vector.removeElementAt(self.vector.getSize);
        } else {
            item = self.vector.elementAt(1);
            self.vector.removeElementAt(1);
        }
        return item;
    }
    /*
     *  Returns a boolean indicating whether the stack is empty or not.
     */
    isEmpty = {
        return self.vector.isEmpty;
    }
    /*
     *  Returns the size of the stack.
     */
    getSize = {
        return self.vector.getSize;
    }
    construct = {self.vector = new Vector;}
    destruct = {delete self.vector;}
;

#pragma C-

#endif /* _STACK_H_ */
