////////////////////////////////////////////////////////////////////////
//  
//  ALT Library File: SysVerb 010123
//
//  Copyright (c) 2000, 2001 Kevin Forchione. All rights reserved.
//  Based on ADV.T (c) and STD.T (c) Michael Roberts.
//
//  This file is part of the ALT replacement library for ADV.T and 
//  STD.T and requires TADS 2.5.1 or later.
//
////////////////////////////////////////////////////////////////////////

#ifndef _SYS_VERB_H_
#define _SYS_VERB_H_

#include <deepverb.h>
#include <darkverb.h>
#include <clock.h>
#include <initrestart.t>
#include <quitgame.t>
#include <scorerank.t>
#include <scorestatus.t>
#include <yesorno.t>
#include <global.t>
#include <story.t>
#include <undoturn.t>

#pragma C+

/*
 *   SysVerb:  A system verb.  Verbs of this class are special verbs that
 *   can be executed without certain normal validations.  For example,
 *   a system verb can be executed in a dark room.  System verbs are
 *   for operations such as saving, restoring, and quitting, which are
 *   not really part of the game.
 */
class SysVerb: DarkVerb
    isSysVerb = true
;

/*
 * 	bannerVerb: SysVerb
 *
 *	This is simply used as a toggle for the sample
 *	gameClock displays, and allows us to move back
 *	and forth between gameClock and classic TADS
 *	banners.
 */
bannerVerb: SysVerb
	verb = 'banner'
	sDesc = "banner"
	action( actor ) = {
		if (global.timeStatus == 2) {
			global.timeStatus = nil;
			"<P><B>";
			"[ Basic gameClock banner display ]";
			"</B><P>";
		} else if ( global.timeStatus == nil ) {
			global.timeStatus = 1;
			"<P><B>";
			"[ Advanced gameClock banner display ]";
			"</B><P>";
		} else if ( global.timeStatus == 1 ) {
			global.timeStatus = 2;
			"<P><B>";
			"[ Classic TADS banner display ]";
			"</B><P>";
		}
		abort;
	}
;
debugVerb: SysVerb
    verb = 'debug'
    sDesc = "debug"
    action(actor) = {
        self.enterDebugger(actor);
        abort;
    }
    enterDebugger(actor) = {
        if (debugTrace())
            "You can't think this game has any bugs left in it... ";
    }
;
quitVerb: SysVerb
    verb = 'quit' 'q'
    sDesc = "quit"
    action(actor) = {
        self.quitTheGame(actor);
        abort;
    }
    quitTheGame(actor) = {
        local ret;

        scoreRank();
        "\bDo you really want to quit? (YES or NO) > ";
        ret = yesOrNo();
        "\b";
        if (ret == 1) {
            story.terminate;    // allow user good-bye message
            quitGame();
        } else {
            "Okay. ";
        }
    }
;
restartVerb: SysVerb
    verb = 'restart'
    sDesc = "restart"
    action(actor) = {
        self.restartGame(actor);
        abort;
    }
    restartGame(actor) = {
        local ret;
        while (true) {
            "Are you sure you want to start over? (YES or NO) > ";
            ret = yesOrNo();
            if (ret == 1) {
                "\n";
                scoreStatus(0, 0);
                restart(initRestart, global.initRestartParam);
                break;
            } else if (ret == 0) {
                "\nOkay.\n";
                break;
            }
        }
    }
;
restoreVerb: SysVerb
    verb = 'restore'
    sDesc = "restore"
    doAction = 'Restore'
    action(actor) = {
        self.restoreGame(actor);
        abort;
    }
    restoreGame(actor) = {
        local saveFile;
        
        saveFile = askfile('File to restore game from',
                            ASKFILE_PROMPT_OPEN, FILE_TYPE_SAVE,
                            ASKFILE_EXT_RESULT);
        switch(saveFile[1]) {
        case ASKFILE_SUCCESS:
            return story.mainRestore(saveFile[2]);

        case ASKFILE_CANCEL:
            "Canceled. ";
            return nil;

        case ASKFILE_FAILURE:
        default:
            "Failed. ";
            return nil;
        }
    }
;
saveVerb: SysVerb
    verb = 'save'
    sDesc = "save"
    doAction = 'Save'
    action(actor) = {
        self.saveGame(actor);
        abort;
    }
    saveGame(actor) = {
        local saveFile;
        
        saveFile = askfile('File to save game in',
                            ASKFILE_PROMPT_SAVE, FILE_TYPE_SAVE,
                            ASKFILE_EXT_RESULT);
        switch(saveFile[1]) {
        case ASKFILE_SUCCESS:
            if (save(saveFile[2])) {
                "An error occurred saving the game position to the file. ";
                return nil;
            } else {
                "Saved. ";
                return true;
            }

        case ASKFILE_CANCEL:
            "Canceled. ";
            return nil;
            
        case ASKFILE_FAILURE:
        default:
            "Failed. ";
            return nil;
        }
    }
;
scoreVerb: SysVerb
    verb = 'score' 'status'
    sDesc = "score"
    action(actor) = {
        self.showScore(actor);
        abort;
    }
    showScore(actor) = {
        scoreRank();
    }
;
scriptVerb: SysVerb
    verb = 'script'
    sDesc = "script"
    doAction = 'Script'
    action(actor) = {
        self.startScripting(actor);
        abort;
    }
    startScripting(actor) = {
        local scriptfile;
        
        scriptfile = askfile('File to write transcript to',
                              ASKFILE_PROMPT_SAVE, FILE_TYPE_LOG,
                              ASKFILE_EXT_RESULT);

        switch(scriptfile[1]) {
        case ASKFILE_SUCCESS:
            logging(scriptfile[2]);
            "All text will now be saved to the script file.
            Type UNSCRIPT at any time to discontinue scripting.";
            break;

        case ASKFILE_CANCEL:
            "Canceled. ";
            break;

        case ASKFILE_FAILURE:
            "Failed. ";
            break;
        }
    }
;
terseVerb: SysVerb
    verb = 'brief' 'terse'
    sDesc = "terse"
    action(actor) = {
        self.terseMode(actor);
        abort;
    }
    terseMode(actor) = {
        "Okay, now in TERSE mode.\n";
        global.verbose = nil;
    }
;
textStyleVerb: SysVerb
    verb = 'text' 'text_style'
    sDesc = "text style"
    action(actor) = {
        global.textStyleAggregate = !global.textStyleAggregate;
        "<B>[Text Style is now set to << global.textStyleAggregate ? "aggregated" : 
            "unaggregated" >>.]</B>";
    }
;
/*	
 *	timeVerb: SysVerb
 *
 *	The time verb produces a display for those systems running a
 *	non-HTML run-time. This allows the game to display the time
 *	whether a banner is present or not.
 */
timeVerb: SysVerb
	verb = 'time'
	sDesc = "time"
	action( actor ) = {
        say(gameClock.getDay(CLOCK_DAY_FULL));
        if (global.timeStatus == 1) {
        	", "; say(gameClock.getDate(CLOCK_DATE_EUROPEAN));
        }
        " "; say(gameClock.getTime(CLOCK_TIME_HHMMTT));
        " ";
        abort;
	}
;
undoVerb: SysVerb
    verb = 'undo'
    sDesc = "undo"
    action(actor) = {
        self.undoMove(actor);
        abort;
    }
    undoMove(actor) = {
        /* do TWO undo's - one for this 'undo', one for previous command */
        if (undoTurn() && undoTurn()) {
            "(Undoing one command)\b";
            parserGetMe().location.lookAround(true);
            scoreStatus(global.score, global.turnsofar);
        } else "No more undo information is available. ";
    }
;
unscriptVerb: SysVerb
    verb = 'unscript'
    sDesc = "unscript"
    action(actor) = {
        self.stopScripting(actor);
        abort;
    }
    stopScripting(actor) = {
        logging(nil);
        "Script closed.\n";
    }
;
verboseVerb: SysVerb
    verb = 'verbose'
    sDesc = "verbose"
    action(actor) = {
        self.verboseMode(actor);
        abort;
    }
    verboseMode(actor) = {
        "Okay, now in VERBOSE mode.\n";
        global.verbose = true;
        parserGetMe().location.lookAround(true);
    }
;
versionVerb: SysVerb
    verb = 'version'
    sDesc = "version"
    action(actor) = {
        self.showVersion(actor);
        abort;
    }
    showVersion(actor) = {
        story.versionDesc;
    }
;

#pragma C-

#endif /* _SYS_VERB_H_ */
