{$B-,F-,I+,R+}

unit CStyle;

{ Define TStyle - a class for various drawing styles }

{ Copyright 1989
  Scott Bussinger
  110 South 131st Street
  Tacoma, WA  98444
  (206)531-8944
  Compuserve 72247,2671 }

interface

uses Graph,CObject,CMouse,CWindow;

const MaxPanes = 16;

type TPaneWindowPtr = ^TPaneWindow;
     TPaneWindow = object(TDrawingWindow)
       constructor Init(Bordered: boolean;X1,Y1,X2,Y2: real);
       procedure Define(Pane: integer); virtual; { Define a new pane }
       procedure DrawIcon(Marked: boolean); virtual; { Draw the icon for this pane }
       function Select: boolean; virtual;        { Select this pane }
       end;

type TMultipanedWindowPtr = ^TMultipanedWindow;
     TMultipanedWindow = object(TWindow)
       fCurrentPane: integer;
       fNumPanes: integer;
       fPane: array[0..MaxPanes-1] of TPaneWindowPtr;
       constructor Init(Bordered: boolean;X1,Y1,X2,Y2: real); { Initialize a multipaned window }
       destructor Done; virtual;                 { Release a multipaned window }
       procedure ChangePane(Pane: integer); virtual; { Change to a new active pane }
       function CheckMouse: boolean; virtual;    { Check if the mouse is in this window }
       function CreatePane(Pane: integer;Bordered: boolean;X1,Y1,X2,Y2: real): TPaneWindowPtr; virtual;
       function CurrentPane: TPaneWindowPtr;     { Get the current pane in window }
       procedure Partition(Bordered: boolean;X1,Y1,X2,Y2: real;Across,Down: integer); { Partition a window with lots of panes }
       procedure SetCursor; virtual;             { Set the mouse cursor for the window }
       end;

type TColorPanePtr = ^TColorPane;
     TColorPane = object(TPaneWindow)
       constructor Init(Bordered: boolean;X1,Y1,X2,Y2: real);
       procedure Define(Pane: integer); virtual; { Define a new color pane }
       procedure DrawIcon(Marked: boolean); virtual; { Draw the icon for this color }
       function Select: boolean; virtual;        { Select this color }
       end;

type TColorWindowPtr = ^TColorWindow;
     TColorWindow = object(TMultipanedWindow)
       constructor Init(Bordered: boolean;X1,Y1,X2,Y2: real); { Initialize a color selection window }
       function CreatePane(Pane: integer;Bordered: boolean;X1,Y1,X2,Y2: real): TPaneWindowPtr; virtual;
       end;

type TFillPanePtr =^TFillPane;
     TFillPane = object(TPaneWindow)
       constructor Init(Bordered: boolean;X1,Y1,X2,Y2: real);
       procedure Define(Pane: integer); virtual; { Define a new fill mask pane }
       procedure DrawIcon(Marked: boolean); virtual; { Draw the icon for this fill mask }
       function Select: boolean; virtual;        { Select this fill mask }
       end;

type TFillWindowPtr = ^TFillWindow;
     TFillWindow = object(TMultipanedWindow)
       constructor Init(Bordered: boolean;X1,Y1,X2,Y2: real); { Initialize a fill mask selection window }
       function CreatePane(Pane: integer;Bordered: boolean;X1,Y1,X2,Y2: real): TPaneWindowPtr; virtual;
       end;

type TLinePanePtr = ^TLinePane;
     TLinePane = object(TPaneWindow)
       constructor Init(Bordered: boolean;X1,Y1,X2,Y2: real);
       procedure Define(Pane: integer); virtual; { Define a new line style pane }
       procedure DrawIcon(Marked: boolean); virtual; { Draw the icon for this line style }
       function Select: boolean; virtual;        { Select this line style pane }
       end;

type TLineWindowPtr = ^TLineWindow;
     TLineWindow = object(TMultipanedWindow)
       constructor Init(Bordered: boolean;X1,Y1,X2,Y2: real);
       function CreatePane(Pane: integer;Bordered: boolean;X1,Y1,X2,Y2: real): TPaneWindowPtr; virtual;
       end;

type TFontPanePtr = ^TFontPane;
     TFontPane = object(TPaneWindow)
       constructor Init(Bordered: boolean;X1,Y1,X2,Y2: real);
       procedure Define(Pane: integer); virtual; { Define a new font pane }
       procedure DrawIcon(Marked: boolean); virtual; { Draw the icon for this font }
       function Select: boolean; virtual;        { Select this font }
       end;

type TFontWindowPtr = ^TFontWindow;
     TFontWindow = object(TMultipanedWindow)
       constructor Init(Bordered: boolean;X1,Y1,X2,Y2: real); { Initialize a font selection window }
       function CreatePane(Pane: integer;Bordered: boolean;X1,Y1,X2,Y2: real): TPaneWindowPtr; virtual;
       end;

type TColorStylePanePtr = ^TColorStylePane;
     TColorStylePane = object(TPaneWindow)
       constructor Init(Bordered: boolean;X1,Y1,X2,Y2: real);
       procedure DrawIcon(Marked: boolean); virtual; { Draw the icon showing the current color }
       end;

type TFillStylePanePtr = ^TFillStylePane;
     TFillStylePane = object(TPaneWindow)
       constructor Init(Bordered: boolean;X1,Y1,X2,Y2: real);
       procedure DrawIcon(Marked: boolean); virtual; { Draw the icon showing the fill mask }
       end;

type TLineStylePanePtr = ^TLineStylePane;
     TLineStylePane = object(TPaneWindow)
       constructor Init(Bordered: boolean;X1,Y1,X2,Y2: real);
       procedure DrawIcon(Marked: boolean); virtual; { Draw the icon showing the line style }
       end;

type TFontStylePanePtr = ^TFontStylePane;
     TFontStylePane = object(TPaneWindow)
       constructor Init(Bordered: boolean;X1,Y1,X2,Y2: real);
       procedure DrawIcon(Marked: boolean); virtual; { Draw the icon showing the font }
       end;

type TStyleWindow = object(TMultipanedWindow)
       fCurrentWindow: TMultipanedWindowPtr;
       fCurrentWindowBordered: boolean;
       fWX1: real;
       fWX2: real;
       fWY1: real;
       fWY2: real;
       constructor Init(Bordered: boolean;X1,Y1,X2,Y2: real); { Initialize a current style selection window }
       destructor Done; virtual;                 { Release a current style window }
       function CheckMouse: boolean; virtual;    { Check if the mouse is in this window }
       function CreatePane(Pane: integer;Bordered: boolean;X1,Y1,X2,Y2: real): TPaneWindowPtr; virtual;
       end;

implementation

constructor TMultipanedWindow.Init(Bordered: boolean;
                                   X1,Y1,X2,Y2: real);
  { Initialize a multipaned window }
  begin
  TWindow.Init(Bordered,X1,Y1,X2,Y2)
  end;

destructor TMultipanedWindow.Done;
  { Release a multipaned window }
  var I: integer;
  begin
  for I := 0 to fNumPanes-1 do
    dispose(fPane[I],Done);
  TWindow.Done
  end;

procedure TMultipanedWindow.ChangePane(Pane: integer);
  { Change to a new active pane }
  begin
  fCurrentPane := Pane                           { Change the current pane }
  end;

function TMultipanedWindow.CheckMouse: boolean;
  { Check if the mouse is in this window }
  var I: integer;
  begin
  CheckMouse := false;
  if TWindow.CheckMouse then                     { See if we're in this window at all }
    begin
    I := 0;                                      { Check a multipaned window by looking at each of the panes }
    while (I<fNumPanes) and not (fPane[I]^.CheckMouse) do
      inc(I);
    if I < fNumPanes then
      begin
      CheckMouse := true;
      SetCursor;                                 { Change to the appropriate mouse cursor }
      if (Mouse.GetButton(Left)=Released) and    { Was the button just released? }
         fPane[I]^.Select then                   { Does this pane cause a mode change? }
        ChangePane(I)
      end
    end
  end;

function TMultipanedWindow.CreatePane(Pane: integer;
                                      Bordered: boolean;
                                      X1,Y1,X2,Y2: real): TPaneWindowPtr;
  { Create a new pane }
  begin
  CreatePane := new(TPaneWindowPtr,Init(Bordered,X1,Y1,X2,Y2))
  end;

function TMultipanedWindow.CurrentPane: TPaneWindowPtr;
  { Get the current pane }
  begin
  CurrentPane := fPane[fCurrentPane]
  end;

procedure TMultipanedWindow.Partition(Bordered: boolean;
                                      X1,Y1,X2,Y2: real;
                                      Across,Down: integer);
  { Partition a window into an array of panes }
  var I: integer;
      R,C: integer;
  begin
  fNumPanes := Across * Down;
  if fNumPanes > MaxPanes then
    begin
    fNumPanes := MaxPanes;
    Across := MaxPanes div Down
    end;
  for I := 0 to fNumPanes-1 do
    begin
    R := I div Across;
    C := I mod Across;
    fPane[I] := CreatePane(I,Bordered,C*(X2-X1)/Across+X1,R*(Y2-Y1)/Down+Y1,(C+1)*(X2-X1)/Across+X1,(R+1)*(Y2-Y1)/Down+Y1);
    fPane[I]^.Define(I);
    fPane[I]^.DrawIcon(false)
    end;
  fCurrentPane := 0;
  ChangePane(fCurrentPane)
  end;

procedure TMultipanedWindow.SetCursor;
  { Set the mouse cursor for the window }
  begin
  Mouse.SetCursor(HandCursor)
  end;

constructor TPaneWindow.Init(Bordered: boolean;
                             X1,Y1,X2,Y2: real);
  { Initialize a pane }
  begin
  TDrawingWindow.Init(Bordered,X1,Y1,X2,Y2)
  end;

procedure TPaneWindow.Define(Pane: integer);
  { Define a new pane }
  begin
  { Should be overridden in all subclasses }
  end;

procedure TPaneWindow.DrawIcon(Marked: boolean);
  { Draw the icon for this pane }
  begin
  Mouse.Hide;                                    { Keep the display clean }
  Activate                                       { Switch to this window }
  end;

function TPaneWindow.Select: boolean;
  { Select this pane }
  { Return true if selecting this pane should change the current pane or
    false if the previous pane stays in effect. }
  begin
  Select := true;
  CurrentCanvas^.Activate
  end;

constructor TColorPane.Init(Bordered: boolean;
                            X1,Y1,X2,Y2: real);
  { Initialize a pane }
  begin
  TPaneWindow.Init(Bordered,X1,Y1,X2,Y2)
  end;

procedure TColorPane.DrawIcon(Marked: boolean);
  { Draw the icon for this color }
  var Viewport: ViewportType;
  begin
  TPaneWindow.DrawIcon(Marked);
  GetViewSettings(Viewport);
  with Viewport do
    Bar(0,0,X2-X1,Y2-Y1)
  end;

procedure TColorPane.Define(Pane: integer);
  { Define a new color pane }
  begin
  TPaneWindow.Define(Pane);
  ChangeColor(Pane)
  end;

function TColorPane.Select: boolean;
  { Select this color }
  var TempColor: integer;
  begin
  TempColor := GetColor;
  Select := TPaneWindow.Select;
  ChangeColor(TempColor)
  end;

constructor TColorWindow.Init(Bordered: boolean;
                              X1,Y1,X2,Y2: real);
  { Initialize a color selection window }
  begin
  TMultipanedWindow.Init(false,X1,Y1,X2,Y2);
  if GetMaxColor = 1                             { Watch for this special case, for a better looking display }
   then
    Partition(Bordered,X1,Y1,X2,Y2,2,1)
   else
    Partition(Bordered,X1,Y1,X2,Y2,(GetMaxColor+1) div 2,2);
  ChangePane(fNumPanes-1)
  end;

function TColorWindow.CreatePane(Pane: integer;
                                 Bordered: boolean;
                                 X1,Y1,X2,Y2: real): TPaneWindowPtr;
  { Create a new color pane }
  begin
  CreatePane := new(TColorPanePtr,Init(Bordered,X1,Y1,X2,Y2))
  end;

constructor TFillPane.Init(Bordered: boolean;
                           X1,Y1,X2,Y2: real);
  { Initialize a pane }
  begin
  TPaneWindow.Init(Bordered,X1,Y1,X2,Y2)
  end;

procedure TFillPane.DrawIcon(Marked: boolean);
  { Draw the icon for this fill mask }
  var SaveFill: FillPatternType;
      Viewport: ViewportType;
  begin
  TPaneWindow.DrawIcon(Marked);
  GetViewSettings(Viewport);
  with Viewport do
    begin
    GetFillPattern(SaveFill);
    ChangeFill(FillPattern[SolidFill],SystemBackground);
    ChangeColor(SystemBackground);
    Bar(0,0,X2-X1,Y2-Y1);
    ChangeFill(SaveFill,SystemWhite);
    ChangeColor(SystemWhite);
    Bar(0,0,X2-X1,Y2-Y1)
    end
  end;

procedure TFillPane.Define(Pane: integer);
  { Define a new fill mask pane }
  begin
  TPaneWindow.Define(Pane);
  ChangeFill(FillPattern[Pane],GetColor)
  end;

function TFillPane.Select: boolean;
  { Select this fill mask }
  var TempFillPattern: FillPatternType;
  begin
  GetFillPattern(TempFillPattern);
  Select := TPaneWindow.Select;
  ChangeFill(TempFillPattern,GetColor)
  end;

constructor TFillWindow.Init(Bordered: Boolean;
                             X1,Y1,X2,Y2: real);
  { Initialize a fill mask selection window }
  begin
  TMultipanedWindow.Init(false,X1,Y1,X2,Y2);
  Partition(Bordered,X1,Y1,X2,Y2,MaxFillPatterns div 2,2);
  ChangePane(fNumPanes-1)
  end;

function TFillWindow.CreatePane(Pane: integer;
                                Bordered: boolean;
                                X1,Y1,X2,Y2: real): TPaneWindowPtr;
  { Create a new fill mask pane }
  begin
  CreatePane := new(TFillPanePtr,Init(Bordered,X1,Y1,X2,Y2))
  end;

constructor TLinePane.Init(Bordered: boolean;
                           X1,Y1,X2,Y2: real);
  { Initialize a pane }
  begin
  TPaneWindow.Init(Bordered,X1,Y1,X2,Y2)
  end;

procedure TLinePane.DrawIcon(Marked: boolean);
  { Draw the icon for this line style }
  var Viewport: ViewportType;
  begin
  TPaneWindow.DrawIcon(Marked);
  GetViewSettings(Viewport);
  with Viewport do
    begin
    ChangeColor(SystemBackground);
    Bar(0,0,X2-X1,Y2-Y1);
    ChangeColor(SystemWhite);
    MoveTo(0,(Y2-Y1) div 3);
    LineTo(X2-X1,(Y2-Y1) div 3);
    MoveTo(0,2*(Y2-Y1) div 3);
    LineTo(X2-X1,2*(Y2-Y1) div 3)
    end
  end;

procedure TLinePane.Define(Pane: integer);
  { Define a new line style pane }
  begin
  TPaneWindow.Define(Pane);
  SetLineStyle(UserBitLn,LineStyle[Pane],NormWidth)
  end;

function TLinePane.Select: boolean;
  { Select this line style }
  var LineSettings: LineSettingsType;
  begin
  GetLineSettings(LineSettings);
  Select := TPaneWindow.Select;
  SetLineStyle(UserBitLn,LineSettings.Pattern,NormWidth)
  end;

constructor TLineWindow.Init(Bordered: boolean;
                             X1,Y1,X2,Y2: real);
  { Initialize a line style selection window }
  begin
  TMultipanedWindow.Init(false,X1,Y1,X2,Y2);
  Partition(Bordered,X1,Y1,X2,Y2,MaxLineStyles div 2,2);
  ChangePane(fNumPanes-1)
  end;

function TLineWindow.CreatePane(Pane: integer;
                                Bordered: boolean;
                                X1,Y1,X2,Y2: real): TPaneWindowPtr;
  { Create a new line style window pane }
  begin
  CreatePane := new(TLinePanePtr,Init(Bordered,X1,Y1,X2,Y2))
  end;

constructor TFontPane.Init(Bordered: boolean;
                           X1,Y1,X2,Y2: real);
  { Initialize a pane }
  begin
  TPaneWindow.Init(Bordered,X1,Y1,X2,Y2)
  end;

procedure TFontPane.DrawIcon(Marked: boolean);
  { Draw the icon for this font }
  var Viewport: ViewportType;
  begin
  TPaneWindow.DrawIcon(Marked);
  GetViewSettings(Viewport);
  with Viewport do
    begin
    ChangeColor(SystemBackground);
    Bar(0,0,X2-X1,Y2-Y1);
    ChangeColor(SystemWhite);
    case CurrentFont of
      Triplex: FitText(Triplex,'Triplex');
      Small: FitText(Small,'Small');
      SansSerif: FitText(SansSerif,'Sans Serif');
      Gothic: FitText(Gothic,'Gothic');
      Bold: FitText(Bold,'Bold');
      Simplex: FitText(Simplex,'Simplex');
      TriplexScript: FitText(TriplexScript,'Tri-Script');
      Script: FitText(Script,'Script');
      EuroStyle: FitText(EuroStyle,'Euro');
      Complex: FitText(Complex,'Complex')
      end
    end
  end;

procedure TFontPane.Define(Pane: integer);
  { Define a new font pane }
  begin
  TPaneWindow.Define(Pane);
  SetFont(Font(Pane),CurrentHeight,CurrentWidth)
  end;

function TFontPane.Select: boolean;
  { Select this font }
  var TempFont: Font;
  begin
  TempFont := CurrentFont;
  Select := TPaneWindow.Select;
  SetFont(TempFont,CurrentHeight,CurrentWidth)
  end;

constructor TFontWindow.Init(Bordered: boolean;
                             X1,Y1,X2,Y2: real);
  { Initialize a font selection window }
  begin
  TMultipanedWindow.Init(false,X1,Y1,X2,Y2);
  Partition(Bordered,X1,Y1,X2,Y2,5,2);
  ChangePane(ord(Triplex))
  end;

function TFontWindow.CreatePane(Pane: integer;
                                Bordered: boolean;
                                X1,Y1,X2,Y2: real): TPaneWindowPtr;
  { Create a new font window pane }
  begin
  CreatePane := new(TFontPanePtr,Init(Bordered,X1,Y1,X2,Y2))
  end;

constructor TColorStylePane.Init(Bordered: boolean;
                                 X1,Y1,X2,Y2: real);
  { Initialize a pane }
  begin
  TPaneWindow.Init(Bordered,X1,Y1,X2,Y2)
  end;

procedure TColorStylePane.DrawIcon(Marked: boolean);
  { Draw the icon for the current color style }
  var TempColor: integer;
      Viewport: ViewportType;
  begin
  CurrentCanvas^.Activate;
  TempColor := GetColor;
  TPaneWindow.DrawIcon(Marked);
  ChangeColor(TempColor);
  GetViewSettings(Viewport);
  with Viewport do
    Bar(0,0,X2-X1,Y2-Y1)
  end;

constructor TFillStylePane.Init(Bordered: boolean;
                                X1,Y1,X2,Y2: real);
  { Initialize a pane }
  begin
  TPaneWindow.Init(Bordered,X1,Y1,X2,Y2)
  end;

procedure TFillStylePane.DrawIcon(Marked: boolean);
  { Draw the icon for the current fill mask }
  var SaveFillPattern: FillPatternType;
      Viewport: ViewportType;
  begin
  CurrentCanvas^.Activate;
  GetFillPattern(SaveFillPattern);
  TPaneWindow.DrawIcon(Marked);
  GetViewSettings(Viewport);
  with Viewport do
    begin
    ChangeFill(FillPattern[SolidFill],SystemBackground);
    ChangeColor(SystemBackground);
    Bar(0,0,X2-X1,Y2-Y1);
    ChangeColor(SystemWhite);
    ChangeFill(SaveFillPattern,SystemWhite);
    Bar((X2-X1) div 10,2*(Y2-Y1) div 10,9*(X2-X1) div 10,8*(Y2-Y1) div 10)
    end
  end;

constructor TLineStylePane.Init(Bordered: boolean;
                                X1,Y1,X2,Y2: real);
  { Initialize a pane }
  begin
  TPaneWindow.Init(Bordered,X1,Y1,X2,Y2)
  end;

procedure TLineStylePane.DrawIcon(Marked: boolean);
  { Draw the icon for the current line style }
  var LineSettings: LineSettingsType;
      Viewport: ViewportType;
  begin
  CurrentCanvas^.Activate;
  GetLineSettings(LineSettings);
  TPaneWindow.DrawIcon(Marked);
  GetViewSettings(Viewport);
  with Viewport do
    begin
    ChangeColor(SystemBackground);
    Bar(0,0,X2-X1,Y2-Y1);
    ChangeColor(SystemWhite);
    SetLineStyle(UserBitLn,LineSettings.Pattern,NormWidth);
    MoveTo(0,(Y2-Y1) div 3);
    LineTo(X2-X1,(Y2-Y1) div 3);
    MoveTo(0,2*(Y2-Y1) div 3);
    LineTo(X2-X1,2*(Y2-Y1) div 3)
    end
  end;

constructor TFontStylePane.Init(Bordered: boolean;
                                X1,Y1,X2,Y2: real);
  { Initialize a pane }
  begin
  TPaneWindow.Init(Bordered,X1,Y1,X2,Y2)
  end;

procedure TFontStylePane.DrawIcon(Marked: boolean);
  { Draw the icon for the current font }
  var TempFont: Font;
      Viewport: ViewportType;
  begin
  CurrentCanvas^.Activate;
  TempFont := CurrentFont;
  TPaneWindow.DrawIcon(Marked);
  GetViewSettings(Viewport);
  with Viewport do
    begin
    ChangeColor(SystemBackground);
    Bar(0,0,X2-X1,Y2-Y1);
    ChangeColor(SystemWhite);
    case TempFont of
      Triplex: FitText(Triplex,'Triplex');
      Small: FitText(Small,'Small');
      SansSerif: FitText(SansSerif,'Sans Serif');
      Gothic: FitText(Gothic,'Gothic');
      Bold: FitText(Bold,'Bold');
      Simplex: FitText(Simplex,'Simplex');
      TriplexScript: FitText(TriplexScript,'Tri-Script');
      Script: FitText(Script,'Script');
      EuroStyle: FitText(EuroStyle,'Euro');
      Complex: FitText(Complex,'Complex')
      end
    end
  end;

constructor TStyleWindow.Init(Bordered: boolean;
                              X1,Y1,X2,Y2: real);
  { Initialize a style selection window }
  begin
  TMultipanedWindow.Init(false,X1,Y1,0.10*(X2-X1)+X1,Y2);
  Partition(Bordered,X1,Y1,0.10*(X2-X1)+X1,Y2,1,4);
  fWX1 := 0.11*(X2-X1)+X1;                       { Remember the window coordinates }
  fWY1 := 0.50*(Y2-Y1)+Y1;                       { Choice window is only half as tall }
  fWX2 := X2;
  fWY2 := Y2;
  fCurrentWindow := new(TColorWindowPtr,Init(Bordered,fWX1,fWY1,fWX2,fWY2));
  fCurrentWindowBordered := Bordered
  end;

destructor TStyleWindow.Done;
  { Release a style selection window }
  begin
  dispose(fCurrentWindow,Done);
  TMultipanedWindow.Done
  end;

function TStyleWindow.CheckMouse: boolean;
  { Check if the mouse is in this window }
  var PreviousActivePane: integer;
  begin
  PreviousActivePane := fCurrentPane;
  CheckMouse := true;
  if TMultipanedWindow.CheckMouse
   then
    begin
    if (Mouse.GetButton(Left)=Released) and      { Was the button just released? }
       (fCurrentPane<>PreviousActivePane) then   { Was a new window selected? }
      begin
      dispose(fCurrentWindow,Done);              { Release the old window }
      case fCurrentPane of                       { Create the new window }
        0: fCurrentWindow := new(TColorWindowPtr,Init(fCurrentWindowBordered,fWX1,fWY1,fWX2,fWY2));
        1: fCurrentWindow := new(TFillWindowPtr,Init(fCurrentWindowBordered,fWX1,fWY1,fWX2,fWY2));
        2: fCurrentWindow := new(TLineWindowPtr,Init(fCurrentWindowBordered,fWX1,fWY1,fWX2,fWY2));
        3: fCurrentWindow := new(TFontWindowPtr,Init(fCurrentWindowBordered,fWX1,fWY1,fWX2,fWY2))
        end
      end
    end
   else
    if fCurrentWindow^.CheckMouse
     then
      begin
      if Mouse.GetButton(Left) = Released then   { Was the button just released? }
        fPane[fCurrentPane]^.DrawIcon(false)
      end
     else
      CheckMouse := false
  end;

function TStyleWindow.CreatePane(Pane: integer;
                                 Bordered: boolean;
                                 X1,Y1,X2,Y2: real): TPaneWindowPtr;
  { Create a new style selection window pane }
  begin
  case Pane of
    0: CreatePane := new(TColorStylePanePtr,Init(Bordered,X1,Y1,X2,Y2));
    1: CreatePane := new(TFillStylePanePtr,Init(Bordered,X1,Y1,X2,Y2));
    2: CreatePane := new(TLineStylePanePtr,Init(Bordered,X1,Y1,X2,Y2));
    3: CreatePane := new(TFontStylePanePtr,Init(Bordered,X1,Y1,X2,Y2))
    end
  end;

end.
