(*  :Title:	Inverse z-Transforms  *)

(*  :Authors:	Brian Evans, James McClellan  *)

(*
    :Summary:	One-dimensional and separable multidimensional
		inverse z-transforms
 *)

(*  :Context:	SignalProcessing`Digital`InvZTransform`  *)

(*  :PackageVersion:  2.4	*)

(*
    :Copyright:	Copyright 1989-1991 by Brian L. Evans
		Georgia Tech Research Corporation

	Permission to use, copy, modify, and distribute this software
	and its documentation for any purpose and without fee is
	hereby granted, provided that the above copyright notice
	appear in all copies and that both that copyright notice and
	this permission notice appear in supporting documentation,
	and that the name of the Georgia Tech Research Corporation,
	Georgia Tech, or Georgia Institute of Technology not be used
	in advertising or publicity pertaining to distribution of the
	software without specific, written prior permission.  Georgia
	Tech makes no representations about the suitability of this
	software for any purpose.  It is provided "as is" without
	express or implied warranty.
 *)

(*  :History:	*)

(*  :Keywords:	z-transform, region of convergence  *)

(*  :Source:	*)

(* 
    :Warning:	Keep conditional clauses on the same line; separating them
		  by RETURNS will confuse Mathematica (e.g., the package
		  context is never reset to Global); you can place a
		  newline after an operator (like &&)

		The n argument of myinvz must always be a symbol NOT
		  an expression (see postinvzrules and local shift operator)
 *)

(*  :Mathematica Version:  1.2 or 2.0  *)

(*  :Limitation:  *)

(*
    :Discussion:  1-D Local state rules base has 57 rules:
		    I.   rational transforms       26 rules
		    II.  non-rational transforms   10 rules
		    III. transform properties       9 rules
		    IV.  transform DSP structures   2 rules
		    V.   transform strategies      10 rules

	Note that  the  partial  fractions  decomposition  strategy is
	implemented  as  two  rules --   one in the rational transform
	pairs section and one in the properties section.

	At each step in the inverse rules base, the current expression
	has a local state associated with it.   This state consists of
	a list of nine boolean values.   Each boolean value is associ-
	ated with a strategy.   If  an  element  is  True,  then  that
	strategy has not been tried yet;  if False, then that strategy
	has already been tried, and it will not be tried again.  Thus,
	local state is used  to  prevent infinite loops which would be
	caused by the repetitive application of certain strategy rules.
	See the section  S T A T E  D E F I N I T I O N  below and see
	section V of the rules.

	Unlike the  other  symbolic  transforms, these rules are main-
	tained in a list called InvZTransformRules.   This  was neces-
	sary because Mathematica  reordered these rules in undesirable
	ways if they were coded  as a set of transformation functions.
	Another benefit of  the list form is that it allowed more con-
	trol over how rules are applied.

	In general,  this rule base works on expressions with negative
	powers of z.  Great  pains  are  taken  to  convert  any input
	expressions  to  this form accordingly.  This conversion works
	very well for rational z-transform functions.

	The  InvZTransform  function  first  calls the interface rules
	(InvZTransformInterfaceRules), which handle default arguments,
	error messages, and multidimensional transforms.   These rules
	simply call  MyInvZTransform  once per dimension of the trans-
	form. This inverse z-transform is biased toward causal systems.
	However,  by specifying the proper region of convergence (ROC),
	anti-causal sequences can be found (see InvZTransform::usage).

	The driver for the one-dimensional  rule base is the six argu-
	ment version of MyInvZTransform.  MyInvZTransform[X, z, n, rm,
	rp, s, op] either returns the  inverse transform as a function
	of n or an approximation to  the actual inverse transform.  If
	the rule base can do neither,  which  should  not  happen, the
	interface function cleanup will report any errors.   Arguments
	of MyInvZTransform:

	X  z-transform function		rm  Rminus of ROC
	z  z-transform variable		rp  Rplus of ROC
	n  "time"-domain variable	s   local state semaphores
					op  options
 *)

(*
   :Functions:	InvZTransform
		MultByExponentialQ
		MyInvZTransform
		MultiDInvZTransform
		PartialFractions
		PartialFractionsQ
 *)


If [ TrueQ[ $VersionNumber >= 2.0 ],
     Off[ General::spell ];
     Off[ General::spell1 ] ];


(*  B E G I N     P A C K A G E  *)

BeginPackage[ "SignalProcessing`Digital`InvZTransform`",
	      "SignalProcessing`Digital`ZSupport`",
	      "SignalProcessing`Support`TransSupport`",
	      "SignalProcessing`Support`ROC`",
	      "SignalProcessing`Support`SigProc`",
	      "SignalProcessing`Support`SupCode`",
	      "SignalProcessing`Support`FilterSupport`" ]


(*  U S A G E     I N F O R M A T I O N  *)

InvZTransform::usage =
	"InvZTransform[e,z] or InvZTransform[e,z,n] gives the inverse \
	z-transform of e.  Here, e can be a function of z, a list, or \
	a z-transform object (see ZTransData).  As a list, e should \
	contain three elements:  F[z], rminus, and rplus, such that \
	the region of convergence is rminus < |z| < rplus.   If e is \
	a z-transform object, then the inverse z-transform is simply \
	InvZTransform[e] because e contains all necessary information. \
	If the returned expression is not the form you desire, then \
	apply Simplify[], Cancel[], Together[], etc., to the expression."

(*  E N D     U S A G E     I N F O R M A T I O N  *)


Begin["`Private`"]


(*  U S E R     I N T E R F A C E  *)

(*  InvZ operator  *)
Unprotect[InvZ]
InvZ/: TheFunction[ InvZ[z_, n_][f_] ] := InvZTransform[f, z, n]
Protect[InvZ]

(*  InvZTransform  *)
InvZTransform/: Options[InvZTransform] :=
	{ Dialogue -> True, Terms -> 10, TransformLookup -> {} }

InvZTransform[x_] :=
	invzdriver[ Options[InvZTransform], x ]
InvZTransform[x_, z_] :=
	invzdriver[ Options[InvZTransform], x, z ]
InvZTransform[x_, z_, n_] :=
	invzdriver[ Options[InvZTransform], x, z, n ]
InvZTransform[x_, z_, n_, op__] :=
	invzdriver[ ToList[op] ~Join~ Options[InvZTransform], x, z, n ]

(*  Driver for inverse z-transform interface  *)
invzdriver[op_, args__] :=
	cleanup [ invztransform[op, args], InformUserQ[Replace[Dialogue, op]] ]

invztransform[op_, args__] :=
	Replace [ zinverse[op, args], InvZTransformInterfaceRules ]

InvZTransformInterfaceRules = {
	zinverse[op_, e_] :>
		invztransform[ op, e, ZVariables[e] ] /; ZTransformQ[e],
	zinverse[op_, e_] :>
		Message[ InvZTransform::novariables, "z", GetVariables[e] ],

	zinverse[op_, e_, z_] :>
		invztransform[op, e, z, DummyVariables[Length[z], Global`n]],

	zinverse[op_, e_List, z_, n_] :>
		invztransform[op, MakeZObject[e, z], z, n],
	zinverse[op_, e_, z_Symbol, n_] :>
		MyInvZTransform[e, z, n, op],
	zinverse[op_, e_, z_List, n_] :>
		MultiDInvZTransform[e, z, n, op]
}

cleanup[ Null, flag_ ] := Null

cleanup[ trans_, flag_ ] :=
	Block [	{},
		If [ flag, explain[trans]; Scan [explain, trans, Infinity] ];
		Simplify [ trans ] ]

explain[ myinvz[f_, z_, rest__] ] :=
	Message[ Transform::incomplete, "inverse z-transform", f, z ]


(*  M E S S A G E S  *)

InvZTransform::badterms = "Non-integer terms for series expansion: ``"
InvZTransform::badROC = "Improper region of convergence in ``."


(*  T R A N S F O R M A T I O N     S T A T E  *)

	(*  Listed in order of appearance in the rule base  *)

polyfactorfield = 1		(* factor denominator of polynomials	     *)
expcheckfield = 2		(* check to see if r^n form		     *)
partialfractionsfield = 3	(* partial fractions on the cur. expression  *)
partialfractionsfield2 = 4	(* partial fractions a second time	     *)
logformfield = 5		(* factoring inside of a logarithmic form    *)
expformfield = 6		(* factoring inside of an exponential form   *)
expandallfield = 7		(* apply ExpandAll[] to current expression   *)
anticausalfield = 8		(* try anti-causal expansion		     *)
cepstrumfield = 9		(* apply rule for complex cepstrum to log    *)
seriesexpansionfield = 10	(* power series expansion of cur. expression *)

statevariables = 10		(* number of state variables *)

(*
    return the initial transformation state, which is a list of whether or not
    partial fractions, the first denominator normalization, and the second
    denominator initialization has been performed yet on that expression.
 *)

initInvZstate[] := Table[True, {statevariables}]
nullInvZstate[] := Table[False, {statevariables}]
anticausalInvZstate[] :=
	Block [	{state},
		state = SetStateField [	nullInvZstate[], logformfield, True ];
		SetStateField [ state, expformfield, True ] ]


(*  S U P P O R T I N G     R O U T I N E S  *)

(*    Since the inverse z-transform rule base is biased toward causal	*)
(*  sequences, the LeftOrRightSided rules adjust the right-sided	*)
(*  inverse returned by InvZTransform.					*)
(*    One of the strategies handles the general case.			*)

leftsided[p_, n_] :=
	( p /. { Step[b_. n + c_.] :> - Step[- b n - b + c] } )
rightsided[p_, n_] := p

LeftOrRightSided[a_, p_, n_, rm_, rp_] := leftsided[p, n]  /; SameQ[Abs[a], rp]
LeftOrRightSided[a_, p_, n_, rm_, rp_] := leftsided[p, n]  /; SameQ[Abs[-a], rp]
LeftOrRightSided[a_, p_, n_, rm_, rp_] := leftsided[p, n]  /; N[Abs[a] > rp]
LeftOrRightSided[a_, p_, n_, rm_, rp_] := rightsided[p, n] /; SameQ[Abs[a], rm]
LeftOrRightSided[a_, p_, n_, rm_, rp_] := rightsided[p, n] /; SameQ[Abs[-a], rm]
LeftOrRightSided[a_, p_, n_, rm_, rp_] := rightsided[p, n] /; N[Abs[a] < rm]
LeftOrRightSided[a_, p_, n_, rm_, rp_] := rightsided[p, n] /; InfinityQ[rp]
LeftOrRightSided[a_, p_, n_, rm_, rp_] := rightsided[p, n]

(*  FixUp -- strip redundant options and remove TransformLookup option  *)
FixUp[ op_ ] :=
	{ Dialogue -> Replace[Dialogue, op], Terms -> Replace[Terms, op] }

(*  MultByExponential  *)
MultByExponentialQ[f_, z_] :=
	Block [	{scale},
		scale = ScalingFactor[f, z];
		( ! SameQ[scale, 1] ) && ( ! SameQ[scale, 0] ) ]

(*  MultiDInvZTransform  *)
MultiDInvZTransform[e_, z_, n_, op_] :=
	MultiDInvTransform [ e, z, n, op, ZTransformQ,
			     MyInvZTransform, MakeZObject, Global`n ]

(*  PartialFractionsDialogue  *)
PartialFractionsDialogue[p_, partfrac_, options_] :=
	Block [ {dialogue},
		dialogue = InformUserQ[Replace[Dialogue, options]];

		(*  Tell the user that we're applying partial fractions  *)
		If [ dialogue,
		     Print["( after the term"];
		     Print["  ", p];
		     Print["  has been broken up into its partial " ];
		     Print["  fractions representation"];
		     Print["  ", partfrac, " )" ] ] ]

(*  PartialFractionsQ  *)
PartialFractionsQ[p_, z_] :=
	MixedPolynomialQ[ Expand[Denominator[p]], z ] &&
	  ! SameQ[ Denominator[p], 1 ]

(*  SeriesDialogue  *)
SeriesDialogue[x_, expansion_, terms_, options_] :=
	Block [	{dialogue},
		dialogue = InformUserQ[ Replace[Dialogue, options] ];

		If [ dialogue,
		     Print["( after breaking up the expression"];
		     Print["  ", x];
		     Print["  into its series representation"];
		     Print["  ", expansion, " )"] ] ]

(*  SeriesExpansion  *)
SeriesExpansion[ True, x_, z_, power_ ] := Series[x, {z, 0, power}]

SeriesExpansion[ False, x_, z_, power_ ] :=
	Block [	{negx, expansion},
		negx = x /. z -> z^-1;
		expansion = Series[negx, {z, 0, power}];
		expansion /. z -> z^-1 ]

(*  SeriesStrategy  *)
SeriesStrategy[ x_, z_, n_, rm_, rp_, s_, op_ ] :=
	Block [ {expansion, exponents, posexpandflag, state, terms},
		terms = Replace[Terms, op];
		state = SetStateField[s, seriesexpansionfield, False];

		(*  If the Terms option is disabled, then take no action.  *)
		If [ ! terms, Return[ myinvz[x, z, n, rm, rp, state, op] ] ];

		If [ ! IntegerQ[terms],
		     Message[ InvZTransform::badterms, terms ];
		     myinvz[x, z, n, rm, rp, state, op] ];

		(*  See if we should expand in pos. or neg. powers	*)
		exponents = GetAllExponents[x, z];
		posexpandflag = If [ Apply[And, Thread[exponents >= 0]],
				     True, False, False ];

		(*  First expansion  *)
		expansion = SeriesExpansion[posexpandflag, x, z, terms - 1];

		(*  Second expansion if first failed  *)
		If [ ! SameQ[Head[expansion], SeriesData],
		     expansion = SeriesExpansion[Not[posexpandflag], x, z, terms - 1] ];

		(*  If this fails, call MyInvZTransform again		*)
		(*  but with series expansion disabled			*)
		If [ ! SameQ[Head[expansion], SeriesData],
		     myinvz[ x, z, n, rm, rp, state, op ],
		     SeriesDialogue[x, expansion, terms, op];
		     state = nullInvZstate[];
		     Map [ myinvz[#1, z, n, rm, rp, state, op]&, 
			   Normal[expansion] ] ] ]


(*  B E G I N     R U L E     B A S E  *)

ROCrewriterules = {
	b_?Positive / Abs[a_] :> Abs[b / a],
	b_?Negative / Abs[a_] :> - Abs[- b / a]
}

MyInvZTransform[ e_, z_, n_, op_ ] :=
	Block [ {rminus, rplus, valid},
		rminus = ReplaceRepeated[GetRMinus[e], SPSimplificationRules];
		rplus = ReplaceRepeated[ GetRPlus[e],
					 Join[SPSimplificationRules,
					      ROCrewriterules] ];
		valid = Apply[And, Thread[ToList[rminus] < ToList[rplus]]];
		If [ TrueQ[! valid],
		     Message[ InvZTransform::badROC, e ],
		     MyInvZTransform[ TheFunction[e], z, n, rminus,
		     		      rplus, initInvZstate[], op ] ] ] /;
	ZTransformQ[e]

MyInvZTransform[ e_, z_, n_, op_ ] :=
	MyInvZTransform[ e, z, n, 0, Infinity, initInvZstate[], op ]


(*   Driver for one-dimensional rule base			*)
(*   First, convert all z /(z - a) forms to 1/(1 - a z^-1)	*) 
(*	since the rule base favors terms with z^-1 terms	*)
(*   Loop until the expression to be inverted does not change.	*)
MyInvZTransform[ e_, z_, n_, rm_, rp_, st_, options_ ] :=
	Block [ {laste, newe, op, trace},
		op = FixUp[options];		(* strip redundant options *)
		trace = SameQ[ Replace[Dialogue, op], All ];
		newe = myinvz[MapAll[convert[#, z]&, e], z, n, rm, rp, st, op];
		While [ ! SameQ[ laste, newe ],
			If [ trace, Print[ newe ]; Print[ "which becomes" ] ];
			laste = newe;
			newe = MapAll[transform, laste] ];

		laste = TransformFixUp[ laste, z, n, options, myinvz,
					False, InvZTransform, Null, Null ];

		newe = laste /. postinvzrules;
		While [ ! SameQ[ laste, newe ],
			If [ trace, Print[ newe ]; Print[ "which becomes" ] ];
			laste = newe;
			newe = laste /. postinvzrules ];
		If [ trace, Print[ newe ] ];
		newe ]

transform[ expr_ ] :=
	If [ SameQ[Head[expr], myinvz],
	     Replace [ expr, InvZTransformRules ],
	     expr ]

convert[ z_^k_ ( z_ + a_ )^j_, z_ ] := ( 1 + a z^-1 )^j  /; j == -k
convert[ z_ / ( z_ + a_ ), z_ ] := 1 / ( 1 + a z^-1 )
convert[ x_, z_ ] := x

postinvzrules = {
	shift[t_, n_, n0_] :> ( t /. n -> n + n0 ),

	(*  These two rules are needed to support a series	*)
	(*  expansion which returns a constant term plus	*)
	(*  terms of the form  approx z^r  .			*)
	myinvz[ b_, z_, n_, rm_, rp_, s_, op_ ] :>
		b Impulse[n] /;
		FreeQ[b, z],		   (* Converges all ROC *)

	myinvz[ a_. z_^r_., z_, n_, rm_, rp_, s_, op_ ] :>
		a Impulse[n + r] /;
		FreeQ[{a,r}, z],	   (* Converges all ROC *)

	(*  Attempt a series expansion about z = 0		*)
	(*  This is the strategy when all else has failed.	*)
	(*  The user can disable this rule (Terms -> False).	*)
	(*  Please keep this as the last rule.			*)
	myinvz[x_, z_, n_, rm_, rp_, s_, op_] :>
		SeriesStrategy[x, z, n, rm, rp, s, op ] /;
		GetStateField[s, seriesexpansionfield]
}

Format[ myinvz[ x_, z_, n_, rm_, rp_, s_, op_ ] ] := 
	SequenceForm[ ColumnForm[{"Z" Superscript[-1],
				  "  " ~StringJoin~ ToString[z]}],
		      { x } ]
Format[ shift[t_, n_, n0_] ] := SequenceForm[ {t}, Subscript[n -> n + n0] ]


(*  Most of the rest of the file is the list of inverse z-transform rules  *)

InvZTransformRules = {


  (*  I.  R A T I O N A L     T R A N S F O R M S			*)


  (*      A.  Simple transform pairs				*)
  (*	      1.  Constant (handles b=0 case as well)		*)
  myinvz[ b_, z_, n_, rm_, rp_, s_, op_ ] :>
	b Impulse[n] /;
	FreeQ[b, z],					(* Converges all ROC *)

  (*	          Redundant rule --  placed here for speed	*)
  myinvz[ a_. z_^r_., z_, n_, rm_, rp_, s_, op_ ] :>
	a Impulse[n + r] /;
	FreeQ[{a,r}, z],				(* Converges all ROC *)

  (*	      2.  Discrete-time sine functions			*)
  myinvz[ 1 / (1 - 2 Cos[w0_] z_^-1 + z_^-2), z_, n_, rm_, rp_, s_, op_ ] :>
	LeftOrRightSided[1, Sin[w0 + w0 n] Step[n] / Sin[w0], n, rm, rp] /;
	FreeQ[w0, z],

  myinvz[ 1 / (1 + c_. z_^-1 + z_^-2), z_, n_, rm_, rp_, s_, op_] :>
	Block [	{sinw0, w0},
		w0 = ArcCos[- c / 2];
		sinw0 = Sqrt[1 - c^2/4];
		LeftOrRightSided[ 1, Sin[w0 n + w0] Step[n] / sinw0,
				  n, rm, rp ] ] /;
	N[-2 < c < 2],

  (*	      3.  Discrete-time cosine functions		*)

  myinvz[ (1 - Cos[w0_] z_^-1) / (1 - 2 Cos[w0_] z_^-1 + z_^-2), z_, n_, rm_, rp_, s_, op_ ] :>
	LeftOrRightSided[1, Cos[w0 n] Step[n], n, rm, rp ] /;
	FreeQ[w0, z],

  (*	      4.  Discrete-time hyperbolic functions		*)

  myinvz[ (Sinh[b_] + Sinh[c_] z_^-1) / (1 - 2 Cosh[w_] z_^-1 + z_^-2), z_, n_, rm_, rp_, s_, op_ ] :>
	LeftOrRightSided[1, Sinh[b + w n] Step[n], n, rm, rp] /;
	FreeQ[{w,b,c},z] && ( c == w - b ),

  myinvz[ (Cosh[b_] + Cosh[c_] z_^-1) / (1 - 2 Cosh[w_] z_^-1 + z_^-2), z_, n_, rm_, rp_, s_, op_] :>
	LeftOrRightSided[1, Cosh[b + w n] Step[n], n, rm, rp] /;
	FreeQ[{w,b,c},z] && ((c == w - b) || (c == b - w)),


  (*      B.  First order sections					*)
  (*	      most covered by multiplication-by-exponential property	*)
  (*	      1/(1+az^-1) is more general than 1/(1-az^-1)		*)

  (*	      1.  Discrete-time pulse has no poles			*)
  myinvz[( 1 + b_. z_^L_ ) / ( 1 + a_. z_^-1 ), z_, n_, rm_, rp_, s_, op_] :>
	(-a)^n Pulse[-L, n] /;
	FreeQ[{a,b,L}, z] && Implies[ IntegerQ[L], N[L < 1] ] && ( Equal[a, b, 1] || SameQ[-b, (-a)^-L] ),

  (*	      2.  General exponential times a step (right-sided pair)	*)
  myinvz[1 / ( 1 + a_. z_^-1 ), z_, n_, rm_, rp_, s_, op_] :>
	LeftOrRightSided[a, (-a)^n Step[n], n, rm, rp] /;
	FreeQ[a,z],

  (*	      3.  General first-order section				*)
  myinvz[(a_. + b_. z_^-1) / (c_ + d_. z_^-1), z_, n_, rm_, rp_, s_, op_ ] :>
	LeftOrRightSided[d/c,
			 a/c (-d/c)^n Step[n] + b/c (-d/c)^(n-1) Step[n-1],
			 n, rm, rp] /;
	FreeQ[{a,b,c,d}, z],


  (*      C.  Second order sections					*)

  myinvz[z_^-1 / ( 1 + b_. z_^-1 + a_. z_^-2 ), z_, n_, rm_, rp_, s_, op_] :>
	Block [	{costheta, fun, rho, theta},
		rho = Sqrt[a];
		costheta = -b / ( 2 rho );
		theta = ArcCos[costheta];
		fun = ( 1 / ( rho Sin[theta] ) ) rho^n Sin[n theta] Step[n];
		LeftOrRightSided[Sqrt[a], fun, n, rm, rp] ] /;
	N[(a > 0) && (4 a > b^2)],

  myinvz[ (1 + b_. z_^-1) / (1 + c_. z_^-1 + a_. z_^-2), z_, n_, rm_, rp_, s_, op_] :>
	Block [	{costheta, fun, rho, theta},
		rho = Sqrt[a];
		costheta = - b / rho;
		theta = ArcCos[costheta];
		fun = rho^n Cos[n theta] Step[n];
		LeftOrRightSided[Sqrt[a], fun, n, rm, rp] ] /;
	N[ (c == 2 b) && (0 < a < b^2) ],

  myinvz[ 1 / ( 1 + b_. z_^-1 + a_. z_^-2 ), z_, n_, rm_, rp_, s_, op_] :>
	Block [ {costheta, rho, sintheta, theta},
		rho = Sqrt[a];
		costheta = - b / ( 2 rho );
		theta = ArcCos[costheta];
		sintheta = Sin[theta];
		fun = rho^n ( Cos[n theta] + Cot[theta] Sin[n theta] ) Step[n];
		LeftOrRightSided[Sqrt[a], fun, n, rm, rp] ] /;
	N[ (a > 0) && (4 a > b^2) ],

  myinvz[1 / ( 1 + a_. z^-2 ), z_, n_, rm_, rp_, s_, op_] :>
	LeftOrRightSided[ Sqrt[-a],
			  ( Sqrt[-a]^n + (- Sqrt[-a])^n ) Step[n] / 2,
			  n, rm, rp ] /;
	FreeQ[a,z] && Implies[ NumberQ[N[a]], N[a < 0] ],

  myinvz[1 / ( 1 + a_. z_^-2 ), z_, n_, rm_, rp_, s_, op_] :>
	LeftOrRightSided[ Sqrt[a],
			  Sqrt[a]^n Cos[n Pi / 2] Step[n],
			  n, rm, rp ] /;
	FreeQ[a,z] && Implies[ NumberQ[N[a]], N[a > 0] ],

  myinvz[1 / ( 1 + a_. z_^-1 )^2, z_, n_, rm_, rp_, s_, op_] :>
	LeftOrRightSided[a, (n + 1) (-a)^n Step[n], n, rm, rp] /;
	FreeQ[a,z],


  (*      D.  Higher order sections					*)
  (*	      Note that k can be real-valued.				*)

  (*	      1.  General denominator section to an integer power	*)
  myinvz[c_. (1 + b_. z_^-1)^k_, z_, n_, rm_, rp_, s_, op_] :>
	LeftOrRightSided[b, c Binomial[n-k-1,-k-1] (-b)^n Step[n], n, rm, rp] /;
	FreeQ[{b,c,k}, z] && ( k < -1 ),

  (*	      2.  General denominator section to a symbolic power	*)
  (*		  Step is different because we don't know what k is.	*)
  myinvz[c_. / (1 + b_. z_^-1)^k_., z_, n_, rm_, rp_, s_, op_] :>
	LeftOrRightSided[b, c Binomial[n+k-1,k-1] (-b)^n Step[n+k-1],
			 n, rm, rp] /;
	FreeQ[{b,c,k}, z],

  (*	      3.  General numerator section to a symbolic power		*)
  myinvz[c_. (a_ + b_. z_^-1)^k_, z_, n_, rm_, rp_, s_, op_] :>
	c LeftOrRightSided[ -b/a, Binomial[k,n] b^n a^(k-n) Step[n],
			    n, rm, rp ] /;
	FreeQ[{a,b,c,k}, z] && Implies[NumberQ[k], N[k > 0]],

  (*	      4.  Reduce expressions in z^k to expressions in z.	*)
  (*		  This applies to expressions other than polys in z.	*)
  myinvz[f_, z_, n_, rm_, rp_, s_, op_] :>
	Block [	{newf, upindex},
		upindex = UpsampleFactor[f, z];
		Off[Replace::rep];		 (* this replacement is legal *)
		newf = f /. z -> z^(1/upindex);	 (* yet Mathematica complains *)
		On[Replace::rep];		 (* so disable Replace::rep   *)
		Upsample[upindex, n]
			[myinvz[newf, z, n, rm, rp, s, op]] ] /;
	ZUpsampledQ[f, z],


  (*	  E.  Factor rational polynomials and make the denominator	*)
  (*		only contain negative powers of z.			*)

  (*	      1.  Factor rational polynomials, denominators having	*)
  (*		  positive powers of z (ensured by PolynomialQ test).	*)
  myinvz[p_, z_, n_, rm_, rp_, s_, op_] :>
	myinvz [ ( Expand[Numerator[p]] ) / ( Factor[Denominator[p]] ),
		 z, n, rm, rp, SetStateField[s, polyfactorfield, False], op ] /;
	GetStateField[s, polyfactorfield] && PolynomialQ[Denominator[p], z] && RationalFunctionQ[p, z],


  (*	      2.  Make factored denominators have neg. powers of z	*)
  (*		  a.  For numeric k, numeric j				*)
  myinvz[((a_ + b_. z_^j_.)^k_) x_., z_, n_, rm_, rp_, s_, op_] :>
	b^k myinvz[ Distribute[Numerator[x] z^(j k)] ( 1 + a z^(-j) / b )^k /
		    Denominator[x], z, n, rm, rp, s, op ] /;
	FreeQ[{a,b}, z] && IntegerQ[k] && (k < 0) && IntegerQ[j] && (j > 0),

  (*		  b.  For symbolic k, numeric j				*)
  myinvz[x_. / ((a_ + b_. z_^j_.)^k_), z_, n_, rm_, rp_, s_, op_] :>
	b^-k myinvz[ Distribute[Numerator[x] z^-(j k)] / (1 + a z^(-j) / b)^k /
		     Denominator[x], z, n, rm, rp, s, op ] /;
	FreeQ[{a,b,k}, z] && IntegerQ[j] && (j > 0),

  (*	      3.  Normalize factored denominators.			*)
  (*		  a.  For numeric k, numeric j				*)
  myinvz[((a_ + b_. z_^j_)^k_) x_., z_, n_, rm_, rp_, s_, op_] :>
	a^k myinvz[ x ( 1 + b z^j / a ) ^ k, z, n, rm, rp, s, op ] /;
	FreeQ[{a,b}, z] && IntegerQ[k] && (k < 0) && IntegerQ[j] && (j < 0) && ! SameQ[a, 1],

  (*		  b.  For symbolic k, numeric j				*)
  myinvz[x_. / ((a_ + b_. z_^j_)^k_), z_, n_, rm_, rp_, s_, op_] :>
	a^-k myinvz[ x / ( 1 + b z^j / a ) ^ k, z, n, rm, rp, s, op ] /;
	FreeQ[{a,b,k}, z] && IntegerQ[j] && (j < 0) && ! SameQ[a, 1],




  (*  II. N O N - R A T I O N A L     T R A N S F O R M S  *)


  (*      A.  Handle Exp[affine] forms; covers Sinh[affine]	*)
  (*          Cosh[affine] forms converges for any ROC		*) 
  myinvz[c_^(b_. + a_. z_^-1), z_, n_, rm_, rp_, s_, op_] :>
	LeftOrRightSided[ a Log[c], c^b a^n (Log[c])^n Step[n] / n!,
			  n, rm, rp ]  /;
	FreeQ[{a,b,c}, z],


  (*	  B.  Handle Log[affine] forms [O&S, 55]		*)
  myinvz[Log[a_ + b_. z_^-1], z_, n_, rm_, rp_, s_, op_] :>
	Log[a] Impulse[n] +
	  LeftOrRightSided[-b/a, -(-b/a)^n Step[n-1] / n, n, rm, rp] /;
	FreeQ[{a,b}, z],


  (*	  C.  Family of causal IIR filters			*)
  myinvz[( 1 + z_^-1 )^r_, z_, n_, rm_, rp_, s_, op_] :>
	Gamma[1 + r] Step[n] / ( Gamma[1 + n] Gamma[1 + r - n] ) /;
	FreeQ[r,z] && Implies[ NumberQ[N[r]], N[r > -1] ],


  (*	  D.  Other unusual forms.				*)
  myinvz[ArcTan[z_^-1], z_, n_, rm_, rp_, s_, op_] :>
	Sin[n Pi / 2] Step[n - 1] / n,

  myinvz[Sqrt[z_] ArcTan[Sqrt[z_^-1]], z_, n_, rm_, rp_, s_, op_] :>
	Step[n] / ( 2 n + 1 ),

  myinvz[Sqrt[z_] ArcTan[z_^-1], z_, n_, rm_, rp_, s_, op_] :>
	Step[n] / ( 2 n + 1 ),

  myinvz[Cosh[ Sqrt[z_^-1] ], z_, n_, rm_, rp_, s_, op_] :>
	Step[n] / ( 2 n ) !,

  myinvz[Sinh[ Sqrt[z_^-1] ], z_, n_, rm_, rp_, s_, op_] :>
	Step[n] / ( 2 n + 1 ) !,

  myinvz[Sqrt[ 1 / ( 1 + a_. z_^-1 ) ], z_, n_, rm_, rp_, s_, op_] :>
	(-a)^n (2 n)! Step[n] / ((2^n n!) ^ 2)  /;
	FreeQ[a, z],


  (*	  E.  Multiplication by exponential sequence (exp)^n		*)
  (*	      Check this possibility out before partial fractions	*)
  (*	      I'm still not sure about how I'm adjusting the ROC:	*)
  (*	      we could instead do a left/right-sided check on exp	*)
  (*	      Moved from rational section because it interfered		*)
  (*	      with the Log lookup.					*)
  myinvz[f_, z_, n_, rm_, rp_, s_, op_] :>
	Block [	{exp, fun, newf, newrm, newrp, state},
		exp = 1 / ScalingFactor[f, z];
		newf = f /. z -> z exp;
		newrm = Abs[rm / exp];
		newrp = If [ InfinityQ[rp], Infinity, Abs[rp / exp] ];
		state = SetStateField[s, expcheckfield, False];
		(exp)^n myinvz[newf, z, n, newrm, newrp, state, op] ] /;
	GetStateField[s, expcheckfield] && MultByExponentialQ[f, z] && ! SameQ[Head[f], Plus],


  (*      F.  Partial fractions decomposition				*)

  (*      partial fractions decomposition for rational polys of z, p[z]	*)
  (*	  once partial fractions have been applied to an expression,	*)
  (*	  they are never applied to any of the resulting sub-		*)
  (*      expressions.  This is	controlled by the partial fractions	*)
  (*      semaphore/flag in the local state; at this point, the		*)
  (*      denominator should be a function of z^-1, but we will do	*)
  (*	  partial fractions on a denominator that is a "two-sided"	*)
  (*	  polynomial (one in both positive and negative powers of z).	*)
  myinvz[p_, z_, n_, rm_, rp_, s_, op_] :>
	Block [ {partfrac, state},
		partfrac = KeepNormalized[p, Apart, z, twoargcall];
		state = SetStateField[s, partialfractionsfield, False];
		state = SetStateField[state, partialfractionsfield2, False];
		If [ ! SameQ[partfrac, p],
		     PartialFractionsDialogue[p, partfrac, op] ];
		myinvz [ partfrac, z, n, rm, rp, state, op ] ] /;
	GetStateField[s, partialfractionsfield] && PartialFractionsQ[p, z],




  (* III. Z - T R A N S F O R M     P R O P E R T I E S			*)

  (*  Omitted these properties:	1.  multiplication by n^m		*)
  (*				2.  multiplication by Cos[bn]		*)
  (*				3.  multiplication by Sin[bn]		*)
  (*				4.  multiplication by x y		*)
  (*									*)
  (*  This section is similar to the z-transform Properties section	*)
  (*  of the forward z-transform rules.					*)


  (*	  A.  Homogeneity --  pick off constants			*)
  myinvz[c_ x_, z_, n_, rm_, rp_, s_, op_] :>
	c myinvz[x, z, n, rm, rp, s, op] /;
	FreeQ[c,z],


  (*	  B.  Additivity						*)
  myinvz[x_+y_, z_, n_, rm_, rp_, s_, op_] :>
	myinvz[x, z, n, rm, rp, s, op] + myinvz[y, z, n, rm, rp, s, op],

  myinvz[(x_+y_)/c_, z_, n_, rm_, rp_, s_, op_] :>
	myinvz[x/c, z, n, rm, rp, s, op] + myinvz[y/c, z, n, rm, rp, s, op],


  (*	  C.  Delay or shift						*)
  myinvz[x_ z_^m_., z_, n_, rm_, rp_, s_, op_] :>
	shift[ myinvz[x, z, n, rm, rp, s, op], n, m ] /;
	FreeQ[m,z] && Implies[ NumberQ[m], IntegerQ[m] ],


  (*	  D.  Try partial fractions without maintaining normalization 	*)
  myinvz[p_, z_, n_, rm_, rp_, s_, op_] :>
	Block [ {partfrac, state},
		partfrac = Apart[p, z];
		state = SetStateField[s, partialfractionsfield2, False];
		state = SetStateField[state, partialfractionsfield, False];
		If [ ! SameQ[partfrac, p],
		     PartialFractionsDialogue[p, partfrac, op] ];
		myinvz [ partfrac, z, n, rm, rp, state, op ] ] /;
	GetStateField[s, partialfractionsfield2],


  (*	  E.  Logarithm properties 					*)
  myinvz[Log[a_. / b_], z_, n_, rm_, rp_, s_, op_] :>
	myinvz[Log[a] - Log[b], z, n, rm, rp, s, op],

  myinvz[Log[a_ b_], z_, n_, rm_, rp_, s_, op_] :>
	myinvz[Log[a] + Log[b], z, n, rm, rp, s, op],


  (*	  G.  Multiplication by exponential sequence (exp)^n		*)
  (*	      I'm still not sure about how I'm adjusting the ROC:	*)
  (*	      we could instead do a left/right-sided check on exp	*)
  myinvz[f_, z_, n_, rm_, rp_, s_, op_] :>
	Block [	{exp, fun, newf, newrm, newrp},
		exp = 1 / ScalingFactor[f, z];
		newf = f /. z -> z exp;
		newrm = Abs[rm / exp];
		newrp = If [ InfinityQ[rp], Infinity, Abs[rp / exp] ];
		fun = (exp)^n myinvz[ newf, z, n, newrm, newrp, s, op ];
		LeftOrRightSided[exp, fun, n, rm, rp] ] /;
	MultByExponentialQ[f, z],




  (* IV.  S I G N A L     P R O C E S S I N G     S T R U C T U R E S	*)


  (*	  A.  Difference equations					*)
  myinvz[x_ (1 - z_^-1)^m_., z_, n_, rm_, rp_, s_, op_] :>
	Difference[m,n][ myinvz[x, z, n, rm, rp, s, op] ]  /;
	FreeQ[m,n] && Implies[NumberQ[m], IntegerQ[m] && m > 0],


  (*	  B.  Summation							*)
  myinvz[x_ / ( 1 - z_^-1 ), z_, n_, rm_, rp_, s_, op_] :>
	Block [	{context, index},
		context = $Context;
		$Context = "Global`";
		index = Unique["index"];
		$Context = context;
		Summation[index,0,n,1][ myinvz[x, z, index, rm, rp, s, op] ] ],


  (*	  C.  Periodic sequence with period k.				*)
  myinvz[x_ / ( 1 - z_^k_), z_, n_, rm_, rp_, s_, op_] :>
	Periodic[-k,n][ myinvz[x, z, n, rm, rp, s, op] ]  /;
	FreeQ[k,n] && Implies[ NumberQ[k], IntegerQ[k] && ( k < 0 ) ],




  (*  V.  Z - T R A N S F O R M    S T R A T E G I E S			*)

  (*        If the following three rules are considered, then the	*)
  (*  the current expression could not be inverted by table lookup	*)
  (*  and applying properties.  So, the following strategy is used:	*)

  (*      A.  Factor terms inside of a log or exponential function.	*)
  (*      B.  Normalize the denominator.				*)
  (*      C.  Expand every term in the z-transform to its simplest	*)
  (*	      components and try again.					*)
  (*      D.  Substitute 1/z for z, take the inverse z-transform, and	*)
  (*	      substitute -n for n in the result.			*)
  (*	  E.  Complex cepstrum.						*)
  (*	  F.  If the current expression is a rational polynomial,	*)
  (*	      then return an ARMA signal processing expression since	*)
  (*	      it could not be inverted using partial fractions.		*)
  (*      G.  Perform a Laurent series expansion using the first	*)
  (*	      Terms terms (Terms is an option) and try again.		*)


  (*      A.  Factor terms inside of a logarithmic or exponential expr.	*)
  myinvz[Log[x_], z_, n_, rm_, rp_, s_, op_] :>
	myinvz [ Log[Factor[x]], z, n,
		 rm, rp, SetStateField[s, logformfield, False], op ] /;
	GetStateField[s, logformfield] && PolynomialQ[x, z],

  myinvz[x_. b_^(exp_), z_, n_, rm_, rp_, s_, op_] :>
	myinvz [ x b^(Factor[exp]), z, n,
		 rm, rp, SetStateField[s, expformfield, False], op ] /;
	GetStateField[s, expformfield] && FreeQ[b, z],


  (*	  B.  Normalize denominator if rational poly.			*)
  (*	      1.  Make denominator have only negative powers of z	*)
  myinvz[x_, z_, n_, rm_, rp_, s_, op_] :>
	Block [	{maxdenexp, maxnumexp, zdenom, znumer},
		zdenom = Expand[Denominator[x]];
		maxdenexp = Exponent[zdenom, z];
		znumer = Expand[Numerator[x]];
		znumer = Distribute[z^(- maxdenexp) znumer];
		zdenom = Distribute[z^(- maxdenexp) zdenom];
		maxnumexp = Exponent[znumer, z];
		If [ ! SameQ[maxnumexp, 0],
		     znumer = Distribute[z^(- maxnumexp) znumer] ];
		shift [ myinvz[ znumer / zdenom, z, n, rm, rp, s, op],
			n, maxnumexp ] ] /;
	RationalFunctionQ[x, z] && PolynomialQ[Denominator[x], z],

  (*	      2.  Make the constant term in denominator be unity	*)
  myinvz[x_, z_, n_, rm_, rp_, s_, op_] :>
	Block [	{adjdenom, const, denom},
		const = ConstantTerm[x, z];
		denom = Denominator[Expand[x]];
		adjdenom = Map[ (# / const)&, denom];
		1/const myinvz[Numerator[x]/adjdenom, z, n, rm, rp, s, op]] /;
	RationalFunctionQ[x, z] && PolynomialQ[Denominator[x] /. z -> z^-1, z] && ! NormalizedQ[Denominator[x], z],


  (*      C.  Expand definitions of all expressions to simplest form	*)
  myinvz[x_, z_, n_, rm_, rp_, s_, op_] :>
	myinvz [ ExpandAll[x], z, n, rm, rp,
		 SetStateField[s, expandallfield, False], op ] /;
	GetStateField[s, expandallfield],


  (*      D.  Substitute z^-1 for z, take the inverse z-transform,	*)
  (*	      and substitute -n for n in the result.			*)
  myinvz[x_, z_, n_, rm_, rp_, s_, op_] :>
	Block [	{newrm, newrp, newx, seq},
		newx = x /. z -> z^-1;
		newrp = If [ ZeroQ[rm], Infinity, 1/rm ];
		newrm = If [ InfinityQ[rp], 0, 1/rp ];

		seq = MyInvZTransform [ newx, z, n, newrm, newrp,
					anticausalInvZstate[], op ];
		If [ InvalidInvZTransformQ[seq],
		     myinvz [ x, z, n, rm, rp,
			      SetStateField[s, anticausalfield, False], op ],
		     seq /. n -> -n ] ] /;
	GetStateField[s, anticausalfield],


  (*	  E.  Complex cepstrum [O&S, 498]				*)
  myinvz[Log[x_], z_, n_, rm_, rp_, s_, op_] :>
	-1/n myinvz [ z D[x, z] / x, z, n, rm, rp,
		      SetStateField[s, cepstrumfield, False], op ] /;
	GetStateField[cepstrumfield, False] && Count[D[x, z], Derivative[1][f_][z], Infinity] == 0,


  (*	  F.  Invert rational polynomial as a cascade of an		*)
  (*	      FIR filter and an IIR filter.				*)
  myinvz[x_, z_, n_, rm_, rp_, s_, op_] :>
	Block [	{firlist, iirlist, maxexp, zdenom, znumer},
		firlist = CoefficientList[znumer /. z -> z^-1, z];
		iirlist = CoefficientList[zdenom /. z -> z^-1, z];
		If [ SameQ[znumer, 1],
		     Shift[ maxexp, n ]
			  [ IIR[n, iirlist] ],
		     Shift[ maxexp, n ]
			  [ FIR[n, firlist]
			    [ IIR[n, iirlist] ] ] ] ] /;
	RationalFunctionQ[x, z] && RationalPolynomialQ[x, z]

}


(*  E N D     P A C K A G E  *)

End[]
EndPackage[]

If [ TrueQ[ $VersionNumber >= 2.0 ],
     On[ General::spell ];
     On[ General::spell1 ] ];


(*  H E L P     I N F O R M A T I O N  *)

Combine[ SPfunctions, {InvZTransform} ]
Protect[ InvZTransform ]


(*  E N D I N G     M E S S A G E  *)

Print["The inverse z-transform rules have been loaded."]
Null
