#include <OI/oi.H>



/*
 *	translations - using a C language function as callbacks.
 *
 *	This program demonstrates the use of the translation mechanism
 *	to add callbacks to Mouse events.  This program should be
 *	executed as:
 *		./bargraph -config trans.ad
 *	Otherwise the appropriate mouse actions will not be set.
 *
 *	The reader should refer back to the README file for
 *	other translations examples.
 *
 *	The reader should refer to the OI documentation for
 *	information on the following member functions.
 *		- OIIntro
 *			OI_init()
 *			OI_begin_interaction()
 *			OI_fini()
 *			OI_add_actions()
 *		- OI_d_tech
 *			app_window()
 *			bdr_pixel()
 *			bkg_pixel()
 *			column_position()
 *			del()
 *			descendant()
 *			display()
 *			fg_pixel()
 *			in_column()
 *			is_derived_from()
 *			layout_associated_object()
 *			loc_x()
 *			loc_y()
 *			override_translations()
 *			row_position()
 *			set_associated_object()
 *			set_gravity()
 *			set_layout()
 *			set_loc()
 *			set_pixels()
 *			set_size()
 *			set_state()
 *			size_x()
 *			size_y()
 *		- OI_app_window
 *			oi_create_app_window()
 */

#define	NUM_BARS	(20)


/*
 *	The OI_actions_rec structure allows the user to specify
 *	either C-functions, or C++ member functions, to be called
 *	when the specified mouse event occurs.
 */
static void pick(OI_d_tech*, const XEvent*, const char **, unsigned int*);
static void drag(OI_d_tech*, const XEvent*, const char **, unsigned int*);
static void addto(OI_d_tech*, const XEvent*, const char **, unsigned int*);


static OI_actions_rec myactions[] = {
{"pick", (OI_translation_fn *)&pick},
{"drag", (OI_translation_fn *)&drag},
{"addto", (OI_translation_fn *)&addto},
};


/*
 *	The BarGraph_translations character array specify the translations
 *	to be applied to the BarGraph box bars.
 *	Please note that these translations could also be written out to
 *	a config file, and added to the application at runtime using
 *		app -config trans.cf
 *	Please see example transC.C for an example of how to use that
 *	method of applying translations to other objects.
 */
static	char *BarGraph_translations = "#override	\n\
		<Btn1Down>:	pick()			\n\
		<Btn1Motion>:	drag()			\n\
		<Btn1Up>:	addto()			\n\
	";



void
main (int argc, char **argv)
{
	int			i;			/* generic counter */
	char			lbl[ 32 ];		/* string for writing box names */
	OI_connection		*conp;			/* the connection to the server */
	OI_box			*box;			/* each of the bars */
	OI_app_window		*appWindow;		/* the enclosing app window */

	
	/*
	 *	Open a connection.
	 */
	if ((conp = OI_init(&argc, argv, "actions", "Actions"))) {
		
		/*
		 *	OI_add_actions makes the connection between the user specified
		 *	mouse events (in trans.ad), with the actual callbacks.
		 */
	        OI_add_actions(myactions, OI_count(myactions));

		/*
		 *	create application window.
		 */
		appWindow = oi_create_app_window("myapp", 1, 1, "C Translations");
		appWindow->set_layout( OI_layout_row );


		for (i=0; i< NUM_BARS; i++) {
			sprintf( lbl, "box_%d", i );
			box = oi_create_box( lbl, 10, 10*(i+1) );
			box->set_gravity( OI_grav_south );
			box->layout_associated_object( appWindow, i, 0, OI_ACTIVE );
			box->override_translations( BarGraph_translations );
		}

		/*
		 *	Display application window.
		 *	Begin interaction.
		 */
		appWindow->set_associated_object(conp->root(), OI_DEF_LOC, OI_DEF_LOC, OI_ACTIVE);
		OI_begin_interaction();
	}

	/*
	 *	Cleanup.  Make sure that we cleanup the library.
	 */
	OI_fini();
}



/*
 *	global structure to hold the draggable box, offsets, and toggle flag
 */
static struct {
	OI_box	*box;
	int	deltaX;
	int	deltaY;
	OI_bool	toggle;
} dr = {
	NULL,
	0, 0,
	OI_NO
};

/*
 *	on ButtonPress
 *		- initialize deltaX and deltaY
 *		- if this was a ButtonPress Event,
 *			capture the actual event location and store as deltaX and deltaY.
 *		- show the dragbox over the source object
 */
static void
pick(OI_d_tech *objp, const XEvent *eventp, const char **, unsigned int *)
{
	if (!dr.box) {
		dr.box = oi_create_box( "dragBox", 1, 1 );
		dr.box->set_pixels(
			dr.box->fg_pixel(), dr.box->bkg_pixel(), dr.box->bdr_pixel() );
		dr.box->set_associated_object( objp->app_window(), 0, 0, OI_NOT_DISPLAYED );
	}

	dr.toggle = OI_NO ;
	dr.deltaX = dr.deltaY = 0;
	if (eventp->type == ButtonPress) {
		dr.deltaX = eventp->xbutton.x;
		dr.deltaY = eventp->xbutton.y;
	}
	
	/*
	 *	You MUST remember here that the x,y is relative to the object
	 */
	dr.box->set_size( objp->size_x(), dr.deltaY ? dr.deltaY : 1 );
	dr.box->set_loc( objp->loc_x(), objp->loc_y() );
	dr.box->set_state( OI_ACTIVE );
}

/*
 *	on MotionNotify (drag event)
 *		- capture the event that caused this procedure to be called.
 *		- process all MotionNotify events in the queue (less choppy movement)
 *		- change the dragBox location from the old location to the new location
 *		- toggle the source box color to show that we are dragging from it.
 */
static void
drag(OI_d_tech *objp, const XEvent *eventp, const char **, unsigned int *)
{
	XEvent event;
	if (!dr.toggle) {
		objp->set_pixels( objp->fg_pixel(), objp->bkg_pixel(), objp->bdr_pixel() );
		dr.toggle = OI_YES;
	}
	if (eventp->type == MotionNotify) {
		event = *eventp;
		while (XCheckTypedEvent(objp->display(), MotionNotify, &event));
		dr.box->set_loc(
			event.xmotion.x + objp->loc_x() - dr.deltaX,
			event.xmotion.y + objp->loc_y() - dr.deltaY);
	}
}

/*
 *	on ButtonRelease
 *	refresh the object color.
 */
static void
addto(OI_d_tech *objp, const XEvent *ev, const char **, unsigned int *)
{
	int		i;
	long		X;
	int		height;
	OI_d_tech	*top;		/* top level window */
	OI_d_tech	*dst;		/* destination object */
	OI_d_tech	*box;		/* child box */
	char		lbl[ 32 ];	/* string for writing box names */

	if (dr.toggle)
		objp->set_pixels( objp->fg_pixel(), objp->bkg_pixel(), objp->bdr_pixel() );
	dr.box->set_state( OI_NOT_DISPLAYED );

	top = objp->app_window();
	if ((dst = OI_find_obj( ev, 1, &top )) && (dst!=objp)) {
		if (dst == top ) {
			X = ev->xbutton.x + objp->loc_x();
			for (i=0; i<NUM_BARS; i++) {
				sprintf( lbl, "box_%d", i );
				if (!(box = top->descendant( lbl )))
					continue ;
				if (!top->in_column( box->column_position(), X, box->row_position() ))
				{
					dst = box ;
					break ;
				}
			}
		}
		if (dst->is_derived_from( OI_box::clsp )) {
			dst->set_size( dst->size_x(), dst->size_y() + dr.box->size_y() );
			height = objp->size_y() - dr.box->size_y(); 
			if (height > 0)
				objp->set_size( objp->size_x(), height );
			else
				objp->del();
		}
	}
}
