/*$__copyright$ */
/*
 *	Shape/AtFS
 *
 *	afarchive.c -- read/write archives of attribute filesystem
 *
 *	Author: Andreas Lampen, TU-Berlin (andy@coma.UUCP)
 *					  (andy@db0tui62.BITNET)
 *
 *	$Header: afarchive.c[1.25] Wed Apr 22 17:42:39 1992 andy@cs.tu-berlin.de accessed $
 *
 *	EXPORT:
 *      afReadData -- read data from archive
 *	afReadAttrs -- read attributes from archive
 *	afWriteArchive -- write it
 */

#include <stdio.h>

#include "afsys.h"
#include "atfs.h"
#include "afarchive.h"

#ifdef MEMDEBUG
extern FILE *memprot;
#endif
extern int af_errno;
extern int nameConflict;

LOCAL short arVersion;

/*==========================================================================
 *	firstitem, nextitem -- isolate items in input line
 *==========================================================================*/

LOCAL char *firstitem (line)
     char *line;
{
  register char *sptr;

  /* skip leading blank */
  if ((sptr = index (&line[1], ' ')) == (char *)0)
    sptr = index (&line[1], '\n');

  *sptr = '\0';
  return (&line[1]);
}

LOCAL char *nextitem (line)
     char *line;
{
  register char *sptr, *finptr;

  sptr = &line[strlen(line)]+1; /* move to next entry */
  if ((finptr = index (sptr, ' ')) == (char *)0)
    finptr = index (sptr, '\n');
  *finptr = '\0';

  return (sptr);
}
  
/*======================= rddata ====================================*/

LOCAL af_rddata (file, dbuf, list)
     FILE   *file;
     char   *dbuf;
     Af_revlist *list;
{
  register char  *itemptr;
  char	         idstr[AF_IDSTRLEN+1], line[AF_LINESIZ];
  register int   i, maxindex;
  int            gen, rev;
  register off_t size, dpos = 0;

  /* if there is a valid busy version */
  if (list->af_list[0].af_class & AF_VALID)
    maxindex = list->af_nrevs;
  else
    maxindex = list->af_nrevs+1;

  idstr[AF_IDSTRLEN] = '\0';
  /* skip busy version */
  for (i=1; i < maxindex; i++)
    {
      (void) fgets (idstr, AF_IDSTRLEN+1, file);
      if (strcmp (idstr, AF_NOTEID))
	FAIL ("rddata", "wrong note-ID in datafile", AF_EINCONSIST, ERROR);
      (void) fgets (line, AF_LINESIZ, file);
      itemptr = firstitem (line);
      gen = atoi (itemptr);
      itemptr = nextitem (itemptr);
      rev = atoi (itemptr);
      itemptr = nextitem (itemptr);
      size = (off_t) atoi (itemptr);

      if ((list->af_list[i].af_gen != gen) || (list->af_list[i].af_rev != rev))
	FAIL ("rddata", "wrong version in datafile", AF_EINCONSIST, ERROR);

      /* read note */
      list->af_list[i].af_notesize = size;
      if (size > 0)
	{
	  (void) fread (&(dbuf[dpos]), sizeof(char), (Size_t) size, file);
	  list->af_list[i].af_note = &(dbuf[dpos]);
	  /* replace newline by nullbyte */
	  list->af_list[i].af_note[size-sizeof(char)] = '\0';
	  dpos = dpos+size;
	}
      else
	{
	  /* skip newline */
	  (void) fread (&(dbuf[dpos]), sizeof(char), (Size_t) 1, file);
	  list->af_list[i].af_note = (char *)0;
	}

      (void) fgets (idstr, AF_IDSTRLEN+1, file);
      if (strcmp (idstr, AF_DATAID))
	FAIL ("rddata", "wrong data-ID in datafile", AF_EINCONSIST, ERROR);
      (void) fgets (line, AF_LINESIZ, file);
      itemptr = firstitem (line);
      gen = atoi (itemptr);
      itemptr = nextitem (itemptr);
      rev = atoi (itemptr);
      itemptr = nextitem (itemptr);
      itemptr = nextitem (itemptr);
      size = (off_t) atoi (itemptr);

      if ((list->af_list[i].af_gen != gen) || (list->af_list[i].af_rev != rev))
	FAIL ("rddata", "wrong version in datafile", AF_EINCONSIST, ERROR);

      /* read data */
      if (size > 0)
	{
	  (void) fread (&(dbuf[dpos]), sizeof(char), (Size_t) size, file);
	  list->af_list[i].af_data = &(dbuf[dpos]);
	  dpos = dpos+size;
	  /* convert version 1 deltas */
	  if ((arVersion == 1) && (list->af_list[i].af_repr == AF_DELTA))
	    {
	      off_t newDeltaSize;
	      char *afConvertDelta();

	      list->af_list[i].af_data = afConvertDelta (list->af_list[i].af_data, list->af_list[i].af_dsize, &newDeltaSize);
	      list->af_datasize -= list->af_list[i].af_dsize;
	      list->af_datasize += newDeltaSize;
	      list->af_list[i].af_dsize = newDeltaSize;
	    }
	}
      else
	{
	  list->af_list[i].af_data = (char *)0;
	}
    }
  return (AF_OK);
} /* af_rddata */

/*==========================================================================
 *	afReadData -- read notes and data section of archive file
 *
 *
 *==========================================================================*/

EXPORT afReadData (list)
     Af_revlist *list;
{
  register char *data;
  char          idstr[AF_SEGSTRLEN+1], archname[MAXPATHLEN], line[AF_LINESIZ];
  register FILE *archfile;

  /* if data are already loaded */
  if ((list->af_extent & AF_DATA) == AF_DATA)
    return (AF_OK);

  (void) strcpy (archname, list->af_arfilename);
  archname[strlen(archname)-sizeof(char)] = AF_DATAEXT;
  if ((archfile = fopen (archname, "r")) == (FILE *)0)
    FAIL ("ReadData", "", AF_ENOATFSFILE, ERROR);

  idstr[AF_SEGSTRLEN] = '\0';
  (void) fgets (idstr, AF_SEGSTRLEN+1, archfile);
  if (strncmp (idstr, AF_DATAHEADER, AF_SEGSTRLEN))
    FAIL ("ReadData", "wrong header in datafile", AF_EINCONSIST, ERROR);

  (void) fgets (line, AF_LINESIZ, archfile);
  arVersion = atoi (line);
  if (arVersion > AF_ARCURVERS)
    FAIL ("ReadData", "unknown archive format version", AF_EINCONSIST, ERROR);

  /* allocate memory for data */
  if ((data = af_malloc (list, (unsigned) list->af_datasize)) == (char *)0)
    return (ERROR);

  if (af_rddata (archfile, data, list) != AF_OK)
    return (ERROR);

  list->af_extent |= AF_DATA;
  (void) fclose (archfile);
  return (AF_OK);
}

/*======================= rdattrs ====================================*/

LOCAL af_rdattrs (file, list, bibufptr)
     FILE	 *file;
     Af_revlist  *list;
     struct stat *bibufptr;
{
  register char *itemptr;
  char          idstr[AF_IDSTRLEN+1], line[AF_LINESIZ];
  register int  i;
  bool          writeok;
  register Af_user *user, *owner;
  Af_user          *af_garown();

  /* skip idstring */
  idstr[AF_IDSTRLEN] = '\0';
  (void) fgets (idstr, AF_IDSTRLEN+1, file);
  
  /* read constant attributes of busy version */
  (void) fgets (line, AF_LINESIZ, file);

  itemptr = firstitem (line);   /* host - ignored ... */
  itemptr = nextitem (itemptr); /* path - ignored ... */

  /* test if name and type are correct */
  /* On System V machines a mismatch points to a name conflict */
  itemptr = nextitem (itemptr); /* name */
  if (strcmp (list->af_list[0].af_name, itemptr))
    {
      if (strcmp (itemptr, AF_NOSTRING) || list->af_list[0].af_name)
	{
	  nameConflict = TRUE;
	  FAIL ("ReadAttrs", "", AF_ENAMETOOLONG, ERROR);
	}
    }
  itemptr = nextitem (itemptr); /* type */
  if (strcmp (NOTMT(list->af_list[0].af_type), itemptr))
    {
      if (strcmp (itemptr, AF_NOSTRING) || list->af_list[0].af_type)
	{
	  nameConflict = TRUE;
	  FAIL ("ReadAttrs", "", AF_ENAMETOOLONG, ERROR);
	}
    }

  itemptr = nextitem (itemptr); /* variant (ignored) */

  /* get owner of archive directory */
  if ((owner = af_garown (list->af_arfilename, &writeok, &list->af_owngid)) == (Af_user *)0)
    FAIL ("rdattrs", "cannot get owner of archive", AF_EINTERNAL, ERROR);
  list->af_cattrs.af_ownname = af_entersym (owner->af_username);
  list->af_cattrs.af_ownhost = af_enterhost (owner->af_userhost);
  list->af_cattrs.af_owndomain = af_enterdomain (owner->af_userdomain);

  if (writeok)
    list->af_extent |= AF_UXWRITE;
  
  /* read owner from archive */
  (void) fgets (idstr, AF_IDSTRLEN+1, file);
  (void) fgets (line, AF_LINESIZ, file); /* name, host and domain of owner */
  /* plausibility of owner should be checked here */

  (void) fgets (idstr, AF_IDSTRLEN+1, file);
  (void) fgets (line, AF_LINESIZ, file); /* predecessor of busy object */
  itemptr = firstitem (line);
  list->af_list[0].af_predgen = atoi (itemptr);
  itemptr = nextitem (itemptr);
  list->af_list[0].af_predrev = atoi (itemptr);

  (void) fgets (idstr, AF_IDSTRLEN+1, file);
  (void) fgets (line, AF_LINESIZ, file); /* locker */
  itemptr = firstitem (line);
  if (strcmp (itemptr, AF_NOSTRING))
    {
      list->af_list[0].af_lckname = af_entersym (itemptr);
      itemptr = nextitem (itemptr);
      list->af_list[0].af_lckhost = af_enterhost (itemptr);
      if (arVersion > 2)
	{
	  itemptr = nextitem (itemptr);
	  list->af_list[0].af_lckdomain = af_enterdomain (itemptr);
	}
      else
	list->af_list[0].af_lckdomain = af_getdomain ();
    }
  else
    {
      list->af_list[0].af_lckname = (char *)0;
      itemptr = nextitem (itemptr);
      list->af_list[0].af_lckhost = (char *)0;
      if (arVersion > 2)
	itemptr = nextitem (itemptr);
      list->af_list[0].af_lckdomain = (char *)0;
    }
  itemptr = nextitem (itemptr);
  list->af_list[0].af_ltime = (time_t)atoi(itemptr);

  /* initialize and attributes for busy version */
  list->af_list[0].af_gen = AF_BUSYVERS;
  list->af_list[0].af_rev = AF_BUSYVERS;
  list->af_list[0].af_state = AF_BUSY;
  list->af_list[0].af_stime = AF_NOTIME;
  list->af_list[0].af_repr = AF_FILE;
  list->af_list[0].af_dsize = (off_t) 0;
  list->af_list[0].af_data = (char *)0;
  list->af_list[0].af_hashname = (char *)0;
  list->af_list[0].af_nrefs = 0;
  list->af_list[0].af_succgen = AF_NOVNUM;
  list->af_list[0].af_succrev = AF_NOVNUM;

  if (bibufptr->st_ino) /* if there is a busy version */
    {
      list->af_list[0].af_class = AF_VALID;
      if ((user = af_afuser (bibufptr->st_uid)) == (Af_user *)0)
	{
	  af_wng ("rdattrs", "invalid userID in inode of busy file");
	  user = af_afuser ((Uid_t) geteuid());
	}
      list->af_list[0].af_auname = af_entersym (user->af_username);
      list->af_list[0].af_auhost = af_enterhost (user->af_userhost);
      list->af_list[0].af_audomain = af_enterdomain (user->af_userdomain);
      list->af_list[0].af_mode = (u_short) bibufptr->st_mode;
      list->af_list[0].af_mtime = bibufptr->st_mtime;
      list->af_list[0].af_atime = bibufptr->st_atime;
      list->af_list[0].af_ctime = bibufptr->st_ctime;
      list->af_list[0].af_fsize = (off_t) bibufptr->st_size;
    }
  else
    {
      list->af_list[0].af_class = 0;
      list->af_list[0].af_auname = (char *)0;
      list->af_list[0].af_auhost = (char *)0;
      list->af_list[0].af_audomain = (char *)0;
      list->af_list[0].af_mode = AF_NOMODE;
      list->af_list[0].af_mtime = AF_NOTIME;
      list->af_list[0].af_atime = AF_NOTIME;
      list->af_list[0].af_ctime = AF_NOTIME;
      list->af_list[0].af_fsize = 0;
    }
      
  /* read list */
  for (i=1; i < list->af_nrevs; i++)
    {
      /* do initializations */
      list->af_list[i].af_class = AF_VALID;
      list->af_list[i].af_notesize = 0;
      list->af_list[i].af_note = (char *)0;
      list->af_list[i].af_data = (char *)0;
      list->af_list[0].af_nrefs = 0;
      list->af_list[i].af_hashname = (char *)0;

      /* enter name (set a pointer to the name-field of af_list[0]) */
      /* skip position 0 */
      if (i != 0)
	{
	  list->af_list[i].af_name = list->af_list[0].af_name;
	  list->af_list[i].af_type = list->af_list[0].af_type;
	}

      /* read revision ID */
      (void) fgets (idstr, AF_IDSTRLEN+1, file);
      if (strcmp (idstr, AF_REVID)) /* could be done for every field */
	FAIL ("rdattrs", 
	      "wrong revision-ID in archive file", AF_EINCONSIST, ERROR);
      (void) fgets (line, AF_LINESIZ, file);
      itemptr = firstitem (line);
      list->af_list[i].af_gen = atoi (itemptr);
      itemptr = nextitem (itemptr);
      list->af_list[i].af_rev = atoi (itemptr);
      itemptr = nextitem (itemptr);
      list->af_list[i].af_state = (short) atoi (itemptr);
      itemptr = nextitem (itemptr);
      (void) sscanf (itemptr, "%ho", &(list->af_list[i].af_mode));
      itemptr = nextitem (itemptr); /* variant (ignored) */

      /* read author*/
      (void) fgetc (file); /* skip tab */
      (void) fgets (idstr, AF_IDSTRLEN+1, file);
      (void) fgets (line, AF_LINESIZ, file); /* predecessor of busy object */
      itemptr = firstitem (line);
      list->af_list[i].af_auname = af_entersym (itemptr);
      itemptr = nextitem (itemptr);
      list->af_list[i].af_auhost = af_enterhost (itemptr);
      if (arVersion > 2)
	{
	  itemptr = nextitem (itemptr);
	  list->af_list[i].af_audomain = af_enterdomain (itemptr);
	}
      else
	list->af_list[i].af_audomain = af_getdomain ();
      itemptr = nextitem (itemptr);
      if (strcmp (itemptr, AF_NOSTRING))
      	{
	  list->af_list[i].af_lckname = af_entersym (itemptr);
	  itemptr = nextitem (itemptr);
	  list->af_list[i].af_lckhost = af_enterhost (itemptr);
	  if (arVersion > 2)
	    {
	      itemptr = nextitem (itemptr);
	      list->af_list[i].af_lckdomain = af_enterdomain (itemptr);
	    }
	  else
	    list->af_list[i].af_lckdomain = af_getdomain ();
	}
      else
	{
	  list->af_list[i].af_lckname = (char *)0;
	  itemptr = nextitem (itemptr);
	  list->af_list[i].af_lckhost = (char *)0;
	  if (arVersion > 2)
	    itemptr = nextitem (itemptr);
	  list->af_list[i].af_lckdomain = (char *)0;
	}

      /* read dates */
      (void) fgetc (file); /* skip tab */
      (void) fgets (idstr, AF_IDSTRLEN+1, file);
      (void) fgets (line, AF_LINESIZ, file);
      itemptr = firstitem (line);
      list->af_list[i].af_mtime = (time_t) atoi (itemptr);
      itemptr = nextitem (itemptr);
      list->af_list[i].af_atime = (time_t) atoi (itemptr);
      itemptr = nextitem (itemptr);
      list->af_list[i].af_ctime = (time_t) atoi (itemptr);
      itemptr = nextitem (itemptr);
      list->af_list[i].af_stime = (time_t) atoi (itemptr);
      itemptr = nextitem (itemptr);
      list->af_list[i].af_ltime = (time_t) atoi (itemptr);

      /* read kind of representation */
      (void) fgetc (file); /* skip tab */
      (void) fgets (idstr, AF_IDSTRLEN+1, file);
      (void) fgets (line, AF_LINESIZ, file);
      itemptr = firstitem (line);
      list->af_list[i].af_repr = (short) atoi (itemptr);
      itemptr = nextitem (itemptr);
      list->af_list[i].af_fsize = (off_t) atoi (itemptr);
      itemptr = nextitem (itemptr);
      list->af_list[i].af_dsize = (off_t) atoi (itemptr);
      itemptr = nextitem (itemptr);
      list->af_list[i].af_succgen = atoi (itemptr);
      itemptr = nextitem (itemptr);
      list->af_list[i].af_succrev = atoi (itemptr);
      itemptr = nextitem (itemptr);
      list->af_list[i].af_predgen = atoi (itemptr);
      itemptr = nextitem (itemptr);
      list->af_list[i].af_predrev = atoi (itemptr);
    }
  /* move lock from busy version to last saved version,
   * when reading a version 2 (or 1) archive
   */
  i--;
  if ((arVersion <= 2) && (i>0))
    {
      if (list->af_list[0].af_lckname)
	{
	  list->af_list[i].af_lckname = list->af_list[0].af_lckname;
	  list->af_list[i].af_lckhost = list->af_list[0].af_lckhost;
	  list->af_list[i].af_lckdomain = list->af_list[0].af_lckdomain;
	  list->af_list[0].af_lckname = (char *)0;
	  list->af_list[0].af_lckhost = (char *)0;
	  list->af_list[0].af_lckdomain = (char *)0;
	}
    }

  if (!(bibufptr->st_ino)) /* if there is no busy version */
    list->af_nrevs--;
  
  return (AF_OK);
} /* af_rdattrs */

/*======================= rdudas ====================================*/

LOCAL af_rdudas (file, list)
     FILE       *file;
     Af_revlist *list;
{
  char	        idstr[AF_IDSTRLEN+1], line[AF_LINESIZ], *malloc(), *realloc();
  register char *udabuf = (char *)0;
#ifdef MEMDEBUG
  char *udaorig;
#endif
  char          *itemptr;
  register int	c, i, j, maxindex;
  int      	gen, rev;
  Af_key        tmpkey;

  tmpkey.af_ldes = list;

  /* if there is a valid busy version */
  if (list->af_list[0].af_class & AF_VALID)
    maxindex = list->af_nrevs;
  else
    maxindex = list->af_nrevs+1;

  (void) getc (file); /* skip newline */
  idstr[AF_IDSTRLEN] = '\0';
  for (i=0; i < maxindex; i++)
    {
      (void) fgets (idstr, AF_IDSTRLEN+1, file);
      if (strcmp (idstr, AF_UDAID))
	FAIL ("rdudas", "wrong uda-ID in archive file", AF_EINCONSIST, ERROR);
      (void) fgets (line, AF_LINESIZ, file);
      itemptr = firstitem (line);
      gen = atoi (itemptr);
      itemptr = nextitem (itemptr);
      rev = atoi (itemptr);

      if ((list->af_list[i].af_gen != gen) || (list->af_list[i].af_rev != rev))
	FAIL ("rdudas", "wrong version in archive file", AF_EINCONSIST, ERROR);
      
      /* build up hashlist and read user defined attributes */
      tmpkey.af_lpos = i;
      (void) afInitUdas (&tmpkey);
      
      if (i == 0) /* initalize only once */
	{
	  if ((udabuf = malloc ((unsigned) (AF_UDASEGSIZ * sizeof (char)))) == (char *)0)
	    FAIL ("rdudas", "malloc", AF_ESYSERR, ERROR);
#ifdef MEMDEBUG
/*	  fprintf (memprot, "%x(udabuf)-AL %d bytes\n", udabuf,
		   AF_UDASEGSIZ * sizeof (char));
	  udaorig = udabuf;
*/
#endif
	}

      /* if there is *no* valid busy version, skip the user defined */
      /* attributes for the busy version */
      if ((i==0) && !(list->af_list[0].af_class & AF_VALID))
	{
	  while (TRUE)
	    {
	      if ((c = getc (file)) == '\0')
		{
		  if ((c = getc (file)) == '\0')
		    break;
		}
	    }
	  (void) getc (file); /* skip trailing newline char */
	  continue;
	}

      j = 0;
      while (TRUE)
	{
	  if ((udabuf[j] = getc (file)) == '\0')
	    { 
	      if (j != 0)
		{
		  tmpkey.af_lpos = i;
		  (void) afEnterUda (&tmpkey, udabuf);
		}
	      /* a second nullbyte indicates the end of the list of udas */
	      if ((c = getc (file)) == '\0')
		break;
	      udabuf[0] = c;
	      j = 1;
	    }
	  else
	    {
	      j++;
	      if ((j % AF_UDASEGSIZ) == 0) /* if segment is full */
		{
		  if ((udabuf = realloc (udabuf, (unsigned) ((j + AF_UDASEGSIZ) * sizeof (char)))) == (char *)0) 
		    FAIL ("rdudas", "realloc", AF_ESYSERR, ERROR);
#ifdef MEMDEBUG
/*		  fprintf (memprot, "%x(udabuf)-RE %d bytes -> %x\n", udaorig,
			   (j + AF_UDASEGSIZ) * sizeof (char), udabuf);
*/
#endif
		}
	    }
	}
      (void) getc (file); /* skip trailing newline char */
    }
  free (udabuf);
#ifdef MEMDEBUG
/*  fprintf (memprot, "%x(udabuf)-FR (%x after realloc)\n", udaorig, udabuf);
*/
#endif
  return (AF_OK);
} /* af_rdudas */

/*==========================================================================
 *	afReadAttrs -- read attributes from archive file
 *                      
 *
 *==========================================================================*/

EXPORT afReadAttrs (list)
     Af_revlist *list;
{
  char	        idstr[AF_SEGSTRLEN+1], line[AF_LINESIZ];
  register char *itemptr;
  register FILE	*archfile;
  struct stat   bibuf, aibuf;
  bool          writeok;
  register Af_user *user, *owner;
  Af_user       *af_garown();
  Af_key        tmpkey;

  tmpkey.af_ldes = list;

  if (!list->af_arfilename)
    {
      list->af_arfilename =
	af_garname (list->af_cattrs.af_syspath,
		    list->af_cattrs.af_globname,
		    list->af_cattrs.af_globtype);
      list->af_extent |= AF_ARCHIVE;
    }
  
  /* if archive file has been modified */
  if (stat (list->af_arfilename, &aibuf) == -1)
    aibuf.st_mtime = (time_t) 0;
  if (list->af_lastmod != aibuf.st_mtime)
    {
      if (list->af_access > 0)
	{
	  af_wng ("ReadAttrs", "archive file has changed");
	  list->af_access = 0;
	}
      list->af_extent &= ~AF_SEGMASK; /* invalidate data */
    }

  /* if attributes are already loaded */
  if ((list->af_extent & AF_ATTRS) == AF_ATTRS)
    {
      /* see if busy version has changed */
      if (stat (list->af_busyfilename, &bibuf) == ERROR)
	{
	  /* no busy version */
	  if (list->af_list[0].af_class & AF_VALID)
	    {
	      list->af_nrevs--;
	      list->af_list[0].af_class &= ~AF_VALID;
	    }
	}
      else
	{
	  if (bibuf.st_ctime != list->af_list[0].af_ctime)
	    {
	      /* update busy version */
	      if (!(list->af_list[0].af_class & AF_VALID))
		{
		  list->af_nrevs++;
		  tmpkey.af_lpos = 0;
		  (void) afInitUdas (&tmpkey);
		  list->af_list[0].af_class = AF_VALID;
		}
	      if ((user = af_afuser (bibuf.st_uid)) == (Af_user *)0)
		{
		  af_wng ("ReadAttrs", "invalid userID in inode of busy file");
		  user = af_afuser ((Uid_t) geteuid());
		}
	      list->af_list[0].af_auname = af_entersym (user->af_username);
	      list->af_list[0].af_auhost = af_enterhost (user->af_userhost);
	      list->af_list[0].af_audomain = af_enterdomain (user->af_userdomain);
	      list->af_list[0].af_mode = (u_short) bibuf.st_mode;
	      list->af_list[0].af_mtime = bibuf.st_mtime;
	      list->af_list[0].af_atime = bibuf.st_atime;
	      list->af_list[0].af_ctime = bibuf.st_ctime;
	      list->af_list[0].af_fsize = (off_t) bibuf.st_size;
	    }
	}
      return (AF_OK);
    }

  if (!list->af_busyfilename)
    {
      if ((list->af_busyfilename =
	   af_gbusname (list->af_cattrs.af_syspath,
			list->af_cattrs.af_globname,
			list->af_cattrs.af_globtype)) == (char *)0)
	return (ERROR);
    }

  if (stat (list->af_busyfilename, &bibuf) == ERROR)
    bibuf.st_ino = 0;

  /* open archive */
  if ((list->af_arfilename == (char *)0) ||
      ((archfile = fopen (list->af_arfilename, "r")) == (FILE *)0))
    {
      if (bibuf.st_ino == 0) /* no busy file */
	{
	  list->af_nrevs = 0;
	  (void) afDetachList (list);
	  SFAIL ("ReadAttrs", "", AF_ENOATFSFILE, ERROR);
	}
      list->af_nrevs = 1;
      list->af_listlen = AF_NEWREVS;
      list->af_extent |= AF_SEGMASK;
      list->af_datasize = 0;

      /* determine author of busy file */
      if ((user = af_afuser (bibuf.st_uid)) == (Af_user *)0)
	{
	  af_wng ("ReadAttrs", "invalid userID in inode of busy file");
	  user = af_afuser ((Uid_t)geteuid());
	}

      /* if an archive-directory exists, get its owner */
      if ((owner = af_garown (list->af_arfilename, &writeok, &list->af_owngid)) == (Af_user *)0)
	{
	  list->af_cattrs.af_ownname = af_entersym (user->af_username);
	  list->af_cattrs.af_ownhost = af_enterhost (user->af_userhost);
	  list->af_cattrs.af_owndomain = af_enterdomain (user->af_userdomain);
	}
      else
	{
	  list->af_cattrs.af_ownname = af_entersym (owner->af_username);
	  list->af_cattrs.af_ownhost = af_enterhost (owner->af_userhost);
	  list->af_cattrs.af_owndomain = af_enterdomain (owner->af_userdomain);
	}
      if (writeok)
	list->af_extent |= AF_UXWRITE;
      
      if ((list->af_list = (Af_vattrs *)af_malloc (list, (unsigned) (list->af_listlen * sizeof(Af_vattrs)))) == (Af_vattrs *)0)
	return (ERROR);
      
      bzero ((char *)list->af_list, list->af_listlen * sizeof (Af_vattrs));
      /* init attrbuf for busy version  (relevant attrs only) */
      list->af_list[0].af_name = list->af_cattrs.af_globname;
      list->af_list[0].af_type = list->af_cattrs.af_globtype;
      list->af_list[0].af_gen = AF_BUSYVERS;
      list->af_list[0].af_rev = AF_BUSYVERS;
      list->af_list[0].af_state = AF_BUSY;
      list->af_list[0].af_class = AF_VALID;
      list->af_list[0].af_auname = af_entersym (user->af_username);
      list->af_list[0].af_auhost = af_enterhost (user->af_userhost);
      list->af_list[0].af_audomain = af_enterdomain (user->af_userdomain);
      list->af_list[0].af_mode = (u_short) bibuf.st_mode;
      list->af_list[0].af_lckname = (char *)0;
      list->af_list[0].af_lckhost = (char *)0;
      list->af_list[0].af_lckdomain = (char *)0;
      list->af_list[0].af_mtime = bibuf.st_mtime;
      list->af_list[0].af_atime = bibuf.st_atime;
      list->af_list[0].af_ctime = bibuf.st_ctime;
      list->af_list[0].af_stime = AF_NOTIME;
      list->af_list[0].af_ltime = AF_NOTIME;
      list->af_list[0].af_notesize = 0;
      list->af_list[0].af_note = (char *)0;
      tmpkey.af_lpos = 0;
      (void) afInitUdas (&tmpkey);
      list->af_list[0].af_repr = AF_FILE;
      list->af_list[0].af_fsize = (off_t) bibuf.st_size;
      list->af_list[0].af_dsize = 0;
      list->af_list[0].af_data = (char *)0;
      list->af_list[0].af_hashname = (char *)0;
      list->af_list[0].af_nrefs = 0;
      list->af_list[0].af_succgen = AF_NOVNUM;
      list->af_list[0].af_succrev = AF_NOVNUM;
      list->af_list[0].af_predgen = AF_NOVNUM;
      list->af_list[0].af_predrev = AF_NOVNUM;
      return (AF_OK);
    }
  
  /* record date of last modification */
  list->af_lastmod = aibuf.st_mtime;

  /* archive file ??? */
  idstr[AF_SEGSTRLEN] = '\0';
  (void) fgets (idstr, AF_SEGSTRLEN+1, archfile);
  if (strncmp (idstr, AF_ARHEADER, AF_SEGSTRLEN))
    FAIL ("ReadAttrs",
	  "wrong header in archive file", AF_EINCONSIST, ERROR);
  
  /* read header */
  (void) fgets (line, AF_LINESIZ, archfile);
  itemptr = firstitem (line);
  arVersion = atoi (itemptr);
  if (arVersion > AF_ARCURVERS)
    FAIL ("ReadAttrs", "unknown archive format version", AF_EINCONSIST, ERROR);

  itemptr = nextitem (itemptr);
  list->af_nrevs = atoi (itemptr);
  itemptr = nextitem (itemptr);
  list->af_datasize = atoi (itemptr);

  /* alloc memory for revision list (plus space for new revs) */
  list->af_listlen = list->af_nrevs + AF_NEWREVS;
  if ((list->af_list = (Af_vattrs *)af_malloc (list, (unsigned) (list->af_listlen * sizeof(Af_vattrs)))) == (Af_vattrs *)0)
    return (ERROR);

  bzero ((char *) list->af_list, list->af_listlen * sizeof (Af_vattrs));

  /* enter name and type */
  list->af_list[0].af_name = list->af_cattrs.af_globname;
  list->af_list[0].af_type = list->af_cattrs.af_globtype;

  if (af_rdattrs (archfile, list, &bibuf) != AF_OK)
    {
      list->af_nrevs = 0;
      (void) afDetachList (list);
      return (ERROR);
    }

  /* read id string for user defined attributes section*/
  (void) fgets (idstr, AF_SEGSTRLEN+1, archfile);
  if (strncmp (idstr, AF_UDASEG, AF_SEGSTRLEN))
    FAIL ("ReadAttrs", "wrong udaseg-ID in archive file", AF_EINCONSIST,ERROR);

  if (af_rdudas (archfile, list) != AF_OK)
    {
      list->af_nrevs = 0;
      (void) afDetachList (list);
      return (ERROR);
    }

  list->af_extent |= (AF_ATTRS | AF_COMPLETE);
  (void) fclose (archfile);
  return (AF_OK);
} /* afReadAttrs */

/*==========================================================================
 *	WriteArchive -- write archive file
 *
 *
 *==========================================================================*/

EXPORT afWriteArchive (list)
     Af_revlist *list;
{
  register int  i, j, maxindex;
  register FILE	*tmpfile, *datafile;
  char          tmpname[MAXPATHLEN], *afilesym, *datasym = (char *)0;
  char          archname[MAXPATHLEN], *ptrlist[AF_MAXUDAS+1];
  off_t	        datasize;
  Af_key        tmpkey, *busyptr, *af_gbuskey();

  tmpkey.af_ldes = list;

  /* if all revisions have been removed */
  if (list->af_nrevs == 0)
    {
/* DISABLE SIGNALS */
      (void) unlink (list->af_arfilename);
      (void) strcpy (archname, list->af_arfilename);
      archname[strlen(archname)-sizeof(char)] = AF_DATAEXT;
      (void) unlink (archname);
/* ENABLE SIGNALS */
      return (AF_OK);
    }

  (void) strcpy (tmpname, list->af_arfilename); 
  tmpname[strlen(tmpname) - sizeof(char)] = AF_ARCHTMP;
  afilesym = af_entersym (tmpname);
  af_regtmpfile (af_entersym (afilesym));
  /* open tmpfile */
  if ((tmpfile = fopen (afilesym, "w")) == (FILE *)0)
    FAIL ("WriteArchive", "fopen", AF_ESYSERR, ERROR);

  /* if there is no busy version - increase "nrevs" temporarily */
  busyptr = af_gbuskey (list);
  if (!(VATTR(busyptr).af_class & AF_VALID))
    list->af_nrevs++;

  /* write header */
  fprintf (tmpfile, "%s %d %d %ld\n", AF_ARHEADER, AF_ARCURVERS,
	   list->af_nrevs, list->af_datasize);

  /* write constant attributes */
  fprintf (tmpfile, "%s %s %s %s %s %s\n", AF_NAMEID, 
	   list->af_cattrs.af_host,
	   list->af_cattrs.af_syspath,
	   list->af_list[0].af_name,
	   NOTMT (list->af_list[0].af_type),
	   AF_NOSTRING); /* former variant string */

  /* write owner */
  fprintf (tmpfile, "%s %s %s %s\n", AF_OWNID,
	   list->af_cattrs.af_ownname,
	   list->af_cattrs.af_ownhost,
	   list->af_cattrs.af_owndomain);


  /* write predecessor and locker of busy version */
  fprintf (tmpfile, "%s %d %d\n%s %s %s %s %d\n", AF_PRDID,
	   VATTR(busyptr).af_predgen,
	   VATTR(busyptr).af_predrev, AF_LOCKID,
	   NOTMT (VATTR(busyptr).af_lckname),
	   NOTMT (VATTR(busyptr).af_lckhost),
	   NOTMT (VATTR(busyptr).af_lckdomain),
	   VATTR(busyptr).af_ltime);

  /* write list of version attributes */
  maxindex = list->af_nrevs-1;
  for (i=1; i <= maxindex; i++)
    {
      /* skip deleted versions */
      if (!(list->af_list[i].af_class & AF_VALID))
	{
	  if (++maxindex == list->af_listlen)
	    {
	      (void) fclose (tmpfile);
	      af_unregtmpfile (afilesym);
	      (void) unlink (afilesym);
	      FAIL ("WriteArchive", "revision count", AF_EINCONSIST, ERROR);
	    }
	  continue;
	}

      /* write revision ID */
      fprintf (tmpfile, "%s %d %d %d %o %s\n", AF_REVID, 
	       list->af_list[i].af_gen,
	       list->af_list[i].af_rev,
	       list->af_list[i].af_state,
	       list->af_list[i].af_mode,
	       AF_NOSTRING); /* former variant string */

      /* write author */
      fprintf (tmpfile, "\t%s %s %s %s %s %s %s\n", AF_AUTHORID,
	       list->af_list[i].af_auname,
	       list->af_list[i].af_auhost,
	       list->af_list[i].af_audomain,
	       NOTMT (list->af_list[i].af_lckname),
	       NOTMT (list->af_list[i].af_lckhost),
	       NOTMT (list->af_list[i].af_lckdomain));

      /* write dates */
      fprintf (tmpfile, "\t%s %ld %ld %ld %ld %ld\n", AF_DATEID,
	       list->af_list[i].af_mtime,
	       list->af_list[i].af_atime,
	       list->af_list[i].af_ctime,
	       list->af_list[i].af_stime,
	       list->af_list[i].af_ltime);

      /* write kind of representation and tree connects */
      fprintf (tmpfile, "\t%s %d %ld %ld %d %d %d %d\n", AF_REPRID,
	       list->af_list[i].af_repr,
	       list->af_list[i].af_fsize,
	       list->af_list[i].af_dsize,
	       list->af_list[i].af_succgen,
	       list->af_list[i].af_succrev,
	       list->af_list[i].af_predgen,
	       list->af_list[i].af_predrev);
    }

  /* write user defined attributes */
  fprintf (tmpfile, "%s\n", AF_UDASEG);
  maxindex = list->af_nrevs-1;
  for (i=0; i <= maxindex; i++)
    {
      /* skip deleted versions but not the busy version */
      if (!(list->af_list[i].af_class & AF_VALID) && 
	  (list->af_list[i].af_state != AF_BUSY))
	{ maxindex++; continue;	}

      fprintf (tmpfile, "%s %d %d\n", AF_UDAID,
	       list->af_list[i].af_gen,
	       list->af_list[i].af_rev);
      tmpkey.af_lpos = i;
      (void) afListUdas (&tmpkey, ptrlist);
      j=0;
      while (ptrlist[j])
	fprintf (tmpfile, "%s%c", ptrlist[j++], '\0');
      if (j==0) /* if no user defined attribute has been written */
	(void) putc ('\0', tmpfile);
      (void) putc ('\0', tmpfile);
      (void) putc ('\n', tmpfile);
    }
  (void) fclose (tmpfile);

  /* if data have been manipulated - write data file */
  if ((list->af_extent & AF_DATA) == AF_DATA)
    {
      (void) strcpy (tmpname, list->af_arfilename);
      tmpname[strlen(tmpname) - sizeof(char)] = AF_DATATMP;
      datasym = af_entersym (tmpname);
      af_regtmpfile (datasym);
      /* open tmpfile for data */
      if ((datafile = fopen (datasym, "w")) == (FILE *)0)
	{
	  af_unregtmpfile (afilesym);
	  (void) unlink (afilesym);
	  FAIL ("WriteArchive", "fopen", AF_ESYSERR, ERROR);
	}

      /* write notes and data */
      fprintf (datafile, "%s %d\n", AF_DATAHEADER, AF_ARCURVERS);
      
      maxindex = list->af_nrevs-1;
      for (i=1; i <= maxindex; i++)
	{
	  /* skip deleted versions */
	  if (!(list->af_list[i].af_class & AF_VALID))
	    { maxindex++; continue; }
	  
	  fprintf (datafile, "%s %d %d %ld\n", AF_NOTEID,
		   list->af_list[i].af_gen,
		   list->af_list[i].af_rev,
		   list->af_list[i].af_notesize);
	  if (list->af_list[i].af_notesize > 0)
	    {
	      if (fwrite (list->af_list[i].af_note, sizeof(char), (Size_t) list->af_list[i].af_notesize - sizeof(char), datafile) == 0)
		{
		  /* special case */
		  if (list->af_list[i].af_notesize > 1)
		    {
		      (void) fclose (datafile);
		      af_unregtmpfile (datasym);
		      (void) unlink (datasym);
		      af_unregtmpfile (afilesym);
		      (void) unlink (afilesym);
		      FAIL ("WriteArchive", "fwrite", AF_ESYSERR, ERROR);
		    }
		}
	    }
	  (void) putc ('\n', datafile);

	  if (list->af_list[i].af_repr == AF_CHUNK)
	    datasize = list->af_list[i].af_fsize;
	  else datasize = list->af_list[i].af_dsize;
	  fprintf (datafile, "%s %d %d %d %ld\n", AF_DATAID,
		   list->af_list[i].af_gen,
		   list->af_list[i].af_rev,
		   list->af_list[i].af_repr, datasize);

	  if (datasize > 0)
	    {
	      if (fwrite (list->af_list[i].af_data, sizeof(char), (Size_t) datasize, datafile) == 0)
		{
		  (void) fclose (datafile);
		  af_unregtmpfile (datasym);
		  (void) unlink (datasym);
		  af_unregtmpfile (afilesym);
		  (void) unlink (afilesym);
		  FAIL ("WriteArchive", "fwrite", AF_ESYSERR, ERROR);
		}
	    }
	}
      (void) fclose (datafile);
    }

/* DISABLE SIGNALS */
  disableSig (SIGINT);
  disableSig (SIGQUIT);
  af_unregtmpfile (afilesym);
  (void) unlink (list->af_arfilename);
  if (link (afilesym, list->af_arfilename) == ERROR)
    {
      af_unregtmpfile (datasym);
      (void) unlink (datasym);
      (void) unlink (afilesym);
      FAIL ("WriteArchive", "link", AF_ESYSERR, ERROR);
    }

  (void) chmod (list->af_arfilename, AF_ARCHMODE);
  (void) chown (list->af_arfilename, geteuid(), list->af_owngid);
  (void) unlink (afilesym);

  /* if the data file has been written */
  if ((list->af_extent & AF_DATA) == AF_DATA)
    {
      (void) strcpy (archname, list->af_arfilename);
      archname[strlen(archname)-sizeof(char)] = AF_DATAEXT;
      af_unregtmpfile (datasym);
      (void) unlink (archname);
      if (link (datasym, archname) == ERROR)
	{
	  (void) unlink (datasym);
	  FAIL ("WriteArchive", "link", AF_ESYSERR, ERROR);
	}
      (void) chmod (archname, AF_ARCHMODE);
      (void) chown (archname, geteuid(), list->af_owngid);
      (void) unlink (datasym);
    }
/* ENABLE SIGNALS */
  enableSig();

  /* decrease "nrevs" again (see beginning of procedure */
  if (!(VATTR(busyptr).af_class & AF_VALID))
    list->af_nrevs--;
  
  return (AF_OK);
} /* afWriteArchive */

