(* InfixStack - we have to implement an operator precedence algorithm both
		for expressions and for patterns; they share the type of
		infix basis. Rather than trying to get too polymorphic in
		the core-ML code in Infixing, it seems neater to have a
		functor and apply it once for expressions and once for
		patterns. *)

(*
$File: Parsing/InfixStack.sml $
$Date: 1992/01/29 15:08:39 $
$Revision: 1.5 $
$Locker:  $
*)

(*$InfixStack: INFIX_BASIS CRASH INFIX_STACK*)
functor InfixStack(structure InfixBasis: INFIX_BASIS

		   type FullObject	(* exp or pat *)
		   type AtomObject	(* atexp or atpat *)
		   eqtype id sharing type id = InfixBasis.id
		   val pr_id: id -> string

		   val atomToFull: AtomObject -> FullObject
		   val fullToAtom: FullObject -> AtomObject

		   val pair: FullObject * FullObject -> AtomObject

		   val asId: AtomObject -> id Option
		   val applyId: id * AtomObject -> FullObject
		   val applyObj: FullObject * AtomObject -> FullObject

		   structure Crash: CRASH
		  ): INFIX_STACK =
  struct
    open InfixBasis
    type InfixBasis = Basis

    type FullObject = FullObject
     and AtomObject = AtomObject

   (* We can stack "operators" (infixes and the implicit applications). *)

    datatype StackEntry = INFIXentry of id * int
			| INFIXRentry of id * int
			| APPLentry

   (* The list of output operands consists of atomic exps/pats (things
      which are passed through unchanged, for example) as well as exps/pats
      (results of applications). *)

    datatype OutputEntry = ATOM of AtomObject
      			 | FULL of FullObject

   (* Coerce an object in the output list to be an expression or an atomic
      expression, according to what we want to do with it. *)

    fun atom(ATOM x) = x
      | atom(FULL x) = fullToAtom x

    and full(ATOM x) = atomToFull x
      | full(FULL x) = x

   (* Keep track of the last thing we passed - needed to spot
      applications (two operands successively, with no intervening
      operator). *)

    datatype LastObj = ARG | OPER | VOID

   (* apply - apply a stack entry (infix(r) or appl) to the list of
      generated arguments. Might fail on source input like "A ::" where we
      run out of arguments while flushing the stack, so we make apply
      bulletproof. Note that the args list is in reverse order. *)

    fun apply(entry, (snd :: fst :: rest): OutputEntry list) =
          let
	    val thePair = pair(full fst, full snd)
	  in
	    FULL(case entry
		   of INFIXentry(id, _) => applyId(id, thePair)
		    | INFIXRentry(id, _) => applyId(id, thePair)
		    | APPLentry => applyObj(full fst, atom snd)
	        ) :: rest
	  end

      | apply(entry, _) =
	  Crash.unimplemented("Error: InfixStack.apply: "
			      ^ (case entry
				   of INFIXentry(id, _) =>
					"<infix " ^ pr_id id ^ ">"
				    | INFIXRentry(id, _) =>
					"<infixr " ^ pr_id id ^ ">"
				    | APPLentry =>
					"<appl>"
			        )
			     )


   (* assocLeft - precedence comparison of infix(r) and appl stack entries.
		  Application is the highest priority, and associates to
		  the left. Other operators associate according to
		  precedence. If the precedences are equal, then they always
		  associate to the left, *unless* they are the same
		  operator, in which case they associate according to their
		  infix/infixr status. *)

    and assocLeft(op1, op2) =
      case (op1, op2)
	of (APPLentry, _) => true	(* APPL is highest (left) precedence. *)
	 | (_, APPLentry) => false
	 | _ =>
	     let
	       fun extract(INFIXentry(id, n)) = (id, n)
		 | extract(INFIXRentry(id, n)) = (id, n)
		 | extract _ = Crash.impossible "assocLeft/extract"

	       val (id1, prec1) = extract op1
	       val (id2, prec2) = extract op2
	     in
	       if prec1 > prec2 then true
	       else if prec1 < prec2 then false
	       else if id1 <> id2 then true	(* distinct id's: left assoc. *)
	       else				(* same id. *)
		 case op1 of INFIXentry _ => true | _ => false
	     end


   (* flushHigher - flush out all entries in the stack with higher
      effective precedence than "entry". Take INFIX, INFIXR, APPL status
      into account. *)

    fun flushHigher(entry, stack, output) =
      case stack
	of nil => (nil, output)
	 | top :: rest =>
	     if assocLeft(top, entry) then
	       flushHigher(entry, rest, apply(top, output))
	     else
	       (stack, output)


   (* flushAll - clear out all the stacked operators at the end of an
      atexp sequence. *)

    fun flushAll(stack, output) =
      case stack
	of nil => (case output
		     of [item] => item
		      | _ => Crash.impossible "InfixStack.flushAll"
		  )

	 | top :: rest => flushAll(rest, apply(top, output))


   (* process - the shunting function, with the usual Rothwell interpretation
      (viz. apply any outgoing operators to things in the output list, rather
      than add them to the list in reverse polish notation). *)

    fun process(iBas: InfixBasis.Basis,
		input: AtomObject list, stack: StackEntry list,
		last: LastObj, output: OutputEntry list
	       ): OutputEntry =
      case input
	of nil =>			(* Nothing more to process *)
	     flushAll(stack, output)

	 | this :: rest =>
	     (case asId this
		of Some id =>		(* Dealing with an unqual. id. *)
		     (case lookup iBas id
			of INFIX n =>
			     operator(iBas, INFIXentry(id, n),
				      rest, stack, output
				     )

			 | INFIXR n =>
			     operator(iBas, INFIXRentry(id, n),
				      rest, stack, output
				     )

			 | NONFIX =>
			     (case last
				of ARG =>	(* Must generate an appl. *)
				     operator(iBas, APPLentry, input,
					      stack, output
					     )

				 | _ =>		(* Just pass the nonfix. *)
				     process(iBas, rest, stack, ARG,
					     ATOM this :: output
					    )
			     )
		     )

		 | None =>		(* Not an unqual. identifier. *)
		     (case last
			of ARG =>	(* Must generate an application *)
			     operator(iBas, APPLentry, input, stack, output)

			 | _ =>		(* Just pass it through. *)
			     process(iBas, rest, stack, ARG,
				     ATOM this :: output
				    )
		     )
	     )

   (* operator - flush all stack entries with higher precedence, and then
      stack this one. *)

    and operator(iBas, entry, input, stack, output) =
      let
	val (stack', output') = flushHigher(entry, stack, output)
      in
	process(iBas, input, entry :: stack', OPER, output')
      end

   (* resolveInfix - takes a list of atomic exps/pats and returns a
      single exp/pat with the nonfix and infix applications in place.
      Usual Dijkstra shunting algorithm stuff, except that we have to
      identify runs of nonfixes (they must be applications), and we have to
      detect ill-formed cases (the input grammer was only specific enough to
      deliver a list of atexps, which includes things like "2 ::"). *)

    fun resolveInfix(iBas, atoms) =
      full(process(iBas, atoms, nil, VOID, nil))
  end;
