/*
 * Command Input Shell
 * Dave Clemans
 * 12/88-1/89
 *
 * "spiritually" based on Bourne, Korn shells
 *
 * Input Lexer
 *
 * $Id: lex.c,v 1.11 89/03/05 15:38:27 dclemans Exp $
 *
 * $Log:	lex.c,v $
 * Revision 1.11  89/03/05  15:38:27  dclemans
 * more fixes
 * 
 * Revision 1.10  89/02/25  17:40:08  dclemans
 * miscellaneous bug fixes/speedups
 * 
 * Revision 1.9  89/02/22  21:47:25  dclemans
 * bug fix ] vs ]]
 * 
 * Revision 1.8  89/02/22  16:26:54  dclemans
 * Implement [[, ]] brackets
 * 
 * Revision 1.7  89/02/22  13:12:27  dclemans
 * Fix bug with null tokens
 * 
 * Revision 1.6  89/02/22  10:16:35  dclemans
 * Fix bugs with process waiting, pid reporting, parsing of &
 * 
 * Revision 1.5  89/02/21  20:29:56  dclemans
 * Fix bug with shell variable references in history lists.
 * 
 * Revision 1.4  89/02/20  20:11:56  dclemans
 * Add RCS identifiers
 * 
 */
#include <stdio.h>
#include "shell.h"

static int dont_expand = 0;

int lex_heredoc_do(iop)
struct iotoken *iop;
{
    int ofd,len,next_char;
    char buffer[BUFSIZ];

    if (iop->type != SYM_LLT && iop->type != SYM_LLTL)
        return -1;
    len = 16;
#ifdef  GEMDOS
    len += strlen(base_env.tmpdir);
#endif  /* GEMDOS */
    iop->tempfile = new_string(len+1);
    if (iop->tempfile == (char *)NULL)
    {   /* enough memory? */
        errmsg(SHERR_NOMEM,LOC("io_heredoc"));
        return -1;
    }
#ifdef  GEMDOS
    sprintf(iop->tempfile,"%stemp%d.tmp",base_env.tmpdir,base_env.temp_count++);
#else
    sprintf(iop->tempfile,"/tmp/sh%d.temp",base_env.temp_count++);
#endif  /* GEMDOS */
    ofd = creat(iop->tempfile,0666);
    if (ofd < 0)
    {   /* was it created? */
        errmsg(0,LOC("io_heredoc"),"couldn't create %s",iop->tempfile);
        free(iop->tempfile);
        iop->tempfile = (char *)NULL;
        return -1;
    }
    len = 0;
    if (iop->file[0] == ESCAPE_CHAR ||
        iop->file[0] == '\'' || iop->file[0] == '"')
        dont_expand = 1;
    stripquotes(iop->file);
    while ((next_char = io_getchar(1)) != SYM_EOF)
    {   /* get chars to fill up heredoc */
        buffer[len++] = next_char;
        if (next_char == '\n' || len >= sizeof buffer)
        {   /* end of line? */
#ifdef  GEMDOS
            if (next_char == '\n')
                buffer[len-1] = '\r';
#endif  /* GEMDOS */
            while (iop->type == SYM_LLTL && isspace(buffer[0]) && len > 0)
            {   /* strip leading white space? */
                len--;
                strncpy(buffer,&buffer[1],len);
            }
            if (len == 0)
                continue;
            if (strncmp(iop->file,buffer,strlen(iop->file)) == 0)
            {   /* heredoc end of file? */
                break;
            }
            write(ofd,buffer,len);
            len = 0;
#ifdef  GEMDOS
            if (next_char == '\n')
                write(ofd,"\n",1);
#endif  /* GEMDOS */
        }
    }
    dont_expand = 0;
    close(ofd);
    return 0;
}   /* end of lex_heredoc_do */

int lex_heredoc()
{
    struct herestack *hs,*hsp,*ohsp;

    hs = base_env.pending_heredocs;
    base_env.pending_heredocs = (struct herestack *)NULL;
    if (hs == (struct herestack *)NULL)
        return 0;
    for (hsp = hs; hsp != (struct herestack *)NULL; )
    {   /* for each pending heredoc */
        ohsp = hsp;
        hsp = hsp->next;
        lex_heredoc_do(ohsp->doc);
        free(ohsp);
    }
    return 0;
}   /* end of lex_heredoc */

int lex_heredoc_save(iop)
struct iotoken *iop;
{
    struct herestack *hs,*hsp;

    hsp = (struct herestack *)malloc(sizeof(*hsp));
    if (hsp == (struct herestack *)NULL)
    {   /* enough memory */
        errmsg(SHERR_NOMEM,LOC("lex_heredoc_save"));
        return -1;
    }
    hsp->next = (struct herestack *)NULL;
    hsp->doc = iop;

    if (base_env.pending_heredocs == (struct herestack *)NULL)
        base_env.pending_heredocs = hsp;
    else
    {   /* tack onto end */
        for (hs = base_env.pending_heredocs; hs->next != (struct herestack *)NULL; hs = hs->next)
            /* do nothing */;
        hs->next = hsp;
    }
    return 0;
}   /* end of lex_heredoc_save */

static int typeof(word)
register char *word;
{
    register char *p;
    int flag;

    if (isdigit(*word) || *word == '-' || *word == '+')
    {   /* is the whole word numeric? */
        flag = SYM_NUMBER;
        for (p = &word[1]; *p; p++)
            if (!isdigit(*p))
                flag = SYM_WORD;
        return flag;
    }
    switch (*word)
    {   /* what kind of word? */
        case '`':
            return SYM_BQUOTE;
        case ';':
            if (word[1] == ';')
                return SYM_DSEMI;
            return SYM_SEMI;
        case '<':
            if (word[1] == '&')
                return SYM_LBT;
            else if (word[1] == '<')
            {   /* if a here doc */
                if (word[2] == '-')
                    return SYM_LLTL;
                return SYM_LLT;
            }
            else    return SYM_LT;
        case '>':
            if (word[1] == '&')
                return SYM_RBT;
            else if (word[1] == '>')
                return SYM_RRT;
            else    return SYM_RT;
        case '{':
            return SYM_LBRACE;
        case '}':
            return SYM_RBRACE;
        case '(':
            if (word[1] == '(')
                return SYM_DLPAREN;
            else return SYM_LPAREN;
        case ')':
            if (word[1] == ')')
                return SYM_DRPAREN;
            return SYM_RPAREN;
        case '&':
            if (word[1] == '&')
                return SYM_ANDIF;
            else    return SYM_BACK;
        case '|':
            if (word[1] == '|')
                return SYM_ORIF;
            else    return SYM_PIPE;
        case '[':
            if (word[1] == '[')
                return SYM_DLBRACK;
            else    return SYM_WORD;
        case ']':
            if (word[1] == ']')
                return SYM_DRBRACK;
            else    return SYM_DRBRACK;
        case '\n':
        case '\r':
            return SYM_EOL;
        default:
            return SYM_WORD;
    }
    /* NOT REACHED */
}   /* end of typeof */

struct token *lex_token(level)
int level;
{
    char workbuf[BUFSIZ];
    register char *first,*ptr;
    int keepgoing,keyword,next_char,la_char;
    register struct token *tp;
    struct phrase *sp;
    char *result;
#define APPEND(c) ((first<&workbuf[sizeof workbuf])?*first++ = c:errmsg(0,LOC("lex_token"),"word too long"))

    while ((next_char = io_getchar(level)) != SYM_EOF)
    {   /* skip leading blanks */
        if (strchr(base_env.separators,next_char) == (char *)NULL)
            break;
    }
    if (next_char == SYM_EOF)
        return (struct token *)NULL;
    keyword = 0;
    keepgoing = 1;
    first = workbuf;
    while (next_char != SYM_EOF && keepgoing && strchr(base_env.separators,next_char) == (char *)NULL)
    {   /* go pick off a word */
        switch (next_char)
        {   /* look for quoting, etc. */
            case '#':   /* rest of line a comment? */
                while (next_char != '\r' && next_char != '\n')
                {   /* scan til end of line */
                    next_char = io_getchar(level);
                    if (next_char == SYM_EOF)
                        break;
                }
                io_pushback(next_char);
                break;
            case '`':   /* `` expansion string */
                if (level == 0)
                {   /* starting string? */
                    sp = lex_sentence(1);
                    exec_sentence(sp,0);
                    io_pushfile(sp->group_end->io->file,1,1,0);
                    sentence_free(sp);
                }
                else
                {   /* end of string */
                    keepgoing = 0;
                    if (first != workbuf)
                        io_pushback(next_char);
                    else APPEND(next_char);
                }
                break;
            case ESCAPE_CHAR:
                APPEND(next_char);
                next_char = io_getchar(1);
                APPEND(next_char);
                break;
            case '"':   /* double quote; allow variables */
                APPEND('"');
                while ((next_char = io_getchar(1)) != '"')
                {   /* until end of string */
                    if (next_char == SYM_EOF)
                        break;
                    if (next_char == ESCAPE_CHAR)
                    {   /* grab next char */
                        APPEND(next_char);
                        next_char = io_getchar(1);
                    }
                    else if (next_char == '$')
                    {   /* expand out a variable */
                        result = var_reference(dont_expand);
                        if (result != (char *)NULL)
                        {   /* stick in expansion */
                            for (ptr = result; *ptr; ptr++)
                                APPEND(*ptr);
                            free(result);
                        }
                        continue;
                    }
                    else if (next_char == '`')
                    {   /* substituted command */
                        if (level == 0)
                        {   /* starting string? */
                            sp = lex_sentence(1);
                            exec_sentence(sp,0);
                            io_pushfile(sp->group_end->io->file,1,1,0);
                            sentence_free(sp);
                        }
                        else
                        {   /* end of string */
                            keepgoing = 0;
                            if (first != workbuf)
                                io_pushback(next_char);
                            else APPEND(next_char);
                        }
                        continue;
                    }
                    APPEND(next_char);
                }
                if (next_char == SYM_EOF)
                {   /* bad end-of-string */
                    errmsg(0,LOC("lex_token"),"end of string not found");
                    break;
                }
                APPEND('"');
                break;
            case '\'':  /* single quote */
                APPEND('\'');
                while ((next_char = io_getchar(1)) != '\'')
                {   /* until end of string */
                    if (next_char == SYM_EOF)
                        break;
                    APPEND(next_char);
                }
                if (next_char == SYM_EOF)
                {   /* bad end-of-string? */
                    errmsg(0,LOC("lex_token"),"missing end of string");
                    break;
                }
                APPEND('\'');
                break;
            case '[':
            case ']':
                la_char = io_getchar(level);
                if (next_char != la_char)
                {   /* if not a double [[ or ]] */
                    io_pushback(la_char);
                    APPEND(next_char);
                    break;
                }
                if (first != workbuf)
                {   /* ending current token? */
                    io_pushback(la_char);
                    io_pushback(next_char);
                    keepgoing = 0;
                    break;
                }
                APPEND(next_char);
                APPEND(la_char);
                keepgoing = 0;
                break;
            case ';':   /* end of phrase */
            case '(':   /* start of group */
            case ')':   /* end of group */
                if (first != workbuf)
                {   /* ending current token? */
                    io_pushback(next_char);
                    keepgoing = 0;
                    break;
                }
                APPEND(next_char);
                next_char = io_getchar(level);
                if (next_char != SYM_EOF && first[-1] == next_char)
                {   /* a two character symbol? */
                    APPEND(next_char);
                }
                else
                {   /* a one character symbol */
                    io_pushback(next_char);
                }
                keepgoing = 0;
                break;
            case SYM_MARKER:
                keepgoing = 0;
                workbuf[0] = '\0';
                first = workbuf;
                break;
            case '\n':  /* end of line */
            case '\r':
                lex_heredoc();
                /* FALL THROUGH! */
            case '{':   /* left brace */
            case '}':   /* right brace */
                if (first != workbuf)
                {   /* ending current token? */
                    io_pushback(next_char);
                    keepgoing = 0;
                    break;
                }
                APPEND(next_char);
                keepgoing = 0;
                break;
            case '|':   /* a pipe? */
            case '&':   /* background? */
            case '<':   /* I/O? */
            case '>':   /* I/O? */
                if (first != workbuf)
                {   /* ending current token? */
                    io_pushback(next_char);
                    keepgoing = 0;
                    break;
                }
                APPEND(next_char);
                keepgoing = 0;
                next_char = io_getchar(level);
                if ((next_char != SYM_EOF) &&
                    (first[-1] == next_char ||
                    (first[-1] == '<' && next_char == '&') ||
                    (first[-1] == '>' && next_char == '&')))
                {   /* a two character symbol? */
                    APPEND(next_char);
                    if (next_char == '<')
                    {   /* three character (<<-)? */
                        next_char = io_getchar(level);
                        if (next_char == '-')
                            APPEND(next_char);
                        else io_pushback(next_char);
                    }
                }
                else
                {   /* a one character symbol */
                    io_pushback(next_char);
                }
                break;
            case '=':   /* possible assignment? */
                keyword = 1;
                /* FALL THROUGH */
            default:    /* and everything else */
                if (next_char == '$')
                {   /* expand a variable? */
                    result = var_reference(dont_expand);
                    if (result != (char *)NULL)
                    {   /* stick in expansion */
                        for (ptr = result; *ptr; ptr++)
                            APPEND(*ptr);
                        free(result);
                    }
                    break;
                }
                APPEND(next_char);
                break;
        }
        if (!keepgoing)
            break;
        if (next_char != SYM_EOF)
            next_char = io_getchar(level);
    }
    if (first == workbuf && next_char != SYM_EOF && keepgoing && strchr(base_env.separators,next_char) != (char *)NULL)
    {   /* reject this as useless token? */
        return lex_token(level);
    }
    tp = new_token((int)(first-workbuf));
    if (tp == (struct token *)NULL)
    {   /* enough memory? */
        errmsg(SHERR_NOMEM,LOC("lex_token"));
        return tp;
    }
    strncpy(tp->name,workbuf,(int)(first-workbuf));
    tp->name[(int)(first-workbuf)] = '\0';
    if (next_char == SYM_MARKER)
        tp->type = SYM_MARKER;
    else tp->type = typeof(tp->name);
    if (keyword && tp->type == SYM_WORD)
        tp->type = SYM_KEYWORD;
    return tp;
#undef  APPEND
}   /* end of lex_token */

struct brackets
{
    char    *name;
    char    *start;
    char    *end;
};
static struct brackets cmd_brackets[] =
{
    { "case",       "",         "esac"  },
    { "if",         "then",     "fi"    },
    { "for",        "do",       "done"  },
    { "for",        "{",        "}"     },
    { "function",   "{",        "}"     },
    { "select",     "do",       "done"  },
    { "select",     "{",        "}"     },
    { "until",      "do",       "done"  },
    { "until",      "{",        "}"     },
    { "while",      "do",       "done"  },
    { "while",      "{",        "}"     },
    { (char *)NULL, (char *)NULL,   (char *)NULL }
};

int reserved_word(word)
char *word;
{
    register int ctr;

    for (ctr = 0; cmd_brackets[ctr].name != (char *)NULL; ctr++)
    {   /* check each bracket set */
        if (strcmp(cmd_brackets[ctr].name,word) == 0)
            return 1;
        if (strcmp(cmd_brackets[ctr].start,word) == 0)
            return 1;
        if (strcmp(cmd_brackets[ctr].end,word) == 0)
            return 1;
    }
    if (strcmp("else",word) == 0)
        return 1;
    if (strcmp("elif",word) == 0)
        return 1;
    return 0;
}   /* end of reserved_word */

struct phrase *lex_paragraph(level,fpp)
int level;
struct phrase *fpp;
{
    register struct phrase *pp;
    struct token *tp,*stp,*etp;
    register int ctr;

    if (fpp == (struct phrase *)NULL)
        return fpp;
    tp = fpp->body;
    if (tp == (struct token *)NULL)
        return fpp;
    for (ctr = 0; cmd_brackets[ctr].name != (char *)NULL; ctr++)
    {   /* start of a paragraph? */
        if (strcmp(tp->name,cmd_brackets[ctr].name) == 0)
            break;
    }
    if (cmd_brackets[ctr].name == (char *)NULL)
        return fpp;
    if (strlen(cmd_brackets[ctr].start) != 0)
    {   /* if need a starting bracket */
        stp = lex_token(1);
        if (stp == (struct token *)NULL)
        {   /* no paragraph start? */
            errmsg(0,LOC("lex_paragraph"),"missing start of paragraph");
            return fpp;
        }
        if (stp->type == SYM_EOL && strcmp(stp->name,"\n") == 0)
        {   /* skip empty token */
            free(stp);
            stp = lex_token(1);
            if (stp == (struct token *)NULL)
            {   /* no paragraph start? */
                errmsg(0,LOC("lex_paragraph"),"missing start of paragraph");
                return fpp;
            }
        }
        for (ctr = 0; cmd_brackets[ctr].name != (char *)NULL; ctr++)
        {   /* now can find "correct" brackets */
            if (strcmp(tp->name,cmd_brackets[ctr].name) == 0 &&
                strcmp(stp->name,cmd_brackets[ctr].start) == 0)
                break;
        }
        if (cmd_brackets[ctr].name == (char *)NULL)
        {   /* unknown combo? */
            errmsg(0,LOC("lex_paragraph"),"unexpected paragraph tokens: %s,%s",
                tp->name,stp->name);
            io_pushtoken(stp->name,0);
            free(stp);
            return fpp;
        }
        pp = new_phrase();
        if (pp == (struct phrase *)NULL)
        {   /* enough memory? */
            free(stp);
            errmsg(SHERR_NOMEM,LOC("lex_paragraph"));
            return fpp;
        }
        pp->body = pp->body_end = stp;
        pp->type = new_token(1);
        if (pp->type == (struct token *)NULL)
        {   /* enough memory? */
            free(pp);
            free(stp);
            errmsg(SHERR_NOMEM,LOC("lex_paragraph"));
            return fpp;
        }
        pp->type->type = SYM_EOL;
        strcpy(pp->type->name,"\n");
        fpp->group = fpp->group_end = pp;
    }

    while (dont_expand = 1,(pp = lex_phrase(1,1)) != (struct phrase *)NULL)
    {   /* loop til eof or end of paragraph */
        if (pp->type == (struct token *)NULL)
            break;
        if (pp->io == (struct iotoken *)NULL &&
            pp->group == (struct phrase *)NULL &&
            pp->var == (struct token *)NULL &&
            pp->type->type == SYM_EOL && pp->body == (struct token *)NULL)
        {   /* ignore leading empty phrase */
            phrase_free(pp);
            continue;
        }
        if (fpp->group == (struct phrase *)NULL)
            fpp->group = fpp->group_end = pp;
        else
        {   /* tack onto end */
            fpp->group_end->next = pp;
            fpp->group_end = pp;
        }
        etp = pp->body;
        if (etp == (struct token *)NULL)
            etp = pp->type;
        if (etp == (struct token *)NULL)
            continue;
        if (strcmp(etp->name,cmd_brackets[ctr].end) == 0)
        {   /* got end of paragraph? */
            dont_expand = 0;
            if (fpp->io == (struct iotoken *)NULL &&
                pp->io != (struct iotoken *)NULL)
            {   /* move io redirection to good place */
                fpp->io = pp->io;
                fpp->io_end = pp->io_end;
                pp->io = pp->io_end = (struct iotoken *)NULL;
            }
            return fpp;
        }
    }
    errmsg(0,LOC("lex_paragraph"),"end of paragraph missing");

    if (pp != (struct phrase *)NULL)
        phrase_free(pp);
    dont_expand = 0;
    return lex_phrase(level,1);
}   /* end of lex_paragraph */

void lex_parens(level,fpp)
int level;
struct phrase *fpp;
{
    register struct phrase *pp;
    int doneflag;

    if (fpp->type == (struct token *)NULL || fpp->type->type != SYM_LPAREN)
        return;
    doneflag = 0;
    while ((pp = lex_phrase(level,1)) != (struct phrase *)NULL)
    {   /* get the parenthized phrases */
        if (pp->type == (struct token *)NULL)
            break;
        if (fpp->group == (struct phrase *)NULL)
            fpp->group = fpp->group_end = pp;
        else
        {   /* tack onto end */
            fpp->group_end->next = pp;
            fpp->group_end = pp;
        }
        if (doneflag)
            break;
        if (pp->type->type == SYM_RPAREN)
            doneflag++;
    }
    if (doneflag)
    {   /* if a parenthized group was found */
        if (pp->io != (struct iotoken *)NULL)
        {   /* move group i/o to front */
            if (fpp->io == (struct iotoken *)NULL)
            {   /* just move it */
                fpp->io = pp->io;
                fpp->io_end = pp->io_end;
            }
            else
            {   /* tack onto end */
                fpp->io_end->next = pp->io;
                fpp->io_end = pp->io_end;
            }
            pp->io = pp->io_end = (struct iotoken *)NULL;
        }
    }
}   /* end of lex_parens */

struct phrase *lex_phrase(level,expflag)
int level;
int expflag;
{
    register struct phrase *pp;
    register struct token *tp,*tp1;
    struct token *tp2;
    struct iotoken *iop;
    int save_expflag;

    pp = new_phrase();
    if (pp == (struct phrase *)NULL)
        return pp;

    save_expflag = dont_expand;
    dont_expand = expflag;
    while ((tp = lex_token(level)) != (struct token *)NULL)
    {   /* get token until end of phrase is reached */
        switch (tp->type)
        {   /* check for end of phrase or other special tokens */
            case SYM_DLPAREN:   /* start of arithmetic expression */
                do
                {   /* fill out rest of expression */
                    if (pp->body == (struct token *)NULL)
                    {   /* at start of list */
                        pp->body = pp->body_end = tp;
                    }
                    else
                    {   /* tack onto end */
                        pp->body_end->next = tp;
                        pp->body_end = tp;
                    }
                    if (tp->type == SYM_DRPAREN)
                        break;
                } while ((tp = lex_token(1)) != (struct token *)NULL);
                break;
            case SYM_DLBRACK:   /* start of conditional expression */
                do
                {   /* fill out rest of expression */
                    if (pp->body == (struct token *)NULL)
                    {   /* at start of list */
                        pp->body = pp->body_end = tp;
                    }
                    else
                    {   /* tack onto end */
                        pp->body_end->next = tp;
                        pp->body_end = tp;
                    }
                    if (tp->type == SYM_DRBRACK)
                        break;
                } while ((tp = lex_token(1)) != (struct token *)NULL);
                break;
            case SYM_LT:    /* various I/O redirection tokens */
            case SYM_LLT:
            case SYM_LLTL:
            case SYM_LBT:
            case SYM_RT:
            case SYM_RRT:
            case SYM_RBT:
                for (tp1 = pp->body; tp1 != (struct token *)NULL && tp1->next != pp->body_end; tp1 = tp1->next)
                    /* do nothing */;
                if (pp->body != (struct token *)NULL && pp->body_end->type == SYM_NUMBER)
                {   /* pick up file descriptor? */
                    tp2 = pp->body_end;
                    if (tp1 != (struct token *)NULL)
                    {   /* remove tail of list */
                        tp1->next = (struct token *)NULL;
                        pp->body_end = tp1;
                    }
                    else    pp->body = pp->body_end = (struct token *)NULL;
                }
                else    tp2 = (struct token *)NULL;
                tp1 = lex_token(level);
                if (tp1 == (struct token *)NULL)
                {   /* EOF or no memory? */
                    errmsg(0,LOC("lex_phrase"),"incomplete I/O redirection");
                    tp = tp1;
                    break;
                }
                iop = new_iotoken(strlen(tp1->name));
                if (iop == (struct iotoken *)NULL)
                {   /* enough memory */
                    errmsg(SHERR_NOMEM,LOC("lex_phrase"));
                }
                else
                {   /* fill in the ioblock */
                    iop->tempfile = (char *)NULL;
                    iop->type = tp->type;
                    strcpy(iop->file,tp1->name);
                    if (iop->type == SYM_RT ||
                        iop->type == SYM_RRT ||
                        iop->type == SYM_RBT)
                        iop->fd = 1;
                    else    iop->fd = 0;
                    if (tp2 != (struct token *)NULL)
                        iop->fd = atoi(tp2->name);
                    if (pp->io == (struct iotoken *)NULL)
                        pp->io = pp->io_end = iop;
                    else
                    {   /* tack onto end */
                        pp->io_end->next = iop;
                        pp->io_end = iop;
                    }
                    if (iop->type == SYM_LLT || iop->type == SYM_LLTL)
                        lex_heredoc_save(iop);
                }
                if (tp2 != (struct token *)NULL)
                    free(tp2);
                free(tp1);
                free(tp);
#ifdef  GEMDOS
                if (iop->fd > MAXUFD ||
                    ((iop->type == SYM_LBT || iop->type == SYM_RBT) &&
                    atoi(iop->file) > MAXUFD))
                {   /* are fds too big? */
                    errmsg(0,LOC("lex_phrase"),"can't redirect fds > %d in GEMDOS",MAXUFD);
                }
#endif  /* GEMDOS */
                break;
            case SYM_MARKER:    /* special boundary marker */
                pp->type = tp;
                dont_expand = save_expflag;
                return pp;
            case SYM_EOL:       /* end of line */
            case SYM_DSEMI:     /* double semicolon */
            case SYM_SEMI:      /* semicolon */
            case SYM_BACK:      /* background */
            case SYM_PIPE:      /* piping together */
            case SYM_ORIF:      /* or conditional */
            case SYM_ANDIF:     /* and conditional */
            case SYM_LPAREN:    /* start of group */
            case SYM_RPAREN:    /* end of group */
            case SYM_BQUOTE:    /* end of substitute group */
                pp->type = tp;
                if (tp->type == SYM_LPAREN)
                    lex_parens(level,pp);
                dont_expand = save_expflag;
                return lex_paragraph(level,pp);
            default:
                if (tp->type == SYM_KEYWORD &&
                   (pp->body == (struct token *)NULL || flag_keywords))
                {   /* define a variable? */
                    if (dont_expand)
                    {   /* save for later? */
                        if (pp->var == (struct token *)NULL)
                            pp->var = pp->var_end = tp;
                        else
                        {   /* tack onto end */
                            pp->var_end->next = tp;
                            pp->var_end = tp;
                        }
                    }
                    else
                    {   /* expand the variable */
                        var_define(tp->name,0,0);
                        free(tp);
                    }
                    break;
                }
                if (pp->body == (struct token *)NULL)
                {   /* at start of list */
                    pp->body = pp->body_end = tp;
                }
                else
                {   /* tack onto end */
                    pp->body_end->next = tp;
                    pp->body_end = tp;
                }
                break;
        }
        if (tp == (struct token *)NULL)
            break;
    }
    dont_expand = save_expflag;
    return pp;
}   /* end of lex_phrase */

struct phrase *lex_sentence(level)
int level;
{
    register struct phrase *sp;
    struct phrase *pp,*xpp;
    struct iotoken *iop;
    int pcount;

    sp = new_phrase();
    if (sp == (struct phrase *)NULL)
    {   /* enough memory? */
        errmsg(SHERR_NOMEM,LOC("lex_sentence"));
        return sp;
    }
    while ((pp = lex_phrase(level,1)) != (struct phrase *)NULL)
    {   /* fill out the sentence */
        if (sp->group == (struct phrase *)NULL &&
            pp->type != (struct token *)NULL &&
            pp->type->type == SYM_EOL &&
            pp->io == (struct iotoken *)NULL &&
            pp->var == (struct token *)NULL &&
            pp->body == (struct token *)NULL)
        {   /* ignore empty line */
            phrase_free(pp);
            continue;
        }
        if (sp->group_end != (struct phrase *)NULL &&
            sp->group_end->type != (struct token *)NULL &&
            sp->group_end->type->type == SYM_PIPE &&
            pp->type != (struct token *)NULL &&
            pp->type->type == SYM_EOL &&
            pp->body == (struct token *)NULL)
        {   /* reject this phrase, to finish pipe */
            level = 1;
            phrase_free(pp);
            continue;
        }
        if (sp->group == (struct phrase *)NULL)
            sp->group = sp->group_end = pp;
        else
        {   /* tack onto end */
            sp->group_end->next = pp;
            sp->group_end = pp;
        }
        if (pp->group != (struct phrase *)NULL)
            xpp = pp->group_end;
        else xpp = pp;
        if (xpp->type == (struct token *)NULL ||
            xpp->type->type == SYM_MARKER ||
            xpp->type->type == SYM_BQUOTE ||
            xpp->type->type == SYM_EOL)
        {   /* end of sentence? */
            pcount = 0;
            for (pp = sp->group; pp != (struct phrase *)NULL; pp = pp->next)
            {   /* look for, set up pipes */
                if ((pp->type != (struct token *)NULL &&
                    pp->type->type == SYM_PIPE) ||
                   (pp->group != (struct phrase *)NULL &&
                    pp->group_end->type != (struct token *)NULL &&
                    pp->group_end->type->type == SYM_PIPE))
                {   /* found a pipe */
                    iop = new_iotoken(16
#ifdef  GEMDOS
                                        +strlen(base_env.tmpdir)
#endif  /* GEMDOS */
                                                                                 );
                    if (iop == (struct iotoken *)NULL)
                    {   /* enough memory */
                        errmsg(SHERR_NOMEM,LOC("lex_sentence"));
                        break;
                    }
                    iop->fd = 1;
#ifdef  GEMDOS
                    sprintf(iop->file,"%spipe%d.tmp",base_env.tmpdir,pcount);
#else
                    sprintf(iop->file,"pipe%d",pcount);
#endif  /* GEMDOS */
                    iop->type = SYM_PIPE;
                    if (pp->io == (struct iotoken *)NULL)
                        pp->io = pp->io_end = iop;
                    else
                    {   /* stick on front */
                        iop->next = pp->io;
                        pp->io = iop;
                    }
                    if (pp->next == (struct phrase *)NULL)
                    {   /* something to pipe to */
                        errmsg(0,LOC("lex_sentence"),"missing pipe destination");
                        continue;
                    }
                    iop = new_iotoken(16
#ifdef  GEMDOS
                                                                  +strlen(base_env.tmpdir)
#endif  /* GEMDOS */
                                                                                 );
                    if (iop == (struct iotoken *)NULL)
                    {   /* enough memory */
                        errmsg(SHERR_NOMEM,LOC("lex_sentence"));
                        break;
                    }
                    iop->fd = 0;
#ifdef  GEMDOS
                    sprintf(iop->file,"%spipe%d.tmp",base_env.tmpdir,pcount);
#else
                    sprintf(iop->file,"pipe%d",pcount);
#endif  /* GEMDOS */
                    iop->type = SYM_PIPER;
                    if (pp->next->io == (struct iotoken *)NULL)
                        pp->next->io = pp->next->io_end = iop;
                    else
                    {   /* stick on front */
                        iop->next = pp->next->io;
                        pp->next->io = iop;
                    }
                }
                pcount++;
            }
            pp = sp->group_end;
            if (pp->type != (struct token *)NULL &&
                pp->type->type == SYM_BQUOTE)
            {   /* need to capture output? */
                iop = new_iotoken(16
#ifdef  GEMDOS
                                                              +strlen(base_env.tmpdir)
#endif  /* GEMDOS */
                                                                             );
                if (iop == (struct iotoken *)NULL)
                {   /* enough memory? */
                    errmsg(SHERR_NOMEM,LOC("lex_sentence"));
                }
                else
                {   /* fill out, tack on io block */
                    iop->fd = 1;
                    iop->type = SYM_RT;
#ifdef  GEMDOS
                    sprintf(iop->file,"%stemp%d.tmp",base_env.tmpdir,base_env.temp_count++);
#else
                    sprintf(iop->file,"/tmp/sh%d.temp",base_env.temp_count++);
#endif  /* GEMDOS */
                    if (pp->io == (struct iotoken *)NULL)
                        pp->io = pp->io_end = iop;
                    else
                    {   /* push onto front */
                        iop->next = pp->io;
                        pp->io = iop;
                    }
                }
            }
            return sp;
        }
    }
    free(sp);
    return (struct phrase *)NULL;
}   /* end of lex_sentence */

int lex_sempty(savep)
register struct strsave *savep;
{
    if (strings == (struct strsave *)NULL || strings == savep)
        return 1;
    if (strings->next == (struct strsave *)NULL && strings->ptr[0] == ' ' && strings->ptr[1] == '\0')
        return 1;
    if (strings->next == savep && strings->ptr[0] == ' ' && strings->ptr[1] == '\0')
        return 1;
    return 0;
}   /* end of lex_sempty */

struct token *lex_reparse_tokens(ftp,strip)
struct token *ftp;
int strip;
{
    register struct token *tp,*stp;
    struct token *ltp;
    register struct strsave *sp;
    struct strsave *savep,*osp,*lsp;

    savep = strings;
    for (tp = ftp; tp != (struct token *)NULL; tp = tp->next)
    {   /* go down initial token chain */
        if (io_pushtoken(tp->name,strip) < 0)
        {   /* did it work? */
            for (sp = strings; sp != savep; )
            {   /* eliminate any junk */
                osp = sp;
                sp = sp->next;
                if (osp->string != (char *)NULL)
                    free(osp->string);
                free(osp);
                strings = savep;
            }
            errmsg(0,LOC("lex_reparse_tokens"),"bad return");
            return (struct token *)NULL;
        }
    }
    osp = (struct strsave *)NULL;
    for (sp = strings; strings != savep && sp != (struct strsave *)NULL; )
    {   /* reverse leading part of list */
        lsp = sp;
        sp = sp->next;
        lsp->next = osp;
        osp = lsp;
        if (sp == savep)
            break;
    }
    if (osp != (struct strsave *)NULL)
    {   /* if some items were reversed */
        strings = osp;
        for (lsp = osp; lsp->next != (struct strsave *)NULL; lsp = lsp->next)
            /* do nothing */;
        lsp->next = savep;
    }

    stp = ltp = (struct token *)NULL;
    while (!lex_sempty(savep) && ((tp = lex_token(1)) != (struct token *)NULL))
    {   /* build the new chain */
        if (stp == (struct token *)NULL)
            stp = ltp = tp;
        else
        {   /* tack onto end */
            ltp->next = tp;
            ltp = tp;
        }
    }
    return stp;
}   /* end of lex_reparse_tokens */
