/*
 * Dynamic Memory Allocation Routines
 *
 *  malloc  - allocate a block of memory
 *      char *malloc(unsigned size);
 *  char *lmalloc(unsigned long size);
 *
 *  calloc  - allocate a block of memory, clearing it to zero first
 *      char *calloc(unsigned unitsize,unsigned nunits);
 *  char *lcalloc(unsigned long unitsize,unsigned long nunits);
 *
 *  realloc - reallocate a block of memory, making it smaller or larger
 *  char *realloc(char *ptr,unsigned size);
 *  char *lrealloc(char *ptr,unsigned long size);
 *
 *  free    - free a block of memory
 *      free(char *ptr);
 *
 * Dave Clemans, 1/88
 *
 * $Id: malloc.c,v 1.3 89/02/20 20:09:31 dclemans Exp $
 *
 * $Log:	malloc.c,v $
 * Revision 1.3  89/02/20  20:09:31  dclemans
 * Add RCS identifiers
 * 
 */
#include <stdio.h>
#ifndef unix
#include <osbind.h>
#endif

#ifdef  unix
extern char *sbrk();
#endif

extern char *malloc(),*lmalloc();
extern char *calloc(),*lcalloc();
extern char *realloc(),*lrealloc();
extern free();

/*
 * Set up the arena we work in
 */
#define HUNK    (18L*1024L)         /* basic hunk size for allocation */

struct memblock
{
    long    size;
    struct memblock *next;
};

static struct memblock *arena = (struct memblock *)NULL;
static struct memblock *arenaTail = (struct memblock *)NULL;
long poolSize = 0;
long mallocTotal = 0;
long mallocHighWater = 0;

/*
 * The user-visible routines are merely shells; they get memory
 * by calling the following routines.  They are declared as
 * "long *" because they guarantee returning a memory block
 * with a starting address suitable for storing a long.
 */
static long *memget();
static memput();

/*
 * Get a block of memory
 */
char *malloc(size)
unsigned size;
{
    return lmalloc((unsigned long)size);
}   /* end of malloc */

char *lmalloc(size)
unsigned long size;
{
    register long *mem;
    register long rsize;

    rsize = size + sizeof(long);
    rsize = ((rsize + (long)sizeof(long) - 1L) / (long)sizeof(long)) * (long)sizeof(long);
    mem = memget(rsize);
    if (mem == (long *)NULL)
        return (char *)NULL;
    *mem++ = rsize;
    mallocTotal += rsize;
    if (mallocTotal > mallocHighWater)
        mallocHighWater = mallocTotal;
    return (char *)mem;
}   /* end of lmalloc */

/*
 * Get a block of memory, clearing it to nulls
 */
char *calloc(unitsize,nunits)
unsigned unitsize;
unsigned nunits;
{
    return lcalloc((unsigned long)unitsize,(unsigned long)nunits);
}   /* end of calloc */

char *lcalloc(unitsize,nunits)
unsigned long unitsize;
unsigned long nunits;
{
    register long rsize,size;
    register char *ptr;
    register long *mem;

    size = (long)unitsize * (long)nunits;
    rsize = size + sizeof(long);
    rsize = ((rsize + (long)sizeof(long) - 1L) / (long)sizeof(long)) * (long)sizeof(long);
    mem = memget(rsize);
    ptr = (char *)mem;
    if (mem == (long *)NULL)
        return (char *)NULL;
    while (size--)
        *ptr++ = '\0';
    *mem++ = rsize;
    mallocTotal += rsize;
    if (mallocTotal > mallocHighWater)
        mallocHighWater = mallocTotal;
    return (char *)mem;
}   /* end of lcalloc */

/*
 * Reallocate a block of memory
 */
char *realloc(ptr,size)
char *ptr;
unsigned size;
{
    return lrealloc(ptr,(unsigned long)size);
}   /* end of realloc */

char *lrealloc(ptr,size)
char *ptr;
unsigned long size;
{
    long *mem;
    register long actual_size;
    register char *newptr;
    register char *to,*from;

    mem = (long *)ptr;
    size = ((size + (long)sizeof(long) - 1L) / (long)sizeof(long)) * (long)sizeof(long);
    actual_size = *--mem;

    if ((size + sizeof(long) + sizeof(struct memblock)) < actual_size)
    {   /* if the block is being shrunk */
        newptr = ptr + size;
        memput(newptr,actual_size - size - sizeof(long));
        mallocTotal -= (actual_size - size - sizeof(long));
        *mem = size + sizeof(long);
    }
    else
    {   /* the block is being expanded */
        newptr = (char *)memget(size + (long)sizeof(long));
        if (newptr == (char *)NULL)
            return newptr;
        mallocTotal += size + sizeof(long);
        if (mallocTotal > mallocHighWater)
            mallocHighWater = mallocTotal;
        actual_size -= sizeof(long);
        to = newptr;
        from = ptr;
        while (actual_size--)
            *to++ = *from++;
        free(ptr);
    }
    return newptr;
}   /* end of lrealloc */

/*
 * Free a block of memory
 */
free(ptr)
char *ptr;
{
    register long *mem;
    register long size;

    mem = (long *)ptr;
    size = *--mem;
    memput((long *)mem,size);
    mallocTotal -= size;
}   /* end of free */

/*
 * Finally! we reach the heart of this memory allocation package
 */

/*
 * Allocate a block of memory
 */
static long *memget(size)
long size;
{
    register struct memblock *mem,*memp;
    register char *fptr;
    register long toAlloc;
    long *ptr;

    if (size < sizeof(struct memblock))
        size = sizeof(struct memblock);

    memp = (struct memblock *)NULL;
    for (mem = arena; mem != (struct memblock *)NULL; mem = mem->next)
    {   /* look for a block with enough space */
        if (mem->size == size || mem->size >= (size + sizeof(struct memblock)))
        {   /* we've found some memory; unlink and return it */
            if (memp == (struct memblock *)NULL)
                arena = mem->next;
            else
                memp->next = mem->next;
            if (mem == arenaTail)
                arenaTail = memp;
            if (mem->size > size)
            {   /* if we need to split this block */
                fptr = (char *)mem;
                fptr += size;
                memput((long *)fptr,mem->size - size);
            }
            return (long *)mem;
        }
        memp = mem;
    }

    /* no block with enough space; try to get more memory from system */
    /* if that fails, we signal no memory */

    toAlloc = (size < HUNK) ? HUNK : (((size + HUNK - 1L) / HUNK) * HUNK);
#ifdef  unix
    ptr = (long *)sbrk(toAlloc);
#else
    ptr = (long *)Malloc(toAlloc);
#endif
    if (ptr == (long *)NULL)
        return (long *)NULL;
    poolSize += toAlloc;
    memput(ptr,toAlloc);
    return memget(size);
}   /* end of memget */

/*
 * Free a block of memory
 */
static memput(ptr,size)
long *ptr;
long size;
{
    register struct memblock *mem,*memp;
    register struct memblock *new;

    if (size < sizeof(struct memblock))
        size = sizeof(struct memblock);

    memp = (struct memblock *)NULL;
    for (mem = arena; mem != (struct memblock *)NULL; mem = mem->next)
    {   /* look for place to put this freed block */
        if (ptr < (long *)mem)
        {   /* then we put the block before this block */
            new = (struct memblock *)ptr;
            new->size = size;
            new->next = mem;
            if (mem == arena)
                arena = new;
            else
                memp->next = new;
            /* now try to compact new block backwards and forwards */
            if ((char *)new + size == (char *)mem)
            {   /* forwards compaction */
                new->size += mem->size;
                new->next = mem->next;
        if (arenaTail == mem)
            arenaTail = new;
            }
            if (memp != (struct memblock *)NULL &&
               ((char *)memp + memp->size == (char *)new))
            {   /* backwards compaction */
                memp->size += new->size;
                memp->next = new->next;
                if (arenaTail == new)
                    arenaTail = memp;
            }
            return;
        }
        memp = mem;
    }

    /* if reach here, just tack block to end of list */
    mem = (struct memblock *)ptr;
    mem->size = size;
    mem->next = (struct memblock *)NULL;
    if (arena == (struct memblock *)NULL)
        arena = arenaTail = mem;
    else
    {   /* tack onto end */
        /* try compacting first */
        if ((char *)arenaTail + arenaTail->size == (char *)mem)
            arenaTail->size += mem->size;
        else
        {   /* really tack onto end */
            arenaTail->next = mem;
            arenaTail = mem;
        }
    }
}   /* end of memput */
