/*
 * emacs		- an "emacs" style single line editor for reader
 *
 * Dave Clemans, 4/84 (initial version); 1/89 (more generality)
 *
 * Commands supported:
 *	kill line	users kill character
 *	erase char	users erase character
 *	next line	^N
 *	prev line	^P
 *	forward		^F
 *	backward	^B
 *	delete		^D
 *	beginning line	^A
 *	end of line	^E
 *	retype line	^L
 *	uppercase char	^C
 *	transpose chars	^T
 *	reverse search	^R
 *	forwards search	^S
 *	kill to end	^K
 *	mark		^@
 *	delete region	^W
 *	yank region	^Y
 *	quote char	^Q
 *	erase char	DEL
 *	kill next word	ESC d
 *	kill prev word	ESC ^H
 *	kill prev word	ESC DEL
 *	expand name	ESC ESC
 *	next word	ESC f
 *	prev word	ESC b
 *	uppercase char	ESC c
 *	uppercase word	ESC C
 *	list expand	ESC ^L
 *	mark		ESC SPACE
 *	copy region	ESC p
 *	yank region	ESC y
 *	delete line	ESC ^K
 *	delete line	ESC ^U
 *	switch mark	^X^X
 *	arg mult	^U
 *	set arg		ESC digits
 *
 * If we return 0, we did not handle the input character; it should be
 * handled by the standard editor.
 * If we return <0, signal end-of-file.
 * Otherwise the character has been
 * handled and the next character can be read.
 *
 * $Id: emacs.c,v 1.2 89/02/20 20:20:12 dclemans Exp $
 *
 * $Log:	emacs.c,v $
 * Revision 1.2  89/02/20  20:20:12  dclemans
 * Add RCS identifiers
 * 
 */
#include <ctype.h>
#ifndef GEMDOS
#include <sgtty.h>
#endif  GEMDOS

#include "history.h"

/*
 * A marked point in a line; corresponds to a cursor position
 */
static short int markedPoint = -1;

/*
 * For numerical arguments
 */
static int argument = 1;
static int argumentMultiplier = 1;

/*
 * Check for "emacs" style commands
 */
int _doEmacs(c)
register char c;
{
	register int counter,saveCursor;

#ifndef GEMDOS
	if (c == _savedState.basicAttributes.sg_erase)
	{	/* erase character */
		for (counter = 0; counter < argument*argumentMultiplier; counter++)
			_doStandardErase(c);
		argument = argumentMultiplier = 1;
		return(1);
	}
#endif  GEMDOS
	switch(c)
	{	/* check to see if we know this command */
#ifdef  GEMDOS
            case '\010':
#endif  GEMDOS
	    case '\177':	/* erase character */
		_doStandardErase(c);
		break;
	    case '\000':	/* mark current position */
		markedPoint = History.currentLine->cursor;
		break;
	    case '\016':	/* next line */
		for (counter = 0; counter < argument*argumentMultiplier; counter++)
			_doStandardNextLine(c);
		break;
	    case '\020':	/* prev line */
		for (counter = 0; counter < argument*argumentMultiplier; counter++)
			_doStandardPrevLine(c);
		break;
	    case '\006':	/* forward */
		for (counter = 0; counter < argument*argumentMultiplier; counter++)
			_doStandardSpace(c);
		break;
	    case '\002':	/* backward */
		for (counter = 0; counter < argument*argumentMultiplier; counter++)
			_doStandardBackspace(c);
		break;
	    case '\004':	/* delete */
		for (counter = 0; counter < argument*argumentMultiplier; counter++)
			_doStandardDelete(c);
		break;
	    case '\001':	/* beginning of line */
		_doStandardStartLine(c);
		break;
	    case '\005':	/* end of line */
		_doStandardEndLine(c);
		break;
	    case '\014':	/* retype line */
		_doStandardRetype(c);
		break;
	    case '\003':	/* capitalize character */
		_doStandardCapitalize(c);
		break;
	    case '\024':	/* transpose characters */
		_doStandardTranspose(c);
		break;
	    case '\022':	/* backwards search */
		_doStandardSrchPrevLine(c,0);
		break;
	    case '\023':	/* forwards search */
		_doStandardSearchNextLine(c,0);
		break;
	    case '\013':	/* kill to end of line */
		_doStandardDelEndLine(c);
		break;
	    case '\027':	/* delete region */
		doEmacsDeleteRegion();
		break;
	    case '\031':	/* yank region */
		_popOffKillRing();
		break;
	    case '\021':	/* quote character */
		if (_savedState.stream < 0)
			return(0);
		c = _readchar(_savedState.stream);
		if (c == EOF)
			return(-1);
		_doStandardCharacter(c);
		break;
	    case '\025':	/* set arg multiplier to 4 */
		argumentMultiplier = 4;
		return(1);
	    case '\033':	/* prefix character */
		if (_savedState.stream < 0)
			return(0);
		c = _readchar(_savedState.stream);
		if (c == EOF)
			return(-1);
		if (isdigit(c))
		{	/* supplying argument? */
			argument = 0;
			for (; isdigit(c); c = _readchar(_savedState.stream))
			{	/* build argument value */
				argument *= 10;
				argument += (int)((int)c - (int)'0');
			}
			if (c == EOF)
				return(-1);
			_savechar(c);
			return(1);
		}
		switch (c)
		{		/* a meta command? */
		    case 'd':	/* delete next word */
			_deleteNextStdWord();
			break;
		    case 'C':	/* capitalize next word */
			_capitalizeStdWord();
			break;
		    case 'c':	/* capitalize next char */
			_doStandardCapitalize(c);
			break;
		    case '\010': /* delete previous word */
		    case '\177': /* delete previous word */
			_deletePrevStdWord();
			break;
		    case '\033': /* expand command or filename */
			_doStandardExpand(c);
			break;
		    case 'f':	/* space forward over a word */
			_spaceNextStdWord();
			break;
		    case 'b':	/* space backward over a word */
			_spacePrevStdWord();
			break;
		    case '\014': /* list command or filename expansion possibilities */
			ExpandAName(History.currentLine->contents,
				sizeof(History.currentLine->contents)-1,
				History.currentLine->size, 0);
			break;
		    case ' ':	/* mark point */
			markedPoint = History.currentLine->cursor;
			break;
		    case 'p':	/* copy region */
			if (!isMarkValid())
			{	/* a good mark? */
				_writechar('\007');
				break;
			}
			_pushOnKillRing(markedPoint,History.currentLine->cursor);
			markedPoint = -1;
			break;
		    case 'y':	/* yank region */
			_popOffKillRing();
			break;
		    case '\013': /* kill line */
		    case '\025': /* kill line */
			_doStandardKill(c);
			break;
		    default:	/* unknown */
			_writechar('\007');
			break;
		}
		break;
	    case '\030':	/* other prefix character */
		if (_savedState.stream < 0)
			return(0);
		c = _readchar(_savedState.stream);
		if (c == EOF)
			return(-1);
		switch (c)
		{		/* a prefix command? */
		    case '\030':/* switch mark and point */
			if ((markedPoint < 0) || (markedPoint > History.currentLine->size))
			{	/* no mark */
				_writechar('\007');
				break;
			}
			saveCursor = History.currentLine->cursor;
			if (saveCursor < markedPoint)
			{	/* need to move forward */
				while (History.currentLine->cursor < markedPoint)
					_doStandardSpace('\000');
			}
			else if (saveCursor > markedPoint)
			{	/* need to move backward */
				while (History.currentLine->cursor > markedPoint)
					_doStandardBackspace('\000');
			}
			markedPoint = saveCursor;
			break;
		    default:	/* unknown */
			_writechar('\007');
			break;
		}
		break;
	    default:		/* we don't know this command */
		return(0);
	}
	argument = argumentMultiplier = 1;
	return(1);
};	/* end of _doEmacs */

/*
 * Delete region from mark to point
 */
static doEmacsDeleteRegion()
{
	register int counter,size;

	if (!isMarkValid())
	{	/* not marked */
		_writechar('\007');
		return;
	}

	_pushOnKillRing(markedPoint,History.currentLine->cursor);

	if (markedPoint < History.currentLine->cursor)
	{	/* a normal region */
		size = History.currentLine->cursor - markedPoint + 1;
		while (History.currentLine->cursor > markedPoint)
			_doStandardBackspace('\000');
		for (counter = 0; counter < size; counter++)
			_doStandardDelete('\000');
		markedPoint = -1;
		return;
	}

	size = markedPoint - History.currentLine->cursor + 1;
	for (counter = 0; counter < size; counter++)
		_doStandardDelete('\000');
	markedPoint = -1;
};	/* end of doEmacsDeleteRegion */

/*
 * Is the current mark valid? (i.e., does it exist and within the current
 * line?)
 */
static int isMarkValid()
{
	if (markedPoint < 0)
		return(0);
	if (History.currentLine->size < markedPoint)
		return(0);
	return(1);
};	/* end of isMarkValid */
