/*
 * Various editing subroutines for internal use by the "reader" package
 *
 * Dave Clemans, 4/84 (initial version); 1/89 (more generality)
 *
 * $Id: editing.c,v 1.2 89/02/20 20:20:06 dclemans Exp $
 *
 * $Log:	editing.c,v $
 * Revision 1.2  89/02/20  20:20:06  dclemans
 * Add RCS identifiers
 * 
 */
#include <ctype.h>
#ifndef GEMDOS
#include <sgtty.h>
#endif  GEMDOS
#ifndef LOC
#define LOC(r)  __FILE__,r,__LINE__
#endif  LOC

#include "history.h"

/*
 * Interesting function definitions
 */
extern char *substr();

/*
 * Is this character printable?  ...  i.e., would we want to print it
 * directly or generate a translation of it.
 */
static int printable(c)
register char c;
{
	if (isprint(c) || (c == '\n') || (c == '\r') || (c == '\t'))
		return(1);
	return(0);
};	/* end of printable */

/*
 * How much space will this character take up on the screen?
 */
static int charwidth(c,ttycursor,cursor,size)
register char c;
register int ttycursor;
{
	register int width;

	if (cursor == size)
		width = 1;
	else if (!printable(c))
		width = 2;
        else if (c == '\r')
                width = 0;
	else if (c != '\t')
		width = 1;
	else
	{	/* expand a tab; tab stop every 8 spaces */
		width = 1;
		while ((ttycursor+width) & 07)
			width++;
	}
	return(width);
};	/* end of charwidth */

/*
 * What is the on-screen width of the line up to given character?
 */
static int linewidth(index)
register int index;
{
	register int counter,width;

	width = 0;
	for (counter = 0; counter < index; counter++)
	{	/* find tty width of line before tab */
		width += charwidth(History.currentLine->contents[counter],width,
			History.currentLine->cursor,History.currentLine->size);
	}
	return(width);
};	/* end of linewidth */

/*
 * Go back a character on the display
 */
static backchar()
{
	register char c;
	register int counter,width;

	counter = History.currentLine->cursor-1;
	if (counter < 0)	/* already all the way back */
		return;
	c = History.currentLine->contents[counter];
	History.currentLine->cursor--;

        if (c != '\r')
		_writechar('\b');
	History.currentLine->ttycursor--;

	if (c != '\t')
	{	/* only tabs handled specially here */
		return;
	}

	width = linewidth(counter);

	while (History.currentLine->ttycursor & 07)	/* tab=8 spaces */
	{	/* go back to tab stop */
		if (width >= History.currentLine->ttycursor)
			break;
		_writechar('\b');
		History.currentLine->ttycursor--;
	}
};	/* end of backchar */

/*
 * Echo a character; if non-printing print as ^c
 */
_echochar(c)
register char c;
{
	History.currentLine->ttycursor++;
	if (printable(c))
	{	/* normal stuff for printable characters */
		if (c == '\t')
		{	/* special stuff for tab */
			_writechar(' ');
			while (History.currentLine->ttycursor & 07) /* tab=8 spaces */
			{	/* expand a tab */
				_writechar(' ');
				History.currentLine->ttycursor++;
			}
		}
		else if (c == '\r')
		    History.currentLine->ttycursor--;
       		else	_writechar(c);
	}
	else
	{	/* special handling for non-printables */
		_writechar('^');
		if (c == '\177')
			_writechar('?');
		else	_writechar((char)((int)c | 0100));
		History.currentLine->ttycursor++;
	}
};	/* end of _echochar */

/*
 * Do a standard character erase
 */
_doStandardErase(c)
char c;
{
	register int i,width;

	if (History.currentLine->cursor > 0)
	{	/* kill a character */
#ifndef GEMDOS
		if (_savedState.localAttributes & LCRTBS)
		{	/* crt style erase */
#endif  GEMDS
			i = 1;
			if (!printable(History.currentLine->contents[History.currentLine->cursor-1]))
				i = 2;
			for (; i > 0; --i)
			{	/* start erasing */
				backchar();
				_writechar(' ');
				_writechar('\b');
				if (i > 1)
					History.currentLine->cursor++;
			}
#ifndef GEMDOS
		}
		else if (_savedState.localAttributes & LPRTERA)
		{	/* paper style erase */
			_writechar('/');
			_echochar(History.currentLine->contents[History.currentLine->cursor-1]);
			_writechar('/');
		}
		else	backchar();
#endif  GEMDOS
		History.currentLine->cursor++;
		if (History.currentLine->cursor == History.currentLine->size)
		{	/* cursor at end of line */
			History.currentLine->size--;
			History.currentLine->cursor--;
		}
		else if (History.currentLine->cursor > History.currentLine->size)
		{	/* line buffer screwed up */
			errmsg(0,LOC("doStandardErase"),"line buffer screwed up");
			if (_savedState.stream >= 0)
				_terminate(_savedState.stream);
			_savedState.stream = -1;
			exit(-1);
		}
		else
		{	/* else take out from middle of line */
			strncpy(&History.currentLine->contents[History.currentLine->cursor-1],
				&History.currentLine->contents[History.currentLine->cursor],
				History.currentLine->size-History.currentLine->cursor);
			History.currentLine->cursor--;
			History.currentLine->size--;
			width = History.currentLine->ttycursor;
			_doStandardEndLine(c);
			_writechar(' ');
			_writechar('\b');
			while (History.currentLine->ttycursor > width)
				_doStandardBackspace(c);
		}
	}
};	/* end of _doStandardErase */

/*
 * Do a standard line kill
 */
_doStandardKill(c)
char c;
{
	register int i,j;
	char cc;

	_doStandardEndLine('\000');
#ifndef GEMDOS
	if (_savedState.localAttributes & LCRTERA)
	{	/* crt style kill */
#endif  GEMDOS
		for (i = History.currentLine->size; i > 0; i--)
		{	/* erase the line */
			j = 1;
			cc = History.currentLine->contents[i-1];
			if (!printable(cc))
				j = 2;
			for (; j > 0; --j)
			{	/* start killing */
				backchar();
				_writechar(' ');
				_writechar('\b');
				if (j > 1)
					History.currentLine->cursor++;
			}
		}
#ifndef GEMDOS
	}
	else if (_savedState.localAttributes & LPRTERA)
	{	/* printing style kill */
		if (c)
			_echochar(c);
		_writechar('\n');
	}
	else
	{	/* go to start of line */
		for (i = History.currentLine->size; i > 0; i--)
                {
			backchar();
                }
	}
#endif  GEMDOS
	History.currentLine->size = 0;
	History.currentLine->cursor = 0;
	History.currentLine->ttycursor = 0;
};	/* end of _doStandardKill */

/*
 * Retype the line
 */
_doStandardRetype(c)
char c;
{
	register int i,cursor;

	cursor = History.currentLine->cursor;
	if (c)
	{	/* if a user typed character */
		_doStandardEndLine(c);
		_echochar(c);
		_writechar('\n');
	}
	History.currentLine->ttycursor = 0;
	for (i = 0; i < History.currentLine->size; i++)
	{	/* reprint the line */
		_echochar(History.currentLine->contents[i]);
	}
	_doStandardStartLine(c);
	while (History.currentLine->cursor < cursor)
		_doStandardSpace(c);
};	/* end of _doStandardRetype */

/*
 * Do unambigous command/filename expansion for last word in current line
 */
_doStandardExpand(c)
char c;
{
	register int rc,i;

	History.currentLine->contents[History.currentLine->size] = '\0';
	rc = ExpandAName(History.currentLine->contents,
		sizeof(History.currentLine->contents)-1,
		History.currentLine->size, 1);
	i = History.currentLine->size;
	while (History.currentLine->contents[i])
	{	/* enter the characters in as input */
		_savechar(History.currentLine->contents[i++]);
	}
	if (rc != 1)
	{	/* if ambigous or unknown */
		_writechar('\007');
	}
};	/* end of _doStandardExpand */

/*
 * Insert or append a character to the current line
 */
_doStandardCharacter(c)
char c;
{
	register int counter,width;
	char buffer[HISTORY_BUFFER];

	if (History.currentLine->cursor == History.currentLine->size)
	{	/* append to line */
		_echochar(c);
		History.currentLine->contents[History.currentLine->size] = c;
		History.currentLine->size++;
		History.currentLine->cursor++;
	}
	else if (History.currentLine->cursor > History.currentLine->size)
	{	/* line buffer screwed up */
		errmsg(0,LOC("doStandardCharacter"),"line buffer screwed up");
		if (_savedState.stream >= 0)
			_terminate(_savedState.stream);
		_savedState.stream = -1;
		exit(-1);
	}
	else
	{	/* insert into line */
		strncpy(buffer,History.currentLine->contents,History.currentLine->cursor);
		buffer[History.currentLine->cursor] = c;
		strncpy(&buffer[History.currentLine->cursor+1],
			&History.currentLine->contents[History.currentLine->cursor],
			History.currentLine->size-History.currentLine->cursor);
		strncpy(History.currentLine->contents,buffer,History.currentLine->size+1);
		History.currentLine->size++;
		History.currentLine->cursor++;
		_echochar(c);
		width = History.currentLine->ttycursor;
		for (counter = History.currentLine->cursor; counter < History.currentLine->size; counter++)
		{	/* repaint rest of line */
			_echochar(History.currentLine->contents[counter]);
			History.currentLine->cursor++;
		}
		while (History.currentLine->ttycursor > width)
			_doStandardBackspace('\0');
	}
};	/* end of _doStandardCharacter */

/*
 * Delete next character in line
 */
_doStandardDelete(c)
char c;
{
	register int counter,width;

	if ((History.currentLine->cursor >= 0) &&
	    (History.currentLine->cursor < History.currentLine->size))
	{	/* something there to delete */
		width = linewidth(History.currentLine->cursor);
		strncpy(&History.currentLine->contents[History.currentLine->cursor],
			&History.currentLine->contents[History.currentLine->cursor+1],
			History.currentLine->size-History.currentLine->cursor-1);
		History.currentLine->size--;
		width = History.currentLine->ttycursor;
		_doStandardEndLine(c);
		_writechar(' ');
		_writechar('\b');
		while (History.currentLine->ttycursor > width)
			_doStandardBackspace(c);
	}
};	/* end of _doStandardDelete */

/*
 * Move the cursor back
 */
_doStandardBackspace(c)
char c;
{
	if (History.currentLine->cursor > 0)
	{	/* if something to backspace over */
		backchar();
	}
};	/* end of _doStandardBackspace */

/*
 * Move the cursor forward
 */
_doStandardSpace(c)
char c;
{
	if (History.currentLine->cursor < History.currentLine->size)
	{	/* if something to go forward over */
		_echochar(History.currentLine->contents[History.currentLine->cursor]);
		History.currentLine->cursor++;
	}
};	/* end of _doStandardSpace */

/*
 * Get an older line from the history list and clobber the current line
 * in the history list with it.
 */
_doStandardPrevLine(c)
char c;
{
	register struct historyLine *hp;

	hp = History.listPointer->next;
	if (hp && hp->command)
	{	/* there's an older line there */
		_doStandardKill('\0');
		strncpy(History.currentLine->contents,hp->contents,hp->size);
		History.currentLine->size = hp->size-1;
		History.currentLine->cursor = hp->size-1;
		_doStandardRetype('\0');
		History.listPointer = hp;
	}
};	/* end of _doStandardPrevLine */

/*
 * Get a newer line from the history list and clobber the current line
 * in the history list with it.
 */
_doStandardNextLine(c)
char c;
{
	register struct historyLine *hp;

	for (hp = History.listTop; hp && (hp->next != History.listPointer); hp = hp->next)
		continue;
	if (hp && (hp != History.currentLine))
	{	/* if a newer line exists */
		_doStandardKill('\0');
		strncpy(History.currentLine->contents,hp->contents,hp->size);
		History.currentLine->size = hp->size-1;
		History.currentLine->cursor = hp->size-1;
		_doStandardRetype('\0');
		History.listPointer = hp;
	}
};	/* end of _doStandardNextLine */

/*
 * Goto the beginning of the line
 */
_doStandardStartLine(c)
char c;
{
	while (History.currentLine->cursor > 0)
	{	/* while stuff to space over */
		backchar();
	}
};	/* end of _doStandardStartLine */


/*
 * Goto the end of the line
 */
_doStandardEndLine(c)
char c;
{
	while (History.currentLine->cursor < History.currentLine->size)
	{	/* if something to go forward over */
		_echochar(History.currentLine->contents[History.currentLine->cursor]);
		History.currentLine->cursor++;
	}
};	/* end of _doStandardEndLine */

/*
 * Capitalize the current character
 */
_doStandardCapitalize(c)
char c;
{
	register int cursor;

	cursor = History.currentLine->cursor;
	if (cursor == History.currentLine->size)
		return;		/* no character at end of line */
	if (islower(History.currentLine->contents[cursor]))
		History.currentLine->contents[cursor] =
		    toupper(History.currentLine->contents[cursor]);
	_echochar(History.currentLine->contents[cursor]);
	History.currentLine->cursor++;
	backchar();
	History.currentLine->cursor--;
};	/* end of _doStandardCapitalize */

/*
 * Transpose the current and next characters
 */
_doStandardTranspose(c)
char c;
{
	register char cc;
	register int cursor;

	cursor = History.currentLine->cursor;
	if (cursor == History.currentLine->size)
		return;		/* nothing to transpose */
	cc = History.currentLine->contents[cursor];
	if (cursor+1 < History.currentLine->size)
	{	/* if room for two chars */
		History.currentLine->contents[cursor]=
			History.currentLine->contents[cursor+1];
		History.currentLine->contents[cursor+1] = cc;
		_echochar(History.currentLine->contents[cursor]);
		History.currentLine->cursor++;
		_echochar(cc);
		backchar();
		History.currentLine->cursor--;
		backchar();
	}
	else
	{	/* else no; treat end of line like space */
		History.currentLine->contents[cursor] = ' ';
		_echochar(' ');
		History.currentLine->cursor++;
		_doStandardCharacter(cc);
		_doStandardBackspace(c);
		backchar();
		History.currentLine->cursor--;
	}
};	/* end of _doStandardTranspose */

/*
 * Search lines backwards in time for a string.
 * If found, set global History.listPointer to point to it.
 * The search starts at History.listPointer.
 * If string not found, History.listPointer is not changed.
 * Returns index of start of search string if found, -1 if not found.
 */
static int backsearchfor(string)
char *string;
{
	register struct historyLine *hp;
	register char *cp;

	cp = (char *)NULL;
	for (hp = History.listPointer->next; hp; hp = hp->next)
	{	/* start searching */
		hp->contents[hp->size] = '\0';
		cp = substr(hp->contents,string);
		if (cp)
			break;
	}
	if (!cp || !hp)
		return(-1);
	History.listPointer = hp;
	return((int)cp - (int)hp->contents);
};	/* end of backsearchfor */

/*
 * Search lines forwards in time for a string.
 * If found, set global History.listPointer to point to it.
 * The search starts at History.listPointer.
 * If string not found, History.listPointer is not changed.
 * Returns index of start of search string if found, -1 if not found.
 */
static int searchfor(string)
char *string;
{
	register struct historyLine *hp,*bp;
	register char *cp;

	cp = (char *)NULL;
	for (hp = History.listPointer; hp && (hp != History.listTop); )
	{	/* start searching */
		for (bp = History.listTop; bp && (bp->next != hp); bp = bp->next)
			continue;
		if (!bp || (bp == History.listTop))
			break;
		hp = bp;
		hp->contents[hp->size] = '\0';
		cp = substr(hp->contents,string);
		if (cp)
			break;
	}
	if (!cp || !hp)
		return(-1);
	History.listPointer = hp;
	return((int)cp - (int)hp->contents);
};	/* end of searchfor */

/*
 * Simple read a line routine; just turn on normal tty mode; read a
 * line with normal tty editing; and put the tty modes back.
 */
static readline(buffer,size)
register char *buffer;
register int size;
{
	register int cursor;
	register int c;
#ifndef GEMDOS
	struct sgttyb ttymodes;
#endif  GEMDOS

	buffer[0] = '\0';
	if (_savedState.stream < 0)
		return;
#ifndef GEMDOS
	ttymodes = _savedState.basicAttributes;
	ttymodes.sg_flags &= ~ECHO;
	ioctl(_savedState.stream, TIOCSETP, &ttymodes);
#endif  GEMDOS
	for (cursor = 0; cursor < size; )
	{	/* start filling the buffer till cr or nl */
		c = _readchar(_savedState.stream);
		if ((c == '\n') || (c == '\r') || (c == EOF))
			break;
		buffer[cursor++] = c;
	}
	buffer[cursor] = '\0';
#ifndef GEMDOS
	ioctl(_savedState.stream, TIOCSETP, &_newState.basicAttributes);
#endif  GEMDOS
};	/* end of readline */

static char searchString[HISTORY_BUFFER];

/*
 * Search for older lines from the history list and clobber the current line
 * in the history list with it.
 */
_doStandardSrchPrevLine(c,flag)
char c;
int flag;
{
	register struct historyLine *hp;
	register int cursor;
	char search[HISTORY_BUFFER];

	search[0] = '\0';
	if (!flag)
		readline(search,sizeof search-1);
	if (!search[0])
		strcpy(search,searchString);
	if (!search[0])
	{	/* nothing to search for */
		_writechar('\007');
		return;
	}
	cursor = backsearchfor(search);
	if (cursor < 0)
		return;
	hp = History.listPointer;
	if (hp)
	{	/* there's an older line there */
		_doStandardKill('\0');
		strncpy(History.currentLine->contents,hp->contents,hp->size);
		History.currentLine->size = hp->size-1;
		History.currentLine->cursor = hp->size-1;
		_doStandardRetype('\0');
		History.listPointer = hp;
	}
	strcpy(searchString,search);
};	/* end of _doStandardSrchPrevLine */

/*
 * Search for newer lines from the history list and clobber the current line
 * in the history list with it.
 */
_doStandardSearchNextLine(c,flag)
char c;
int flag;
{
	register struct historyLine *hp;
	register int cursor;
	char search[HISTORY_BUFFER];

	search[0] = '\0';
	if (!flag)
		readline(search,sizeof search-1);
	if (!search[0])
		strcpy(search,searchString);
	if (!search[0])
	{	/* nothing to search for */
		_writechar('\007');
		return;
	}
	cursor = searchfor(search);
	if (cursor < 0)
		return;
	hp = History.listPointer;
	if (hp)
	{	/* if a newer line exists */
		_doStandardKill('\0');
		strncpy(History.currentLine->contents,hp->contents,hp->size);
		History.currentLine->size = hp->size-1;
		History.currentLine->cursor = hp->size-1;
		_doStandardRetype('\0');
		History.listPointer = hp;
	}
	strcpy(searchString,search);
};	/* end of _doStandardSearchNextLine */

/*
 * move to a given position in the current line
 */
static moveto(position)
register int position;
{
	if (position < 0)
		position = 0;
	else if (position > History.currentLine->size)
		position = History.currentLine->size;
	if (History.currentLine->cursor > position)
	{	/* move backwards */
		while (History.currentLine->cursor > position)
			_doStandardBackspace('\0');
	}
	else if (History.currentLine->cursor < position)
	{	/* move forwards */
		while (History.currentLine->cursor < position)
			_doStandardSpace('\0');
	}
};	/* end of moveto */

/*
 * delete a given range of characters
 */
_deleteTo(start,stop)
register int start,stop;
{
	register int count;
	register int savecursor;

	if (start < 0)
		start = 0;
	else if (start > History.currentLine->size)
		start = History.currentLine->size;
	if (stop < 0)
		stop = 0;
	else if (stop > History.currentLine->size)
		stop = History.currentLine->size;
	savecursor = History.currentLine->cursor;

	_pushOnKillRing(start,stop);

	count = stop-start+1;
	if (count <= 0)
		return;

	moveto(start);
	for (; count > 0; --count)
	{	/* start deleting characters */
		_doStandardDelete('\0');
	}
	moveto(savecursor);
};	/* end of _deleteTo */

/*
 * Delete from current position to end of line
 */
_doStandardDelEndLine(c)
char c;
{
	_deleteTo(History.currentLine->cursor,History.currentLine->size);
};	/* end of _doStandardDelEndLine */
