/*
 * Command Input Shell
 * Dave Clemans
 * 12/88-1/89
 *
 * "spiritually" based on Bourne, Korn shells
 *
 * Signal handling; the "kill" built-in; the "jobs" built-in
 *
 * $Id: trap.c,v 1.9 89/03/05 15:38:55 dclemans Exp $
 *
 * $Log:	trap.c,v $
 * Revision 1.9  89/03/05  15:38:55  dclemans
 * more fixes
 * 
 * Revision 1.8  89/02/23  08:08:42  dclemans
 * make kill work with background jobs list
 * 
 * Revision 1.7  89/02/22  21:32:04  dclemans
 * Implement simple background job monitoring facility
 * 
 * Revision 1.6  89/02/21  20:37:29  dclemans
 * Fix bug with signal_map on systems with varying
 * numbers of signals
 * 
 * Revision 1.5  89/02/21  20:30:00  dclemans
 * Fix bug with shell variable references in history lists.
 * 
 * Revision 1.4  89/02/21  08:36:51  dclemans
 * Implement pseudo-signals ERR, EXIT, DEBUG
 * 
 * Revision 1.3  89/02/20  22:54:24  dclemans
 * correct signal names
 * 
 * Revision 1.2  89/02/20  22:49:11  dclemans
 * bug fix
 * implement kill -l
 * 
 * Revision 1.1  89/02/20  22:27:23  dclemans
 * Initial revision
 * 
 */
#include <stdio.h>
#include "shell.h"

#ifndef GEMDOS
#include <signal.h>
#else
#define NSIG    0
#endif  /* GEMDOS */

static char *signal_map[] =
{
    "EXIT",
#ifndef GEMDOS
    "HUP",
    "INT",
    "QUIT",
    "ILL",
    "TRAP",
    "IOT",
    "EMT",
    "FPE",
    "KILL",
    "BUS",
    "SEGV",
    "SYS",
    "PIPE",
    "ALRM",
    "TERM",
    "USR1",
#if NSIG > 16
    "USR2",
    "CLD",
    "",
    "STOP",
    "TSTP",
    "",
    "CHLD",
    "TTIN",
    "TTOU",
    "TINT",
    "XCPU",
    "XFSZ",
    "VTALRM",
    "PROF",
    "URG",
    "WINCH",
#endif
#endif  /* GEMDOS */
    "ERR",
    "DEBUG",
    (char *)NULL
};

struct  signal_record
{
    struct  phrase *action;
    int     status;
};
#define IGNORED 0x0001
#define ACTION  0x0002
static struct signal_record signals[NSIG+3];

static int signal_number(sig)
register char *sig;
{
    register int i;
    register char *cp;
    char buffer[16];

    for (i = 0; signal_map[i] != (char *)NULL; i++)
    {   /* look for a match */
        if (strlen(signal_map[i]) == 0)
            continue;
        if (strcmp(sig,signal_map[i]) == 0)
            return i;
        strcpy(buffer,signal_map[i]);
        for (cp = buffer; *cp; cp++)
            if (isupper(*cp))
                *cp = _tolower(*cp);
        if (strcmp(sig,buffer) == 0)
            return i;
    }
    return -1;
}   /* end of signal_name */

int signal_handler(sig)
int sig;
{
    register struct signal_record *this_sig;
    char buffer[256];

    if (sig >= 1 || sig <= NSIG)
    {   /* need to reset the signal? */
#ifndef GEMDOS
#ifndef USG
        /* don't need anything here; BSD semantics don't need resetting */
#else
        signal(sig,signal_handler);
#endif  /* USG */
#endif  /* GEMDOS */
    }
    if (sig < 0 || sig >= NSIG+3)
        return -1;
    this_sig = &signals[sig];
    if (this_sig->status & IGNORED)
        return 0;
    if (this_sig->status & ACTION)
    {   /* execute the specified phrase */
        if (this_sig->action != (struct phrase *)NULL)
            exec_phrase(this_sig->action,0);
        return 0;
    }
    if (sig == 0 || sig > NSIG)
        return 0;
    errmsg(0,LOC("signal_handler"),"received signal #%d; exiting.",sig);
    cmd_forceexit = 1;
    return 0;
}   /* end of signal_handler */

void force_signal(name)
char *name;
{
    register int sig;

    sig = signal_number(name);
    if (sig < 0)
    {   /* if unknown signal */
        errmsg(0,LOC("force_signal"),"unable to decode pseudo-signal name %s",name);
        return;
    }
    signal_handler(sig);
}   /* end of force_signal */

int signal_init()
{
#ifdef  never
    /* don't know what to do here yet... don't want to hit every signal */
    register int i;
#ifndef GEMDOS
#ifndef USG
    struct sigvec handler;
#endif  /* USG */
#endif  /* GEMDOS */

#ifndef GEMDOS
#ifndef USG
    handler.sv_handler = signal_handler;
    handler.sv_onstack = 0;
    for (i = 1; i <= NSIG; i++)
    {   /* set up the appropriate handlers */
        handler.sv_mask = sigmask(i);
        sigvec(i,&handler,(struct sigvec *)NULL);
    }
#else
    for (i = 1; i <= NSIG; i++)
        signal(i,signal_handler);
#endif  /* USG */
#endif  /* GEMDOS */
#endif  /* never */
}   /* end of signal_init */

int cmd_trap(pp)
struct phrase *pp;
{
    register struct token *tp;
    register int sig;
    struct phrase *torun,*toend,*np;
    char *cp;
    char tempfile[64];
    struct token *action;
    int status,fd;
#ifndef GEMDOS
#ifndef USG
    struct sigvec handler;
#endif  /* USG */
#endif  /* GEMDOS */

    if (pp->body->next == (struct token *)NULL)
    {   /* any parms? */
        errmsg(0,LOC("cmd_trap"),"missing action or condition args");
        return -1;
    }
    action = pp->body->next;
    tp = action->next;
    if (tp == (struct token *)NULL)
    {   /* any conditions? */
        errmsg(0,LOC("cmd_trap"),"missing condition args");
        return -1;
    }
    if (strlen(action->name) == 0)
    {   /* ignore signal action? */
        action = (struct token *)NULL;
        status = IGNORED;
    }
    else if (strcmp(action->name,"-") == 0)
    {   /* reset signal action? */
        action = (struct token *)NULL;
        status = 0;
    }
    else status = ACTION;
    if (action->type == SYM_NUMBER || signal_number(action->name) >= 0)
    {   /* was a action REALLY specified? */
        tp = action;
        status = 0;
        action = (struct token *)NULL;
    }
    torun = (struct phrase *)NULL;
    if (action != (struct token *)NULL)
    {   /* need to reparse tokens into an executable phrase */
        cp = strcopy(action->name);
        if (cp == (char *)NULL)
        {   /* enough memory? */
            /* message already printed */
            return -1;
        }
        stripquotes(cp);
#ifdef  GEMDOS
        sprintf(tempfile,"%stemp%d.tmp",base_env.tmpdir,base_env.temp_count++);
#else
        sprintf(tempfile,"/tmp/sh%d.temp",base_env.temp_count++);
#endif  /* GEMDOS */
        fd = creat(tempfile,0666);
        if (fd < 0)
        {   /* did we get a temporary file? */
            errmsg(0,LOC("cmd_trap"),"unable to open tempfile");
            return 1;
        }
        write(fd,cp,strlen(cp));
        write(fd,"\n",1);
        close(fd);
        free(cp);

        if (io_pushfile(tempfile,1,0,1) < 0)
        {   /* if saving the file failed */
            errmsg(0,LOC("cmd_trap"),"unable to put tempfile into input stream");
            delete_later(tempfile);
            return 1;
        }
        torun = toend = (struct phrase *)NULL;
        while ((np = lex_phrase(1,0)) != (struct phrase *)NULL)
        {   /* get back contents of file */
            if (np->type != (struct token *)NULL && np->type->type == SYM_MARKER)
            {   /* reached end of file */
                phrase_free(np);
                break;
            }
            if (torun == (struct phrase *)NULL)
                torun = toend = np;
            else
            {   /* tack onto end */
                toend->next = np;
                toend = np;
            }
        }
    }

    while (tp != (struct token *)NULL)
    {   /* while there is a condition we need to set */
        if (tp->type == SYM_NUMBER)
            sig = atoi(tp->name);
        else sig = signal_number(tp->name);
        if (sig < 0 || sig >= NSIG+3)
        {   /* good signal number? */
            errmsg(0,LOC("cmd_trap"),"bad signal number %s=%d",tp->name,sig);
            tp = tp->next;
            continue;
        }
        if (signals[sig].action != (struct phrase *)NULL)
        {   /* free old action? */
            phrase_free(signals[sig].action);
            signals[sig].action = (struct phrase *)NULL;
        }
        switch (status)
        {   /* what to do... */
            case ACTION:
#ifndef GEMDOS
                if (sig >= 1 && sig <= NSIG)
                {   /* if a valid system signal */
#ifndef USG
                    handler.sv_handler = signal_handler;
                    handler.sv_mask = sigmask(sig);
                    handler.sv_onstack = 0;
                    sigvec(sig,&handler,(struct sigvec *)NULL);
#else
                    signal(sig,signal_handler);
#endif  /* USG */
                }
#endif  /* GEMDOS */
                signals[sig].status = ACTION;
                if (torun != (struct phrase *)NULL)
                    signals[sig].action = copy_phrase(torun,0,0,1);
                break;
            case IGNORED:
#ifndef GEMDOS
                if (sig >= 1 && sig <= NSIG)
                {   /* if a valid system signal */
#ifndef USG
                    handler.sv_handler = SIG_IGN;
                    handler.sv_mask = sigmask(sig);
                    handler.sv_onstack = 0;
                    sigvec(sig,&handler,(struct sigvec *)NULL);
#else
                    signal(sig,SIG_IGN);
#endif  /* USG */
                }
#endif  /* GEMDOS */
                signals[sig].status = IGNORED;
                break;
            default:
#ifndef GEMDOS
                if (sig >= 1 && sig <= NSIG)
                {   /* if a valid system signal */
#ifndef USG
                    handler.sv_handler = SIG_DFL;
                    handler.sv_mask = sigmask(sig);
                    handler.sv_onstack = 0;
                    sigvec(sig,&handler,(struct sigvec *)NULL);
#else
                    signal(sig,SIG_DFL);
#endif  /* USG */
                }
#endif  /* GEMDOS */
                signals[sig].status = 0;
                break;
        }
        tp = tp->next;
    }
    if (torun != (struct phrase *)NULL)
        phrase_free(torun);

    return 0;
}   /* end of cmd_trap */

#ifndef GEMDOS
int cmd_kill(pp)
struct phrase *pp;
{
    register struct token *tp;
    register int sig;
    register char *p;
    struct bg_job *job;
    struct procs *cmd;
    char *savep;
    int i,ncol,maxlength;
    char buffer[BUFSIZ];

    tp = pp->body->next;
    if (tp == (struct token *)NULL)
    {   /* anything to do? */
        errmsg(0,LOC("cmd_kill"),"no args specified");
        return -1;
    }
    if (strcmp(tp->name,"-l") == 0 || strcmp(tp->name,"-L") == 0)
    {   /* list out known signal names */
        maxlength = 0;
        for (sig = 1; sig <= NSIG; sig++)
            if (strlen(signal_map[sig]) > maxlength)
                maxlength = strlen(signal_map[sig]);
        maxlength += 7;     /* for column sep, etc. */
        ncol = base_env.columns / maxlength;
        if ((NSIG < (base_env.lines*2/3)) || (ncol < 1))
        {   /* just a single list? */
            for (sig = 1; sig <= NSIG; sig++)
            {   /* print the menu */
                sprintf(buffer,"%d) %s\n",sig,signal_map[sig]);
                io_writestring(0,buffer);
            }
        }
        else
        {   /* a multi-column list? */
            for (sig = 1; sig <= NSIG; )
            {   /* build lines... */
                p = buffer;
                *p = '\0';
                for (i = 0; i < ncol && (sig+i) <= NSIG; i++)
                {   /* for each column */
                    sprintf(p,"%d) %s",sig+i,signal_map[sig+i]);
                    savep = p;
                    while (*p)
                        p++;
                    while ((int)(p - savep) < maxlength)
                        *p++ = ' ';
                    *p = '\0';
                }
                sig += ncol;
                strcpy(p,"\n");
                io_writestring(0,buffer);
            }
        }
        return 0;
    }
    if (tp->name[0] != '-')
        sig = SIGTERM;
    else
    {   /* decode the value */
        sig = atoi(&tp->name[1]);
        if (sig < 1 || sig > NSIG)
            sig = signal_number(&tp->name[1]);
        if (sig < 1 || sig > NSIG)
        {   /* a good value? */
            errmsg(0,LOC("cmd_kill"),"bad signal number %d; range is 1 <-> %d",sig,NSIG);
            return 1;
        }
        tp = tp->next;
    }
    for (; tp != (struct token *)NULL; tp = tp->next)
    {   /* send sig to the specified pids */
        if (tp->name[0] == '%')
        {   /* check in pending jobs list? */
            i = 1;
            for (job = base_env.jobs; job != (struct bg_job *)NULL; job = job->next)
            {   /* found what we are looking for? */
                if (i == atoi(&tp->name[1]))
                    break;
                i++;
            }
            if (job == (struct bg_job *)NULL)
            {   /* if we didn't find the job */
                errmsg(0,LOC("cmd_kill"),"job %s doesn't exist",tp->name);
                continue;
            }
            for (cmd = job->cmds; cmd != (struct procs *)NULL; cmd = cmd->next)
            {   /* kill each process involved in job */
                if (kill(cmd->pid,sig) < 0)
                    errmsg(0,LOC("cmd_kill"),"unable to send signal %d to process %d",sig,cmd->pid);
            }
            continue;
        }
        if (kill(atoi(tp->name),sig) < 0)
            errmsg(0,LOC("cmd_kill"),"unable to send signal %d to process %s",sig,tp->name);
    }

    return 0;
}   /* end of cmd_kill */

void cmd_jobs_dump(curr,num,stat)
struct phrase *curr;
int num;
char *stat;
{
    char buffer[33];
    register struct phrase *pp;

    if (curr == (struct phrase *)NULL)
        return;
    sprintf(buffer,"[%d] + %s",num,stat);
    while (strlen(buffer) < 32)
        strcat(buffer," ");
    io_writestring(0,buffer);
    for (pp = curr; pp != (struct phrase *)NULL; pp = pp->next)
    {   /* dump out the lines */
        phrase_dump(pp,1,32);
        if (pp->next != (struct phrase *)NULL)
            io_writestring(0,"\n                                ");
    }
}   /* end of cmd_jobs_dump */

int cmd_jobs()
{
    register int cmd;
    register struct bg_job *job;

    exec_waitfor();
    cmd = 1;
    for (job = base_env.jobs; job != (struct bg_job *)NULL; job = job->next)
    {   /* step through pending jobs */
        cmd_jobs_dump(job->job,cmd,"Running");
        cmd++;
    }
}   /* end of cmd_jobs */
#endif  /* GEMDOS */
