/*
 * Command Input Shell
 * Dave Clemans
 * 12/88-1/89
 *
 * "spiritually" based on Bourne, Korn shells
 *
 * Command Execution
 *
 * $Id: exec.c,v 1.11 89/03/05 15:38:22 dclemans Exp $
 *
 * $Log:	exec.c,v $
 * Revision 1.11  89/03/05  15:38:22  dclemans
 * more fixes
 * 
 * Revision 1.10  89/02/25  17:40:03  dclemans
 * miscellaneous bug fixes/speedups
 * 
 * Revision 1.9  89/02/22  21:31:56  dclemans
 * Implement simple background job monitoring facility
 * 
 * Revision 1.8  89/02/22  13:19:03  dclemans
 * Implement $!
 * 
 * Revision 1.7  89/02/22  10:16:05  dclemans
 * Fix bugs with process waiting, pid reporting, parsing of &
 * 
 * Revision 1.6  89/02/21  20:29:52  dclemans
 * Fix bug with shell variable references in history lists.
 * 
 * Revision 1.5  89/02/21  08:36:36  dclemans
 * Implement pseudo-signals ERR, EXIT, DEBUG
 * 
 * Revision 1.4  89/02/20  20:14:25  dclemans
 * Add RCS identifiers
 * 
 */
#include <stdio.h>
#include "shell.h"
#ifndef GEMDOS
#include <errno.h>
#include <sys/types.h>
#ifndef USG
#include <sys/wait.h>
#include <sys/time.h>
#include <sys/resource.h>
#include <sys/file.h>
#else
#include <time.h>
#include <fcntl.h>
#endif  /* USG */
#endif  /* GEMDOS */

#ifdef  MWC_ARGV
static char mwc_iovec[] = "????????????????????????????";
#endif  /* MWC_ARGV */

#ifndef GEMDOS
static struct procs *waitStack = (struct procs *)NULL;
static struct procs *pipeStack = (struct procs *)NULL;
static struct procs *pipeStack_end;
static struct procs *bgJobs = (struct procs *)NULL;
#endif  /* GEMDOS */

int exec_builtin(cmd,pp,iosetup)
char *cmd;
struct phrase *pp;
int iosetup;
{
    register int i,rc;

    if (pp->body == (struct token *)NULL)
        return -1;
    for (i = 0; cmd_builtin[i].name != (char *)NULL; i++)
    {   /* is this a builtin command? */
        rc = strcmp(cmd,cmd_builtin[i].name);
        if (rc > 0)
            continue;
        if (rc < 0)
            return 0;
        if (!iosetup)
        {   /* make our own environment? */
            if (main_iopush() < 0)
            {   /* if couldn't save i/o env */
                /* message already printed */
                main_iopop();
                cmd_lastrc = -1;
                return -1;
            }
        }
        cmd_lastrc = (*cmd_builtin[i].cmd)(pp);
        if (!iosetup)
            main_iopop();
        return 1;
    }
    return 0;
}   /* end of exec_builtin */

struct function *func_get(name)
char *name;
{
    register struct function *sfp;
    register int rc;

    for (sfp = base_env.func_table; sfp != (struct function *)NULL; )
    {   /* look for where to put the function */
        rc = strcmp(name,sfp->name);
        if (rc == 0)
        {   /* found it */
            return sfp;
        }
        else if (rc < 0)
        {   /* go down the left side? */
            if (sfp->left == (struct function *)NULL)
                break;
            sfp = sfp->left;
        }
        else
        {   /* go down right side */
            if (sfp->right == (struct function *)NULL)
                break;
            sfp = sfp->right;
        }
    }
    return (struct function *)NULL;
}   /* end of func_get */

int exec_function(cmd,fpp)
char *cmd;
struct phrase *fpp;
{
    register struct function *sfp;
    register struct phrase *pp,*cpp;
    struct phrase *lpp;

    sfp = func_get(cmd);
    if (sfp != (struct function *)NULL)
    {   /* if the function was found */
        if (main_iopush() < 0)
        {   /* if couldn't save i/o env */
            /* message already printed */
            main_iopop();
            cmd_lastrc = -1;
            return -1;
        }
        if (main_varpush() < 0)
        {   /* if couldn't save var env */
            /* message already printed */
            main_iopop();
            cmd_lastrc = -1;
            return -1;
        }
        if (main_argspush(fpp) < 0)
        {   /* if couldn't save args env */
            /* message already printed */
            main_iopop();
            main_varpop();
            cmd_lastrc = -1;
            return -1;
        }

        if (sfp->code != (struct phrase *)NULL)
        for (pp = sfp->code->group; pp != (struct phrase *)NULL; )
        {   /* execute the sentence */
            cpp = copy_phrase(pp,1,0,1);
            lpp = pp;
            pp = pp->next;
            if (cpp == (struct phrase *)NULL)
            {   /* enough memory? */
                /* message already printed */
                main_iopop();
                main_varpop();
                main_argspop();
                cmd_lastrc = -1;
                return -1;
            }
            if (flag_echoexec)
                phrase_dump(cpp,0,0);
            exec_phrase(cpp,0);
            if (cpp->body != (struct token *)NULL)
                var_define0("_",cpp->body_end->name,0);
            phrase_free(cpp);
            if (cmd_returnexit)
                break;

            if (lpp->type != (struct token *)NULL) switch (lpp->type->type)
            {   /* any special handling? */
                case SYM_ANDIF:
                    if (cmd_lastrc != 0 && pp != (struct phrase *)NULL)
                        pp = pp->next;
                    break;
                case SYM_ORIF:
                    if (cmd_lastrc == 0 && pp != (struct phrase *)NULL)
                        pp = pp->next;
                    break;
            }
            if (pp == (struct phrase *)NULL)
                break;
        }
        cmd_returnexit = 0;
        main_iopop();
        main_varpop();
        main_argspop();
        force_signal("EXIT");
        return 1;
    }
    return 0;
}   /* end of exec_function */

void exec_alias(name,path)
char *name;
register char *path;
{
    register char *newpath;
    register int plen;

    plen = strlen(path);
    if (strchr(path,DIR_SEPARATOR) == (char *)NULL
#ifdef  GEMDOS
        || strchr(path,':') == (char *)NULL
#endif  /* GEMDOS */
                            )
    {   /* if only a partial path name, add on currdir */
        plen += strlen(base_env.dir->current)+1;
    }
    if (flag_cmdhash || alias_tracked(name))
    {   /* if hashing commands */
        newpath = new_string(plen+1);
        if (newpath == (char *)NULL)
        {   /* enough memory? */
            errmsg(SHERR_NOMEM,LOC("exec_alias"));
            return;
        }
        if (plen != strlen(path))
            sprintf(newpath,"%s%c%s",base_env.dir->current,DIR_SEPARATOR,path);
        else    strcpy(newpath,path);
        alias_sdefine(name,newpath,TYPE_TRACKED);
        free(newpath);
    }
}   /* end of exec_alias */

#ifndef GEMDOS
static int savepid(pp,pid)
register struct phrase *pp;
int pid;
{
    register struct procs *newWait,*ws;
    char buffer[64];

    if (pp == (struct phrase *)NULL)
        return;
    if (pp->type != (struct token *)NULL && pp->type->type != SYM_BACK && pp->type->type != SYM_PIPE)
    {   /* if need to wait for this job */
        for (ws = pipeStack; ws != (struct procs *)NULL; )
        {   /* this job included some previous pipe stuff */
            newWait = ws;
            ws = ws->next;
            newWait->next = waitStack;
            waitStack = newWait;
        }
        pipeStack = pipeStack_end = (struct procs *)NULL;
        newWait = (struct procs *)malloc(sizeof(*newWait));
        if (newWait == (struct procs *)NULL)
        {   /* enough memory? */
            errmsg(SHERR_NOMEM,LOC("exec_real"));
            return -1;
        }
        newWait->pid = pid;
        newWait->next = waitStack;
        waitStack = newWait;
    }
    else if (pp->type != (struct token *)NULL && pp->type->type == SYM_PIPE)
    {   /* save a pipe pid for later reference */
        newWait = (struct procs *)malloc(sizeof(*newWait));
        if (newWait == (struct procs *)NULL)
        {   /* enough memory? */
            errmsg(SHERR_NOMEM,LOC("exec_real"));
            return -1;
        }
        newWait->pid = pid;
        newWait->next = (struct procs *)NULL;
        if (pipeStack == (struct procs *)NULL)
            pipeStack = pipeStack_end = newWait;
        else
        {   /* tack onto end */
            pipeStack_end->next = newWait;
            pipeStack_end = newWait;
        }
    }
    else if (pp->type != (struct token *)NULL && pp->type->type == SYM_BACK)
    {   /* say what we did */
        for (ws = pipeStack; ws != (struct procs *)NULL; ws = ws->next)
        {   /* this background job included some previous pipe stuff */
            sprintf(buffer,"%d ",ws->pid);
            io_writestring(0,buffer);
        }
        bgJobs = pipeStack;
        pipeStack = pipeStack_end = (struct procs *)NULL;
        newWait = (struct procs *)malloc(sizeof(*newWait));
        if (newWait == (struct procs *)NULL)
        {   /* enough memory? */
            errmsg(SHERR_NOMEM,LOC("exec_real"));
            return -1;
        }
        newWait->pid = pid;
        newWait->next = bgJobs;
        bgJobs = newWait;
        base_env.background_pid = pid;
        sprintf(buffer,"%d\n",pid);
        io_writestring(0,buffer);
    }
}   /* end of savepid */
#endif  /* GEMDOS */

#ifdef  GEMDOS
int exec_real(cmd,pp)
char *cmd;
struct phrase *pp;
{
    char cmdtail[128];
    struct token *tp;
    char *arg,*env,*newenv,*envp;
    int length,rlength;
    int warn127;

    var_define0("_",cmd,0);
    warn127 = 0;
    cmdtail[1] = '\0';
    for (tp = pp->body->next; tp != (struct token *)NULL; tp = tp->next)
    {   /* build a GEMDOS command tail */
        arg = strcopy(tp->name);
        if (arg == (char *)NULL)
        {   /* enough memory? */
            /* message already printed */
            return -1;
        }
        stripquotes(arg);
        if (!warn127 && ((strlen(&cmdtail[1])+strlen(arg)+1) > 127))
        {   /* will it fit? */
            errmsg(0,LOC("exec_real"),"warning: command args > 127 chars, command may not work");
            warn127++;
            break;
        }
        if (cmdtail[1] != '\0')
            strcat(&cmdtail[1]," ");
        strcat(&cmdtail[1],arg);
        free(arg);
    }
    cmdtail[0] = strlen(&cmdtail[1]);
    /* build environment variables */
    env = var_makeenv();
#ifdef  MWC_ARGV
    /* add in MWC ARGV */
    /* iovector: C=console, A=aux, P=printer, F=file */
    length = 2;
    if (env != (char *)NULL)
    {   /* find out how big the environment is */
        for (envp = env; ; envp++)
        {   /* until we reach the end */
            if (envp[0] == '\0' && envp[1] == '\0')
                break;
            length++;
        }
    }
    rlength = length;
    length += 34;       /* add in ARGV=<iovector> */
    for (tp = pp->body; tp != (struct token *)NULL; tp = tp->next)
    {   /* get total length of args */
        length += strlen(tp->name)+1;
    }
    newenv = new_string(length);
    if (newenv == (char *)NULL)
    {   /* enough memory? */
        errmsg(SHERR_NOMEM,LOC("exec_real"));
        if (env != (char *)NULL)
            free(env);
        return -1;
    }
    newenv[0] = newenv[1] = '\0';
    if (env != (char *)NULL)
        memcpy(newenv,env,rlength);
    for (envp = newenv; ; envp++)
    {   /* scan for end */
        if (envp[0] == '\0' && envp[1] == '\0')
            break;
    }
    if (envp != newenv)
        envp++;
    sprintf(envp,"ARGV=%s",mwc_iovec);
    while (*envp)
        envp++;
    *envp++;
    for (tp = pp->body; tp != (struct token *)NULL; tp = tp->next)
    {   /* now put the args in */
        strcpy(envp,tp->name);
        stripquotes(envp);
        while (*envp)
            envp++;
        envp++;
    }
    *envp = '\0';
    if (env != (char *)NULL)
        free(env);
    env = newenv;
#endif  /* MWC_ARGV */

    cmd_lastrc = Pexec(0,cmd,cmdtail,env);
    if (env != (char *)NULL)
        free(env);
    if (NO_FILE(cmd_lastrc))
        return 0;
    if (NOT_EXEC(cmd_lastrc) && base_env.shellname != (char *)NULL && strlen(base_env.shellname) != 0)
    {   /* try as shell file */
        if (strcmp(pp->body->name,base_env.shellname) == 0)
            return 0;
        tp = new_token(strlen(base_env.shellname));
        if (tp == (struct token *)NULL)
        {   /* enough memory? */
            errmsg(SHERR_NOMEM,LOC("exec_real"));
            return 0;
        }
        tp->type = SYM_WORD;
        strcpy(tp->name,base_env.shellname);
        tp->next = pp->body;
        pp->body = tp;
        length = exec_phrase(pp,1);
        if (pp->body != (struct token *)NULL)
            var_define0("_",pp->body_end->name,0);
        return length;
    }
    return 1;
}   /* end of exec_real */
#else
int exec_real(cmd,pp)
char *cmd;
struct phrase *pp;
{
    register struct token *tp;
    register char *cp;
    int pid;
    char *envp,*first;
    char **arg_vector,**env_vector;
    int arg_count;

    arg_count = 0;
    for (tp = pp->body; tp != (struct token *)NULL; tp = tp->next)
        arg_count++;
    arg_vector = new_argv(arg_count);
    if (arg_vector == (char **)NULL)
    {   /* enough memory? */
        errmsg(SHERR_NOMEM,LOC("exec_real"));
        return -1;
    }
    arg_count = 0;
    for (tp = pp->body; tp != (struct token *)NULL; tp = tp->next)
    {   /* copy in args */
        stripquotes(tp->name);
        arg_vector[arg_count++] = tp->name;
    }
    arg_vector[arg_count] = (char *)NULL;

    arg_count = 0;
    envp = var_makeenv();
    for (cp = envp; ; cp++)
    {   /* until double null is found */
        if (*cp == '\0')
        {   /* found end of one env string */
            arg_count++;
            if (*(cp+1) == '\0')
                break;
        }
    }
    env_vector = new_argv(arg_count);
    if (env_vector == (char **)NULL)
    {   /* enough memory? */
        errmsg(SHERR_NOMEM,LOC("exec_real"));
        free(arg_vector);
        return -1;
    }
    arg_count = 0;
    first = envp;
    for (cp = envp; ; cp++)
    {   /* until double null is found */
        if (*cp == '\0')
        {   /* found end of one env string */
            env_vector[arg_count++] = first;
            first = cp+1;
            if (*first == '\0')
                break;
        }
    }
    env_vector[arg_count] = (char *)NULL;

    /* exec cmd */
    pid = fork();
    if (pid < 0)
    {   /* can we get a child process? */
        errmsg(0,LOC("exec_real"),"can't fork to run command");
        free(env_vector);
        if (envp != (char *)NULL)
            free(envp);
        free(arg_vector);
        return -1;
    }
    if (pid == 0)
    {   /* this is the child; run the command */
        cmd_lastrc = execve(cmd,arg_vector,env_vector);
        if (NO_FILE(cmd_lastrc))
            exit(-1);
        if (NOT_EXEC(cmd_lastrc) && base_env.shellname != (char *)NULL && strlen(base_env.shellname) != 0)
        {   /* try as shell file */
            if (strcmp(pp->body->name,base_env.shellname) == 0)
                return 0;
            tp = pp->body;
            pp->body = tp->next;
            free(tp);
            tp = new_token(strlen(cmd));
            if (tp == (struct token *)NULL)
            {   /* enough memory? */
                errmsg(SHERR_NOMEM,LOC("exec_real"));
                exit(-1);
            }
            tp->type = SYM_WORD;
            strcpy(tp->name,cmd);
            tp->next = pp->body;
            pp->body = tp;
            tp = new_token(strlen(base_env.shellname));
            if (tp == (struct token *)NULL)
            {   /* enough memory? */
                errmsg(SHERR_NOMEM,LOC("exec_real"));
                exit(-1);
            }
            tp->type = SYM_WORD;
            strcpy(tp->name,base_env.shellname);
            tp->next = pp->body;
            pp->body = tp;
            pid = exec_phrase(pp,1);
            if (pp->body != (struct token *)NULL)
                var_define0("_",pp->body_end->name,0);
            exit(pid);
        }
        exit(-1);
    }

    free(env_vector);
    if (envp != (char *)NULL)
        free(envp);
    free(arg_vector);
    savepid(pp,pid);

    return pid;
}   /* end of exec_real */
#endif  /* GEMDOS */

char *exec_pathsearch(name)
char *name;
{
    register char *pf,*pl,*p;
    char sep;
#ifdef  GEMDOS
    char *spf,*spl,*lp;
#endif  /* GEMDOS */
    char buffer[BUFSIZ];

#ifdef  GEMDOS
    sep = ',';
    spf = spl = base_env.exec_suff;
#else
    sep = ':';
#endif  /* GEMDOS */
    if (isfullpath(name))
    {   /* if don't need a path search */
        strcpy(buffer,name);
#ifdef  GEMDOS
        if (strchr(name,'.') == (char *)NULL)
        {   /* need to add a suffix? */
            for (lp = buffer; *lp; lp++)
                /* do nothing */;
            for (;;)
            {   /* while there are suffixes */
                if (*spl != ',' && *spl != '\0')
                {   /* not there yet? */
                    spl++;
                    continue;
                }
                strncpy(lp,spf,(int)(spl - spf));
                lp[(int)(spl - spf)] = '\0';
                if (isexec(buffer))
                    return strcopy(buffer);
                if (*spl == '\0')
                    break;
                spl++;
                spf = spl;
            }
        }
        else if (isexec(buffer))
            return strcopy(buffer);
#else
        if (isexec(buffer))
            return strcopy(buffer);
#endif  /* GEMDOS */
    }
    pf = pl = base_env.exec_path;
    for (;;)
    {   /* look through the path options... */
        if (*pl != sep && *pl != '\0')
        {   /* keep going? */
            pl++;
            continue;
        }
        strncpy(buffer,pf,(int)(pl - pf));
        buffer[(int)(pl - pf)] = '\0';
        if ((int)(pl - pf) != 0)
        {   /* need a dir separator? */
            for (p = buffer; *p; p++)
                /* do nothing */;
            if (p[-1] != DIR_SEPARATOR)
                *p++ = DIR_SEPARATOR;
            *p = '\0';
        }
        strcat(buffer,name);
#ifdef  GEMDOS
        if (strchr(name,'.') == (char *)NULL)
        {   /* need to add a suffix? */
            for (lp = buffer; *lp; lp++)
                /* do nothing */;
            for (;;)
            {   /* while there are suffixes */
                if (*spl != ',' && *spl != '\0')
                {   /* not there yet? */
                    spl++;
                    continue;
                }
                strncpy(lp,spf,(int)(spl - spf));
                lp[(int)(spl - spf)] = '\0';
                if (isexec(buffer))
                {   /* if it apparently worked */
                    exec_alias(name,buffer);
                    return strcopy(buffer);
                }
                if (*spl == '\0')
                    break;
                spl++;
                spf = spl;
            }
        }
        else if (isexec(buffer))
        {   /* if it apparently worked */
            exec_alias(name,buffer);
            return strcopy(buffer);
        }
#else
        if (isexec(buffer))
        {   /* if it apparently worked */
            exec_alias(name,buffer);
            return strcopy(buffer);
        }
#endif  /* GEMDOS */
        if (*pl == '\0')
            break;
        pl++;
        pf = pl;
#ifdef  GEMDOS
        spf = spl = base_env.exec_suff;
#endif  /* GEMDOS */
    }
    return (char *)NULL;
}   /* end of exec_pathsearch */

int exec_normal(cmd,pp)
char *cmd;
struct phrase *pp;
{
    register char *command;
    int rc;

    command = exec_pathsearch(cmd);
    if (command == (char *)NULL)
        return 0;
    rc = exec_real(command,pp);
    free(command);
    return rc;
}   /* end of exec_normal */

#ifdef  GEMDOS
int exec_open(file,type,fd)
register char *file;
int type;
int fd;
{
    int rc,rc1;

    if (strcmp(file,"tty:") == 0 ||
        strcmp(file,"TTY:") == 0 ||
        strcmp(file,"CON:") == 0 ||
        strcmp(file,"con:") == 0)
    {   /* get console? */
        rc = Fforce(fd,base_env.fd_con);
        if (rc < 0)
            errmsg(SHERR_PERROR,LOC("exec_open"),"can't setup %d<->con:",fd);
#ifdef  MWC_ARGV
        else mwc_iovec[fd] = 'C';
#endif  /* MWC_ARGV */
        return rc;
    }
    if (strcmp(file,"aux:") == 0 ||
        strcmp(file,"AUX:") == 0)
    {   /* get aux port */
        rc = Fforce(fd,base_env.fd_aux);
        if (rc < 0)
            errmsg(SHERR_PERROR,LOC("exec_open"),"can't setup %d<->aux:",fd);
#ifdef  MWC_ARGV
        else mwc_iovec[fd] = 'A';
#endif  /* MWC_ARGV */
        return rc;
    }
    if (strcmp(file,"prn:") == 0 ||
        strcmp(file,"PRN:") == 0 ||
        strcmp(file,"prt:") == 0 ||
        strcmp(file,"PRT:") == 0)
    {   /* get printer */
        rc = Fforce(fd,base_env.fd_prn);
        if (rc < 0)
            errmsg(SHERR_PERROR,LOC("exec_open"),"can't setup %d<->prn:",fd);
#ifdef  MWC_ARGV
        else mwc_iovec[fd] = 'P';
#endif  /* MWC_ARGV */
        return rc;
    }
    rc = -1;
    switch (type)
    {   /* what kind of open? */
        case 0:
            rc = Fopen(file,0);
            break;
        case 1:
            Fdelete(file);
            rc = Fcreate(file,0);
            break;
        case 2:
            rc = Fopen(file,1);
            break;
    }
    if (rc < 0)
    {   /* if didn't work */
        errmsg(SHERR_PERROR,LOC("exec_open"),"can't open %s",file);
        return rc;
    }
    rc1 = Fforce(fd,rc);
    if (rc1 < 0)
    {   /* if didn't work */
        Fclose(rc);
        errmsg(SHERR_PERROR,LOC("exec_open"),"can't associate %d<->%s",fd,file);
        return rc1;
    }
    Fclose(rc);
#ifdef  MWC_ARGV
    mwc_iovec[fd] = 'F';
#endif  /* MWC_ARGV */
    return 0;
}   /* end of exec_open */
#endif  /* GEMDOS */

static long fds_opened = 0;

int exec_ioredir(io)
struct iotoken *io;
{
    register struct iotoken *iop;

#ifdef  GEMDOS
    int rc,tfd,sfd;

    for (iop = io; iop != (struct iotoken *)NULL; iop = iop->next)
    {   /* for each thing we need to switch */
        if (iop->fd > MAXUFD)
            continue;
        switch (iop->type)
        {   /* what kind of redirection? */
            case SYM_LT:
            case SYM_PIPER:
                rc = exec_open(iop->file,0,iop->fd);
                break;
            case SYM_LLT:
            case SYM_LLTL:
                rc = exec_open(iop->tempfile,0,iop->fd);
                break;
            case SYM_LBT:
            case SYM_RBT:
                tfd = atoi(iop->file);
                if (tfd > MAXUFD)
                    continue;
                rc = Fdup(tfd);
                if (rc < 0)
                {   /* did it work? */
                    errmsg(SHERR_PERROR,LOC("exec_ioredir"),"can't duplicate %d",tfd);
                    break;
                }
                sfd = rc;
                rc = Fforce(iop->fd,sfd);
                Fclose(sfd);
                if (rc < 0)
                    errmsg(SHERR_PERROR,LOC("exec_ioredir"),"can't associate %d<->%d",iop->fd,tfd);
                break;
            case SYM_RT:
            case SYM_PIPE:
                rc = exec_open(iop->file,1,iop->fd);
                if (iop->type == SYM_PIPE)
                    delete_later(iop->file);
                break;
            case SYM_RRT:
                rc = exec_open(iop->file,2,iop->fd);
                if (rc >= 0)
                    rc = Fseek(0L,iop->fd,2);
                break;
        }
        if (rc < 0)
        {   /* couldn't set up the io */
            /* message already printed */
            return rc;
        }
        fds_opened |= (1L << (long)iop->fd);
    }
    if (!(fds_opened & (1L<<0L)))
    {   /* was stdin setup */
        rc = Fforce(0,base_env.io->input);
        if (rc < 0)
        {   /* did it work? */
            errmsg(SHERR_PERROR,LOC("exec_ioredir"),"can't set default stdin");
            return rc;
        }
#ifdef  MWC_ARGV
        else mwc_iovec[0] = 'C';
#endif  /* MWC_ARGV */
        fds_opened |= (1L<<0L);
    }
    if (!(fds_opened & (1L<<1L)))
    {   /* was stdout setup */
        rc = Fforce(1,base_env.io->output);
        if (rc < 0)
        {   /* did it work? */
            errmsg(SHERR_PERROR,LOC("exec_ioredir"),"can't set default stdout");
            return rc;
        }
#ifdef  MWC_ARGV
        else mwc_iovec[1] = 'C';
#endif  /* MWC_ARGV */
        fds_opened |= (1L<<1L);
    }
    if (!(fds_opened & (1L<<2L)))
    {   /* was stderr setup */
        rc = Fforce(2,base_env.io->errout);
        if (rc < 0)
        {   /* did it work? */
            errmsg(SHERR_PERROR,LOC("exec_ioredir"),"can't set default stderr");
            return rc;
        }
#ifdef  MWC_ARGV
        else mwc_iovec[2] = 'C';
#endif  /* MWC_ARGV */
        fds_opened |= (1L<<2L);
    }
    return 0;
#else
    int rc,tfd;
    static int pipefds[2];

    for (iop = io; iop != (struct iotoken *)NULL; iop = iop->next)
    {   /* for each thing we need to switch */
        if (iop->fd > MAXUFD)
            continue;
        switch (iop->type)
        {   /* what kind of redirection? */
            case SYM_PIPER:
                rc = 0;
                if (pipefds[0] < 0)
                {   /* pipe not created? */
                    errmsg(0,LOC("exec_ioredir"),"no pipe created to read from");
                    rc = -1;
                    break;
                }
                if (pipefds[0] != iop->fd)
                {   /* if need to move fd */
                    rc = dup2(pipefds[0],iop->fd);
                    close(pipefds[0]);
                    main_fdput(pipefds[0]);
                }
                pipefds[0] = pipefds[1] = -1;
                break;
            case SYM_PIPE:
                rc = pipe(pipefds);
                if (rc < 0)
                {   /* can't set up pipe? */
                    errmsg(0,LOC("exec_ioredir"),"can't create pipe");
                    pipefds[0] = pipefds[1] = -1;
                    break;
                }
                if (pipefds[1] != iop->fd)
                {   /* if need to move fd */
                    rc = dup2(pipefds[1],iop->fd);
                    close(pipefds[1]);
                }
                if (rc >= 0)
                {   /* if successful so far, move pipefds[0] out of harms way */
                    tfd = main_fdget();
                    rc = dup2(pipefds[0],tfd);
                    close(pipefds[0]);
                    if (rc >= 0)
                    {   /* if everything worked */
                        pipefds[0] = tfd;
                        pipefds[1] = iop->fd;
                    }
                    else
                    {   /* it didn't work... */
                        errmsg(0,LOC("exec_ioredir"),"can't finish off pipe");
                        pipefds[0] = pipefds[1] = -1;
                        main_fdput(tfd);
                        break;
                    }
                }
                break;
            case SYM_LT:
                rc = tfd = open(iop->file,O_RDONLY);
                if (rc >= 0 && rc != iop->fd)
                {   /* move to correct fd */
                    rc = dup2(rc,iop->fd);
                    close(tfd);
                }
                if (rc < 0)
                    errmsg(0,LOC("exec_ioredir"),"can't open file %s",iop->file);
                break;
            case SYM_LLT:
            case SYM_LLTL:
                rc = tfd = open(iop->tempfile,O_RDONLY);
                if (rc >= 0 && rc != iop->fd)
                {   /* move to correct fd */
                    rc = dup2(rc,iop->fd);
                    close(tfd);
                }
                if (rc < 0)
                    errmsg(0,LOC("exec_ioredir"),"can't open tempfile %s",iop->tempfile);
                break;
            case SYM_LBT:
            case SYM_RBT:
                tfd = atoi(iop->file);
                if (tfd > MAXUFD)
                    continue;
                rc = dup2(tfd,iop->fd);
                if (rc < 0)
                    errmsg(SHERR_PERROR,LOC("exec_ioredir"),"can't associate %d<->%d",iop->fd,tfd);
                break;
            case SYM_RT:
                rc = tfd = open(iop->file,O_WRONLY|O_CREAT|O_TRUNC,0666);
                if (rc >= 0 && rc != iop->fd)
                {   /* move to correct fd */
                    rc = dup2(rc,iop->fd);
                    close(tfd);
                }
                if (rc < 0)
                    errmsg(0,LOC("exec_ioredir"),"can't create %s",iop->file);
                break;
            case SYM_RRT:
                rc = tfd = open(iop->file,O_WRONLY|O_CREAT,0666);
                if (rc >= 0 && rc != iop->fd)
                {   /* move to correct fd */
                    rc = dup2(rc,iop->fd);
                    close(tfd);
                }
                if (rc >= 0)
                    rc = lseek(iop->fd,0L,2);
                if (rc < 0)
                    errmsg(0,LOC("exec_ioredir"),"can't create %s",iop->file);
                break;
        }
        if (rc < 0)
        {   /* couldn't set up the io */
            /* message already printed */
            return rc;
        }
        fds_opened |= (1L << (long)iop->fd);
    }
    if (!(fds_opened & (1L<<0L)))
    {   /* was stdin setup */
        rc = dup2(base_env.io->input,0);
        if (rc < 0)
        {   /* did it work? */
            errmsg(SHERR_PERROR,LOC("exec_ioredir"),"can't set default stdin");
            return rc;
        }
        fds_opened |= (1L<<0L);
    }
    if (!(fds_opened & (1L<<1L)))
    {   /* was stdout setup */
        rc = dup2(base_env.io->output,1);
        if (rc < 0)
        {   /* did it work? */
            errmsg(SHERR_PERROR,LOC("exec_ioredir"),"can't set default stdout");
            return rc;
        }
        fds_opened |= (1L<<1L);
    }
    if (!(fds_opened & (1L<<2L)))
    {   /* was stderr setup */
        rc = dup2(base_env.io->errout,2);
        if (rc < 0)
        {   /* did it work? */
            errmsg(SHERR_PERROR,LOC("exec_ioredir"),"can't set default stderr");
            return rc;
        }
        fds_opened |= (1L<<2L);
    }
    return 0;
#endif  /* GEMDOS */
}   /* end of exec_ioredir */

static void exec_iodone()
{
    register int i;

    for (i = 0; i <= MAXUFD; i++)
    {   /* get back to known state */
        if (fds_opened & (1L<<(long)i))
        {   /* if it was opened */
#ifdef  GEMDOS
#ifdef  MWC_ARGV
            mwc_iovec[i] = '?';
#endif  /* MWC_ARGV */
            Fclose(i);
#else
            close(i);
#endif  /* GEMDOS */
        }
    }
    fds_opened = 0;
}   /* end of exec_iodone */

#ifndef GEMDOS
int exec_waitfor()
{
    register int i;
    int pid;
    register struct procs *wp,*owp;
    struct bg_job *jb,*ojb,*ljb;
#ifndef USG
    union wait status;
    struct rusage usage;
#else
    int status;
#endif  /* USG */

    while (waitStack != (struct procs *)NULL)
    {   /* wait for running jobs that need waiting for */
#ifndef USG
        pid = wait3(&status,0,&usage);
        cmd_lastrc = status.w_T.w_Retcode;
#else
        pid = wait(&status);
        cmd_lastrc = (status >> 8) & 0xFF;
#endif  /* USG */
        if (pid <= 0)
            break;
        owp = (struct procs *)NULL;
        for (wp = waitStack; wp != (struct procs *)NULL; )
        {   /* anything we can delete? */
            if (wp->pid == pid)
            {   /* then can delete this entry */
                if (owp != (struct procs *)NULL)
                    owp->next = wp->next;
                else waitStack = wp->next;
                free(wp);
                break;
            }
            owp = wp;
            wp = wp->next;
        }
        i = 1;
        for (jb = base_env.jobs; jb != (struct bg_job *)NULL; )
        {   /* or any background stuff to delete? */
            ojb = jb;
            jb = jb->next;
            owp = (struct procs *)NULL;
            for (wp = ojb->cmds; wp != (struct procs *)NULL; )
            {   /* anything we can delete? */
                if (wp->pid == pid)
                {   /* then can delete this entry */
                    if (owp != (struct procs *)NULL)
                        owp->next = wp->next;
                    else ojb->cmds = wp->next;
                    free(wp);
                    break;
                }
                owp = wp;
                wp = wp->next;
            }
            if (ojb->cmds == (struct procs *)NULL)
            {   /* if this job completely finished */
                if (flag_monitor)
                    cmd_jobs_dump(ojb->job,i,"Done");
                phrase_free(ojb->job);
                for (ljb = base_env.jobs; ljb != (struct bg_job *)NULL && ljb->next != ojb; ljb = ljb->next)
                    /* do nothing */;
                if (ljb != (struct bg_job *)NULL && ljb->next == ojb)
                    ljb->next = ojb->next;
                free(ojb);
                if (ojb == base_env.jobs)
                    base_env.jobs = jb;
            }
            i++;
        }
    }
    return 0;
}   /* end of exec_waitfor */
#endif  /* GEMDOS */

int exec_phrase(pp,iosetup)
struct phrase *pp;
int iosetup;
{
    int rc,pid,i;
    char *cmd;
    register struct token *tp,*ltp,*new;
    struct token *ftp;
    struct aliases *ap;
    struct phrase *lpp,*llpp;
    int sf,sr;

    for (tp = pp->var; tp != (struct token *)NULL; tp = tp->next)
    {   /* do any pending var definitions */
        var_define(tp->name,0,0);
    }
    if (!iosetup)
    {   /* if need redirection */
        fds_opened = 0;
        rc = exec_ioredir(pp->io);
        if (rc != 0)
        {   /* if the redirection didn't work */
            /* message already printed */
            return 1;
        }
    }
    /* Alias expansion */
    if (pp->body != (struct token *)NULL)
    {   /* is this an alias */
        ap = alias_get(pp->body->name);
        if (ap != (struct aliases *)NULL && ap->tp != (struct token *)NULL)
        {   /* it is an alias; do the replacement */
            ftp = ltp = (struct token *)NULL;
            for (tp = ap->tp; tp != (struct token *)NULL; tp = tp->next)
            {   /* first make copy of tokens */
                new = new_token(strlen(tp->name));
                if (new == (struct token *)NULL)
                {   /* enough memory? */
                    errmsg(SHERR_NOMEM,LOC("exec_phrase"));
                    tokens_free(ftp);
                    return -1;
                }
                new->type = tp->type;
                strcpy(new->name,tp->name);
                if (ftp == (struct token *)NULL)
                    ftp = ltp = new;
                else
                {   /* tack onto end */
                    ltp->next = new;
                    ltp = new;
                }
            }
            if (ltp != (struct token *)NULL)
            {   /* if something to substitute with */
                ltp->next = pp->body->next;
                tp = pp->body;
                pp->body = ftp;
                free(tp);
            }
        }
    }
    /* Wild card expansion */
    if (flag_noglob == 0 && pp->body != (struct token *)NULL)
    {   /* walk through the phrase */
        tp = pp->body;
        ltp = (struct token *)NULL;
        for (tp = tp->next; tp != (struct token *)NULL; tp = tp->next)
        {   /* do expansion */
            new = wild_search(tp->name,(struct token **)NULL);
            if (new == (struct token *)NULL)
            {   /* if no expansion */
                ltp = tp;
                continue;
            }
            if (ltp == (struct token *)NULL)
                pp->body->next = new;
            else ltp->next = new;
            ltp = tp;
            while (new->next != (struct token *)NULL)
                new = new->next;
            tp = new;
            tp->next = ltp->next;
            free(ltp);
            ltp = tp;
        }
    }
    if (pp->body != (struct token *)NULL && pp->type != (struct token *)NULL)
    {   /* if something to run */
        cmd = strcopy(pp->body->name);
        if (cmd == (char *)NULL)
        {   /* enough memory? */
            /* message already printed */
            return -1;
        }
        rc = exec_function(cmd,pp);
        if (rc <= 0)
        {   /* if to try other places */
            stripquotes(cmd);
            rc = exec_builtin(cmd,pp,iosetup);
            if (rc <= 0)
            {   /* if not builtin, try normal execution */
                rc = exec_normal(cmd,pp);
            }
        }
        if (rc <= 0)
        {   /* the command didn't work */
            errmsg(SHERR_PERROR,LOC("exec_phrase"),"%s: command not found",cmd);
            cmd_lastrc = 1;
        }
        free(cmd);
    }
    else    rc = 0;
    if (pp->type != (struct token *)NULL && pp->type->type == SYM_LPAREN)
    {   /* exec the contents of the parenthized group */
#ifdef  GEMDOS
        if (main_iopush() < 0)
        {   /* if couldn't save i/o env */
            /* message already printed */
            main_iopop();
            cmd_lastrc = -1;
            return -1;
        }
        if (main_varpush() < 0)
        {   /* if couldn't save var env */
            /* message already printed */
            main_iopop();
            cmd_lastrc = -1;
            return -1;
        }
        sf = cmd_forceexit;
        sr = cmd_returnexit;
        cmd_forceexit = cmd_returnexit = 0;
        for (lpp = pp->group; lpp != (struct phrase *)NULL; )
        {   /* exec the contents of the parenthized group */
            llpp = lpp;
            lpp = lpp->next;
            exec_phrase(llpp,0);
            if (cmd_forceexit || cmd_returnexit)
                break;
            if (llpp->body != (struct token *)NULL)
                var_define0("_",llpp->body_end->name,0);
        }
        cmd_forceexit = sf;
        cmd_returnexit = sr;
        main_iopop();
        main_varpop();
#else
        pid = fork();
        if (pid < 0)
        {   /* can't fork -> can't run */
            errmsg(0,LOC("exec_phrase"),"can't fork to run parenthesized group of commands");
        }
        else if (pid == 0)
        {   /* run the commands */
            for (lpp = pp->group; lpp != (struct phrase *)NULL; )
            {   /* exec the contents of the parenthized group */
                llpp = lpp;
                lpp = lpp->next;
                exec_phrase(llpp,0);
                if (cmd_forceexit || cmd_returnexit)
                    break;
                if (llpp->body != (struct token *)NULL)
                    var_define0("_",llpp->body_end->name,0);
            }
            cmd_forceexit = 1;
        }
        else
        {   /* set up to wait for completion if necessary */
            for (lpp = pp->group; lpp != (struct phrase *)NULL && lpp->next != (struct phrase *)NULL;
                lpp = lpp->next)
                /* do nothing */;
            savepid(lpp,pid);
        }
#endif  /* GEMDOS */
    }
    if (!iosetup)
        exec_iodone();
#ifndef GEMDOS
    exec_waitfor();
#endif  /* GEMDOS */
    if (cmd_lastrc != 0)
        force_signal("ERR");
    force_signal("DEBUG");
    return rc;
}   /* end of exec_phrase */

static int empty_sentence(sp)
register struct phrase *sp;
{
    if (sp->group == (struct phrase *)NULL)
        return 1;
    if (sp->group != (struct phrase *)NULL &&
        sp->group == sp->group_end &&
        sp->group->io == (struct iotoken *)NULL &&
        sp->group->body == (struct token *)NULL &&
        sp->group->var == (struct token *)NULL &&
        sp->group->type != (struct token *)NULL &&
        sp->group->type->type == SYM_EOL)
        return 1;
    return 0;
}   /* end of empty_sentence */

int exec_sentence(sp,iosetup)
struct phrase *sp;
int iosetup;
{
    register struct phrase *pp,*lpp;
#ifndef GEMDOS
    struct procs *wp,*owp;
    struct bg_job *job,*jl;
#endif  /* GEMDOS */
    int rc;

#ifndef GEMDOS
    for (wp = bgJobs; wp != (struct procs *)NULL; )
    {   /* delete any current stuff */
        owp = wp;
        wp = wp->next;
        free(owp);
    }
    bgJobs = (struct procs *)NULL;
#endif  /* GEMDOS */

    if (empty_sentence(sp))
        return 0;
    if (flag_echoexec)
        sentence_dump(sp);
    base_env.break_level = base_env.continue_level = 0;
    for (pp = sp->group; pp != (struct phrase *)NULL; )
    {   /* for each part of sentence */
        lpp = pp;
        pp = pp->next;
        if (flag_noexec)
            continue;
        lpp = copy_phrase(lpp,1,0,1);
        if (lpp == (struct phrase *)NULL)
        {   /* enough memory? */
            /* message already printed */
            continue;
        }
        if (lpp->type != (struct token *)NULL &&
            lpp->type->type == SYM_SEMI &&
            lpp->type->name[0] == '\0')
            rc = exec_sentence(lpp,iosetup);
        else rc = exec_phrase(lpp,iosetup);
        if (lpp->body != (struct token *)NULL)
            var_define0("_",lpp->body_end->name,0);
        if (rc < 0)
        {   /* abort this sentence? */
            phrase_free(lpp);
            continue;
        }
        if (base_env.break_level > 0 || base_env.continue_level > 0)
        {   /* break out of sentence? */
            phrase_free(lpp);
            break;
        }

        if (lpp->type != (struct token *)NULL) switch (lpp->type->type)
        {   /* any special handling? */
            case SYM_ANDIF:
                if (cmd_lastrc != 0 && pp != (struct phrase *)NULL)
                    pp = pp->next;
                break;
            case SYM_ORIF:
                if (cmd_lastrc == 0 && pp != (struct phrase *)NULL)
                    pp = pp->next;
                break;
        }
        phrase_free(lpp);
        if (pp == (struct phrase *)NULL)
            break;
    }
#ifndef GEMDOS
    if (bgJobs != (struct procs *)NULL)
    {   /* if part of this sentence is running in the background */
        for (jl = base_env.jobs; jl != (struct bg_job *)NULL && jl->next != (struct bg_job *)NULL; jl = jl->next)
            /* do nothing */;
        job = (struct bg_job *)malloc(sizeof(*job));
        if (job != (struct bg_job *)NULL)
        {   /* enough memory? */
            job->next = (struct bg_job *)NULL;
            job->cmds = bgJobs;
            bgJobs = (struct procs *)NULL;
            job->job = copy_group(sp,0,0,1);
            if (base_env.jobs == (struct bg_job *)NULL)
                base_env.jobs = job;
            else
            {   /* tack onto end */
                jl->next = job;
            }
        }
        else
        {   /* out of memory */
            errmsg(SHERR_NOMEM,LOC("exec_sentence"));
        }
    }
#endif  /* GEMDOS */
    if (cmd_forceexit || cmd_returnexit)
        return 1;
    return 0;
}   /* end of exec_sentence */

void phrase_dump(pp,printbody,pad)
struct phrase *pp;
int printbody;
int pad;
{
    register struct token *tp;
    register struct iotoken *iop;
    register struct phrase *cpp;
    char buffer[16];
    int i,flag;

    if (pp == (struct phrase *)NULL)
        return;
    flag = 0;
    for (tp = pp->var; tp != (struct token *)NULL; tp = tp->next)
    {
        io_writestring(0,tp->name);
        if (tp->next != (struct token *)NULL)
            io_writestring(0," ");
        flag++;
    }
    for (tp = pp->body; tp != (struct token *)NULL; tp = tp->next)
    {
        io_writestring(0,tp->name);
        if (tp->next != (struct token *)NULL)
            io_writestring(0," ");
        flag++;
    }
    for (iop = pp->io; iop != (struct iotoken *)NULL; iop = iop->next)
    {   /* io redirections? */
        if (iop->type != SYM_PIPE && iop->type != SYM_PIPER)
        {   /* what fd is used? */
            sprintf(buffer," %d",iop->fd);
            io_writestring(0,buffer);
            flag++;
        }
        switch (iop->type)
        {   /* what action? */
            case SYM_LT:
                io_writestring(0,"<");
                flag++;
                break;
            case SYM_LLT:
                io_writestring(0,"<<");
                flag++;
                break;
            case SYM_LLTL:
                io_writestring(0,"<<-");
                flag++;
                break;
            case SYM_LBT:
                io_writestring(0,"<&");
                flag++;
                break;
            case SYM_RT:
                io_writestring(0,">");
                flag++;
                break;
            case SYM_RRT:
                io_writestring(0,">>");
                flag++;
                break;
            case SYM_RBT:
                io_writestring(0,">&");
                flag++;
                break;
            case SYM_PIPE:
            case SYM_PIPER:
                break;
        }
        if (iop->type != SYM_PIPE && iop->type != SYM_PIPER)
        {   /* file being accessed */
            io_writestring(0,iop->file);
            if (strlen(iop->file) > 0)
                flag++;
        }
    }
    if (pp->type != (struct token *)NULL && pp->type->type != SYM_BQUOTE &&
        strlen(pp->type->name) > 0 && pp->type->type != SYM_EOL)
    {   /* if phrase type printable, print it */
        if (flag)
            io_writestring(0," ");
        io_writestring(0,pp->type->name);
    }
    else if (pp->type != (struct token *)NULL && pp->type->type == SYM_EOL)
    {   /* add in eol if needed */
        io_writestring(0,"\n");
    }
    for (cpp = pp->group; printbody && cpp != (struct phrase *)NULL; cpp = cpp->next)
    {   /* dump out contents of group */
        for (i = 0; flag && i < pad; i++)
            io_writestring(0," ");
        phrase_dump(cpp,printbody,pad);
    }
}   /* end of phrase_dump */

void sentence_dump(sp)
struct phrase *sp;
{
    register struct phrase *pp;
    register struct phrase *fsp;
    int len;

    for (fsp = sp; fsp != (struct phrase *)NULL; fsp = fsp->next)
    {
        len = io_prompt(3);
        for (pp = fsp->group; pp != (struct phrase *)NULL; pp = pp->next)
        {
            phrase_dump(pp,1,len);
        }
    }
}   /* end of sentence_dump */
